/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  AFILE *AFopenRead (const char Fname[], long int *Nsamp, long int *Nchan,
		     float *Sfreq, FILE *fpout)

Purpose:
  Open an audio file for reading

Description:
  This routine opens an audio file for reading.  If unsuccessful, the program
  is terminated.  The companion routine AFreadData reads data from the file.
  Routine AFclose should be used to close the file.

  This routine reads the audio file header and optionally prints the header
  information.  Several file header formats are supported.  For files with
  no header or unrecognized headers, the file format can be declared by
  calling routine AFsetNH.

  Sun (AFsp) audio file:
    8-bit mu-law, 8-bit A-law, 8-bit integer, 16-bit integer, and 32-bit IEEE
    floating-point data formats are supported.
  RIFF WAVE:
    8-bit mu-law, 8-bit A-law, offset-binary 8-bit integer, and 16-bit integer
    data formats are supported.
  AIFF or AIFF-C audio file:
    8-bit mu-law, 8-bit A-law, 8-bit integer, and 16-bit integer data formats
    are supported.
  NIST SPHERE audio file:
    8-bit mu-law and 16-bit integer data formats are supported.
  IRCAM soundfile:
    8-bit mu-law, 8-bit A-law, 16-bit integer, and 32-bit floating-point data
    formats are supported.
  ESPS sampled data feature file:
    16-bit integer and 32-bit IEEE floating-point data formats are supported.
  SPPACK sampled data file:
    8-bit mu-law, 8-bit A-law and 16-bit integer data formats are supported.
  INRS-Telecommunications audio file:
    16-bit integer format is supported.
  Text audio file:
    Data in character format (usually representing 16-bit integer values).
  Headerless audio file:
    Data format is specified by calling routine AFsetNH.

  For the fixed point file data representations, read operations return data
  values as follows.
     format            returned values
    8-bit mu-law   - [ -32124, +32124 ]
    8-bit A-law    - [ -32256, +32256 ]
    8-bit integer  - [ -16384, -16383 ]
    16-bit integer - [ -32768, +32767 ]
  For Sun (AFsp) files, floating-point data in the file are scaled by 32768.
  For file data values between -1 and +1, the returned values will be in the
  range [ -32768, +32768 ].  For ESPS sample data feature files and IRCAM
  soundfiles, floating-point values in the file are returned with unity
  scaling.

Parameters:
  <-  AFILE *AFopenRead
      Audio file pointer for the audio file
   -> const char Fname[]
      Character string specifying the file name
  <-  long int *Nsamp
      Total number of samples in the file (all channels)
  <-  long int *Nchan
      Number of channels
  <-  float *Sfreq
      Sampling frequency
   -> FILE *fpout
      File pointer for printing audio file information.  If fpout is not NULL,
      information about the audio file is printed on the stream selected by
      fpout.

Author / revision:
  P. Kabal  Copyright (C) 1996
  $Revision: 1.44 $  $Date: 1996/08/14 17:46:12 $

-------------------------------------------------------------------------*/

static char rcsid[] = "$Id: AFopenRead.c 1.44 1996/08/14 AFsp-V2R1 $";

#include <stdio.h>
#include <libtsp.h>
#include <libtsp/nucleus.h>
#include <libtsp/AFpar.h>

AFILE *
AFopenRead (Fname, Nsamp, Nchan, Sfreq, fpout)

     const char Fname[];
     long int *Nsamp;
     long int *Nchan;
     float *Sfreq;
     FILE *fpout;

{
  FILE *fp;
  AFILE *AFp;
  int Ftype;
  struct AF_NHpar *Fdef;

/* Open the file for reading */
  fp = fopen (Fname, "rb");
  if (fp == NULL)
    UTerror ("AFopenRead: Cannot open file \"%s\"", Fname);

/* Identify the file type */
  Ftype = AFfileType (fp);

/* Read and print the header information */
  switch (Ftype) {
  case FT_NH:
    Fdef = AFgetNH ();
    if (Fdef->Format == FD_TEXT) {
      fclose (fp);
      fp = fopen (Fname, "r");	/* Reopen the file as a text file */
      if (fp == NULL)
	UTerror ("AFopenRead: Cannot open file \"%s\"", Fname);
    }
    AFp = AFgetNHpar (fp, Fname, Nsamp, Nchan, Sfreq, fpout);
    break;

  case FT_AFSP:
  case FT_SUN:
    AFp = AFgetAUpar (fp, Fname, Nsamp, Nchan, Sfreq, fpout);
    break;

  case FT_WAVE:
    AFp = AFgetWVpar (fp, Fname, Nsamp, Nchan, Sfreq, fpout);
    break;

  case FT_AIFF:
  case FT_AIFF_C:
    AFp = AFgetAIpar (fp, Fname, Nsamp, Nchan, Sfreq, fpout);
    break;

  case FT_SPHERE:
    AFp = AFgetSPpar (fp, Fname, Nsamp, Nchan, Sfreq, fpout);
    break;

  case FT_SF:
    AFp = AFgetSFpar (fp, Fname, Nsamp, Nchan, Sfreq, fpout);
    break;

  case FT_ESPS:
    AFp = AFgetESpar (fp, Fname, Nsamp, Nchan, Sfreq, fpout);
    break;

  case FT_SPPACK:
    AFp = AFgetBLpar (fp, Fname, Nsamp, Nchan, Sfreq, fpout);
    break;

  case FT_INRS:
    AFp = AFgetINpar (fp, Fname, Nsamp, Nchan, Sfreq, fpout);
    break;

  case FT_TXAUD:
    fclose (fp);
    fp = fopen (Fname, "r");	/* Reopen the file as a text file */
    if (fp == NULL)
      UTerror ("AFopenRead: Cannot open file \"%s\"", Fname);
    AFp = AFgetTApar (fp, Fname, Nsamp, Nchan, Sfreq, fpout);
    break;

  default:
    fclose (fp);
    UThalt ("AFopenRead: Unknown audio file format");
    break;
  }

  return AFp;
}
