/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  LPsyn [options] -p PFile -c LPFile AFileI [AFileO]

Purpose:
  LPC synthesis from a residual file

Description:
  This program does linear prediction synthesis given a file of residual
  samples and a file of linear prediction coefficients.  The output is an
  audio file containing the reconstructed signal.

  The steps involved in forming the linear prediction coefficients are as
  follows.
   1: Use the all-pole LPC filter to filter the residual signal
   2: Deemphasize the reconstructed signal

Options:
  The command line specifies options and file names.  The last file name
  specifies the output file of the linear prediction residual.  Options specify
  the level of information printed during processing.
  -p PFile, --parameter_file=PFile
      Parameter file.
  -c LPFile, --lpc_file=LPFile
      LPC predictor coefficient file.
  -s, --statistics
      Print frame-by-frame statistics.
  -I INFO, --info=INFO
      Header information string.
  -h, --help
      Print a list of options and exit.
  -v, --version
      Print the version number and exit.

  The analysis parameters are read from the parameter file.
    preemphasis_factor = float  ! preemphasis factor (0 to 1, default 0)
    window_length = int         ! analysis window length
    window_offset = int         ! initial offset of the center of the analysis
                                ! window from the center of the frame (negative
                                ! values mean that the analysis window precedes
                                ! frame, default  0)
    window_type = char          ! window type (Hamming or rectangular, default
                                ! Hamming)
    frame_length = int          ! frame size
    LPC_number = int            ! number of LPC coefficients
    LPC_BW_expansion = float    ! bandwidth expansion factor (1 gives no
                                ! bandwidth expansion, default 1)

  For AFsp output files, the audio file header contains an information string.
    Standard Header Information:
      date:1994/01/25 19:19:39 UTC    date
      user:kabal@aldebaran            user
      program:LPsyn                   program name
      parameters:<parameters>         list of parameters
  This information can be changed with the header information string which is
  specified as one of the command line options.  Structured information records
  should adhere to the above format with a named field terminated by a colon,
  followed by numeric data or text.  Comments can follow as unstructured
  information.  For the purpose of this program, records are terminated by
  newline characters.  However in the header itself, the newline characters are
  replaced by nulls.  To place a newline character into the header, escape
  the newline character by preceding it with a '\' character.  If the first
  character of the user supplied header information string is a newline
  character, the header information string is appended to the standard header
  information.  If not, the user supplied header information string replaces
  the standard header information.

Environment variables:
  AUDIOPATH:
  This environment variable specifies a list of directories to be searched when
  opening the input audio files.  Directories in the list are separated by
  colons (semicolons for MS-DOS).

Author / version:
  P. Kabal / v1r3  1995/09/12  Copyright (C) 1996

-------------------------------------------------------------------------*/

static char rcsid[] = "$Id: LPsyn.c 1.11 1996/08/14 AFsp-V2R1 $";

#include <stdio.h>
#include <stdlib.h>
#include <libtsp.h>
#include <libtsp/AFpar.h>
#include "LPsyn.h"

#ifndef EXIT_SUCCESS
#  define EXIT_SUCCESS	0	/* Normally in stdlib.h */
#endif

#define MAXHEADER	1024

int
main (argc, argv)

     int argc;
     const char *argv[];

{
  AFILE *AFpI, *AFpL, *AFpO;
  const char *Hinfo;
  const char *Fname[4];
  char Fn[FILENAME_MAX+1];
  float Sfreq;
  int Fstats;
  long int Nsamp, Nchan;
  const float *Win;
  int Lwin, Woffs, Lframe, Np;
  float pre, bwexp;
  float Frate;
  long int Npx, Ntcoef;
  int Fformat;

/* Get the input parameters */
  LPoptions (argc, argv, &Fstats, &Hinfo, Fname);
  Fformat = FD_INT16 + FW_AFSP;

/* Read the analysis parameters */
  LPlpcPar (Fname[0], &pre, &Win, &Lwin, &Woffs, &Lframe, &Np, &bwexp);

/* Open the input residual file */
  FLpathList (Fname[1], "$AUDIOPATH", Fn);
  AFpI = AFopenRead (Fn, &Nsamp, &Nchan, &Sfreq, stdout);
  if (Nchan != 1)
    UThalt ("%s: Multiple input channels not supported", PROGRAM);

/* Open the LPC coefficient file */
  FLpathList (Fname[2], "$AUDIOPATH", Fn);
  AFpL = AFopenRead (Fn, &Ntcoef, &Npx, &Frate, stdout);
  if (Np != Npx)
    UThalt ("%s: No. LPC coefficients does not match parameter file", PROGRAM);
  if (Frate * Lframe != Sfreq)
    UThalt ("%s: Frame rate, frame length and sample rate are incompatible",
	    PROGRAM);
  if (Ntcoef * Lframe != Np * Nsamp)
    UThalt ("%s: No. residual samples incompatible with no. frames", PROGRAM);
  printf ("\n");

/* Open the output audio file */
  if (Hinfo != NULL)
    AFsetHinfo (Hinfo);
  if (Fname[3] != NULL) {
    FLbackup (Fname[3]);
    AFpO = AFopenWrite (Fname[3], Fformat, 1L, Sfreq, stdout);
  }
  else
    AFpO = NULL;

/* Process the residual file */
  LPlpcSyn (AFpI, AFpL, AFpO, Fstats, pre, Lframe, Np);

/* Close the files */
  AFclose (AFpI);
  AFclose (AFpL);
  if (AFpO != NULL)
    AFclose (AFpO);

  return EXIT_SUCCESS;
}
