/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  InfoAudio [options] AFile1 [AFile2 ...  ]

Purpose:
  Print audio file information

Description:
  This program prints information about a number of audio files to standard
  output.  The information about the internal structure of the files is in a
  form that can be easily parsed by a shell script.

  The full output consists of the file type and name (indented part of the
  output below), the file data parameters, and the header information records.
     AFsp audio file: /usr/local/src/audio/InfoAudio/test/xx.au
       Number of samples : 23808  1994/02/04 14:12:50 UTC
       Sampling frequency: 8000 Hz
       Number of channels: 1 (16-bit integer)
    File name: xx.au
    Header length: 104
    Sampling frequency: 8000
    No. samples: 23808
    No. channels: 1
    Data type: integer16         - mu-law8, integer16 or float32
    File byte order: big-endian  - big-endian, little-endian, or byte-stream 
    Host byte order: big-endian  - big-endian or little-endian
    --Header information records--
    date:1994/02/04 14:12:50 UTC
    user:library@aldebaran
    program:CopyAudio

  For single byte data, the "File byte order" field is set to "byte-stream".

  Non-printing characters (as defined by the routine isprint) in the header
  information records are replaced by question marks.

Options:
  The command line specifies options and the file name.
  -P PARMS, --parameters=PARMS
      Parameters to be used for headerless input files.  This option may be
      given more than once.  Each invokation applies to the files that follow
      the option.  See the description of the environment variable RAWAUDIOFILE
      below for the format of the parameter specification.
  -i ICODE, --info_code=ICODE
      Parameter to choose the information to be printed.  The information that
      can be printed consists of three parts.  The ICODE parameter is the sum
      of the codes which control printing of the individual parts.
        1:  Print the file type and name
        2:  Print the file data parameters
        4:  Print the header information records
      The default is to print all of the information (ICODE=7).  For ICODE=0,
      no information is printed, but the program return code indicates if all
      files were opened successfully.
  -h, --help
      Print a list of options and exit.
  -v, --version
      Print the version number and exit.

Environment variables:
  RAWAUDIOFILE:
  This environment variable defines the data format for headerless or
  non-standard input audio files.  The string consists of a list of parameters
  separated by commas.  The form of the list is
    "Format, Start, Sfreq, Swapb, Nchan, ScaleF"
  Format: File data format
      The lowercase versions of these format specifiers cause a headerless
      file to be accepted only after checking for standard file headers; the
      uppercase versions cause a headerless file to be accepted without
      checking the file header.
       "undefined"                - Headerless files will be rejected
       "mu-law8" or "MU-LAW8"     - 8-bit mu-law data
       "A-law8" or "A-LAW8"       - 8-bit A-law data
       "unsigned8" or "UNSIGNED8" - offset-binary 8-bit integer data
       "integer8" or "INTEGER8"   - two's-complement 8-bit integer data
       "integer16" or "INTEGER16" - two's-complement 16-bit integer data
       "float32" or "FLOAT32"     - 32-bit floating-point data
       "text" or "TEXT"           - text data
  Start: byte offset to the start of data (integer value)
  Sfreq: sampling frequency in Hz (floating point number)
  Swapb: Data byte swap parameter
       "native" - no byte swapping
       "little-endian" - file data is in little-endian byte order
       "big-endian" - file data is in big-endian byte order
       "swap" - swap the data bytes as the data is read
  Nchan: number of channels
      The data consists of interleaved samples from Nchan channels
  ScaleF: Scale factor
      Scale factor applied to the data from the file
  The default values for the audio file parameters correspond to the following
  string.
      "undefined, 0, 8000., native, 1, 1.0"

  AUDIOPATH:
  This environment variable specifies a list of directories to be searched when
  opening the input audio files.  Directories in the list are separated by
  colons (semicolons for MS-DOS).

Author / version:
  P. Kabal / v2r2  1996/08/12  Copyright (C) 1996

-------------------------------------------------------------------------*/

static char rcsid[] = "$Id: InfoAudio.c 1.47 1996/08/19 AFsp-V2R1 $";

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <libtsp.h>
#include <libtsp/nucleus.h>
#include <libtsp/AFpar.h>
#include "InfoAudio.h"

#define EXTCODE(x,n)	((x / n) % 2)

#ifndef EXIT_SUCCESS
#  define EXIT_SUCCESS	0	/* Normally in stdlib.h */
#endif

static const char *DataFormat[NFD] = {
  "undefined",
  "mu-law8",
  "A-law8",
  "unsigned8",
  "integer8",
  "integer16",
  "float32",
  "text"
};
static const int DataSize[NFD] = {
  0,
  FDL_MULAW8,
  FDL_ALAW8,
  FDL_UINT8,
  FDL_INT8,
  FDL_INT16,
  FDL_FLOAT32,
  FDL_TEXT
};
static const char *DataSwap[] = {
  "big-endian",
  "little-endian",
  "native",		/* AFp->Swapb is native or swap */
  "swap"
};

int
main (argc, argv)

     int argc;
     const char *argv[];

{
  const char *NHparms[MAXFILE];
  int i, Icode, Nfiles;
  const char *Fname[MAXFILE];
  char Fn[FILENAME_MAX+1];
  FILE *fpout;
  long int Nsamp, Nchan;
  float Sfreq;
  char *Hinfo;
  int Ninfo, ls, n, Hbo, Dbo;
  AFILE *AFp;

/* Option handling */
  IAoptions (argc, argv, NHparms, &Icode, Fname, &Nfiles);

/* Loop over the input files */
  for (i = 0; i < Nfiles; ++i) {

    /* Set up standard headerless audio file handling */
    if (NHparms[i] != NULL)
      AFsetNH (NHparms[i]);
    else
      AFsetNH ("$RAWAUDIOFILE");

    /* Open the audio file */
    fpout = NULL;
    if (EXTCODE (Icode, 1) != 0)
      fpout = stdout;
    FLpathList (Fname[i], "$AUDIOPATH", Fn);
    AFp = AFopenRead (Fn, &Nsamp, &Nchan, &Sfreq, fpout);

    /* Print the audio file parameter information */
    if (EXTCODE (Icode, 2) != 0) {
      printf ("File name: %s\n", Fn);
      printf ("Header length: %ld\n", AFp->Start);
      printf ("Sampling frequency: %.5g\n", Sfreq);
      printf ("No. samples: %ld\n", Nsamp);
      printf ("No. channels: %ld\n", Nchan);
      printf ("Data type: %s\n", DataFormat[AFp->Format]);
      Hbo = UTbyteOrder ();
      if (AFp->Swapb == DS_NATIVE)
	Dbo = Hbo;
      else
	Dbo = (Hbo == SF_EL) ? SF_EB : SF_EL;
      if (DataSize[AFp->Format] <= 1)
	printf ("File byte order: byte-stream\n");
      else
	printf ("File byte order: %s\n", DataSwap[Dbo]);
      printf ("Host byte order: %s\n", DataSwap[Hbo]);
    }

    if (DataSize[AFp->Format] <= 1 && AFp->Swapb != DS_NATIVE)
      UTwarn ("%s - Data swap should be native for this data format", PROGRAM);

    /* Print the header information strings */
    if (EXTCODE (Icode, 4) != 0) {
      Hinfo = AFreadHinfo (AFp, &Ninfo);
      if (Ninfo > 0) {
	for (n = 0; n < Ninfo; ++n) {
	  if (! (Hinfo[n] == '\0' || Hinfo[n] == '\n' || Hinfo[n] == '\t') &&
	      ! isprint (Hinfo[n]))
	    Hinfo[n] = '?';	/* Replace non-printing characters by ? */
	}
	printf ("--Header information records--\n");
	n = 0;
	while (n < Ninfo) {
	  ls = strlen (&Hinfo[n]);
	  if (ls > 0)
	    printf ("%s\n", &Hinfo[n]);
	  n = n + ls + 1;
	}
      }
    }

    AFclose (AFp);
    if (i < Nfiles-1)
      printf ("\n");

  }
  return EXIT_SUCCESS;
}
