/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  CopyAudio [options] AFileA AfileB ... AFileO

Purpose:
  Copy audio files (combine, scale and shift data)

Description:
  This program copies input audio file(s) to an output audio file.  The samples
  in the output file are linear combinations of the samples in the different
  channels in the input files.  Options are available to change the sampling
  frequency and data format of the output file.  Another option allows for
  selection of the range of input samples which will appear in the output file.

  The sample limits option allows for selecting the range of data to be taken
  from the input files.  The data is conceptually organized into frames, with
  samples from individual channels making up a frame.  Frame numbers start at
  zero.  Frames with negative numbers may be  specified - the samples in these
  frames have zero values.  Similarly, frame numbers beyond the end-of-file
  may be specified - these are assumed to have zero-valued values.

  For more than one input file, channels assignments are considered to cross
  between input files.  Consider two input files, with the first having 3
  channels, and the second having 2 channels.  For the combined input, these
  channels would be labelled A to E, with channels A, B and C from the first
  input file and channels D and E from the second file.  The channel selection
  option allows a linear combination of the input channels to appear in each
  output channel.  The data from the input files is padded out with zeros to
  become equal length before combining.

  The default data format for the output file is chosen according to a data
  format promotion rule based on the data types of the input files.  For single
  input files, the output data format will be the same as the input data format
  as long as that data format is compatible with the output file type.

Options:
  The command line specifies options and file names.  The last file name is
  that of the output file.  The other file names specify input files.
  -s SFREQ, --srate=SFREQ
      Sampling frequency for the output file, default from the input audio
      file(s).  This option only changes the sampling frequency field in the
      output header; the audio data itself is unaffected.
  -D DFORMAT, --data_format=DFORMAT
      Data format for the output file.
        "mu-law8"   - 8-bit mu-law data
        "A-law8"    - 8-bit A-law data
        "unsigned8" - offset-binary 8-bit integer data
        "integer8"  - two's-complement 8-bit integer data
        "integer16" - two's-complement 16-bit integer data
        "float32"   - 32-bit IEEE floating-point data
        "text"      - text data
      The data formats available depend on the output file type.
      AFsp (Sun) audio files:
        mu-law, A-law, 8-bit integer, 16-bit integer, float
      RIFF WAVE files:
        mu-law, A-law, offset-binary 8-bit integer, 16-bit integer
      AIFF-C audio files:
        mu-law, A-law, 8-bit integer, 16-bit integer
      Headerless files:
        all data formats
  -F FTYPE, --file_type=FTYPE
      File type, default "AFsp".
        "AFsp", "Sun" or "sun" - AFsp (Sun) audio file
        "WAVE" or "wave"       - RIFF WAVE file
        "AIFF-C"               - AIFF-C audio file
        "raw" or "raw_native"  - Headerless file (native byte order)
        "raw_swap"             - Headerless file (byte swapped)
        "raw_big-endian"       - Headerless file (big-endian byte order)
        "raw_little-endian"    - Headerless file (little-endian byte order)
  -l L:U, --limits=L:U
      Frame limits for the input files (numbered from zero).  The default frame
      limits correspond to the longest input audio file.
  -g GAIN, --gain=GAIN
      A gain factor applied to all input channels.  This option is incompatible
      with the options specifying individual channel gains.  The default value
      is one.
  -cA GAINS, --chanA=GAINS
      Gain factors for output channel A.  The output data for the specified
      output channel is created by adding scaled samples from the specified
      input channels.  The default is to have each output channel equal to the
      corresponding input channel.  Input channels are labelled A, B, C, ... ,
      R, S, T. 
  -cB GAINS, --chanB=GAINS
      Channel gain factors for output channel B.
        ...
  -cL GAINS, --chanL=GAINS
      Channel gain factors for output channel L.
  -P PARMS, --parameters=PARMS
      Parameters to be used for headerless input files.  This option may be
      given more than once.  Each invokation applies to the files that follow
      the option.  See the description of the environment variable RAWAUDIOFILE
      below for the format of the parameter specification.
  -I INFO, --info=INFO
      Header information string.
  -h, --help
      Print a list of options and exit.
  -v, --version
      Print the version number and exit.

  The gain factor string takes the form
    [+/-] [gain *] chan +/- [gain *] chan ...)
  where chan is A through E.  Consider a case with a two channel input file.
  The output file is to have 3 channels.  Output channel A is input channel
  A.  Output channel B is the average of input channels A and B.  Output
  channel C is input channel B.  The following output channel specifications
  accomplish this mapping.
    --chanA=A --chanB="0.5*A + 0.5*B" --chanC=B
  Note that * is a special character to Unix shells and should appear only
  within quotes to prevent the shell from interpreting it.

  This program allows direct specification of the gains for 10 output channels
  and 20 input channels.  The program can handle larger numbers of channels
  for the case that the input channels are in a one-to-one correspondence with
  the output channels.  A gain factor applying to all channels can be specified
  with the -g or --gain option.

  By default for AFsp output files, the audio file header contains a standard
  information string.
    Standard Header Information:
      date:1994/01/25 19:19:39 UTC    date
      user:kabal@aldebaran            user
      program:CopyAudio               program name
  This information can be changed with the header information string which is
  specified as one of the command line options.  Structured information records
  should adhere to the above format with a named field terminated by a colon,
  followed by numeric data or text.  Comments can follow as unstructured
  information.  For the purpose of this program, records are terminated by
  newline characters.  However in the header itself, the newline characters are
  replaced by nulls.  To place a newline character into the header, escape
  the newline character by preceding it with a '\' character.  If the first
  character of the user supplied header information string is a newline
  character, the header information string is appended to the standard header
  information.  If not, the user supplied header information string replaces
  the standard header information.

Examples:
   1: File copy.
      Copy audio file abc.au to new.au.  The output audio file is an AFsp file
      with 16-bit integer data.  The number of channels in the output file is
      the same as the number of channels in the input file.
        CopyAudio abc.au new.au
   2: Difference between values.
      Create an output audio file in which each sample is the difference
      between corresponding samples in two single channel input audio files.
        CopyAudio --chanA=A-B abc1.au abc2.au diff.au
   3: Scale sample values.
      Scale the samples in the input single-channel file by 0.5.
        CopyAudio --gain=0.5 abc.au scaled.au
      The same result can be obtained by specifying the gain for the (single)
      output channel.
        CopyAudio --chanA="0.5*A" abc.au scaled.au
   4: Byte-swap data values.
      Let the input audio file be headerless and contain 16-bit data.  Create a
      headerless audio file with byte-swapped data.
        CopyAudio -P integer -N -S swap abc.au swap.au
   5: Extract samples.
      Extract samples 1000 to 1999 inclusive from the input audio file. The
      output audio file will have 1000 samples.
        CopyAudio -l 1000:1999 abc.au out.au
   6: Create a stereo file.
      Form a stereo (2-channel) audio file from two single channel audio files.
        CopyAudio abc1.au abc2.au stereo.au

Environment variables:
  RAWAUDIOFILE:
  This environment variable defines the data format for headerless or
  non-standard input audio files.  The string consists of a list of parameters
  separated by commas.  The form of the list is
    "Format, Start, Sfreq, Swapb, Nchan, ScaleF"
  Format: File data format
      The lowercase versions of these format specifiers cause a headerless
      file to be accepted only after checking for standard file headers; the
      uppercase versions cause a headerless file to be accepted without
      checking the file header.
       "undefined"                - Headerless files will be rejected
       "mu-law8" or "MU-LAW8"     - 8-bit mu-law data
       "A-law8" or "A-LAW8"       - 8-bit A-law data
       "unsigned8" or "UNSIGNED8" - offset-binary 8-bit integer data
       "integer8" or "INTEGER8"   - two's-complement 8-bit integer data
       "integer16" or "INTEGER16" - two's-complement 16-bit integer data
       "float32" or "FLOAT32"     - 32-bit floating-point data
       "text" or "TEXT"           - text data
  Start: byte offset to the start of data (integer value)
  Sfreq: sampling frequency in Hz (floating point number)
  Swapb: Data byte swap parameter
       "native"        - no byte swapping
       "little-endian" - file data is in little-endian byte order
       "big-endian"    - file data is in big-endian byte order
       "swap"          - swap the data bytes as the data is read
  Nchan: number of channels
      The data consists of interleaved samples from Nchan channels
  ScaleF: Scale factor
      Scale factor applied to the data from the file
  The default values for the audio file parameters correspond to the following
  string.
      "undefined, 0, 8000., native, 1, 1.0"

  AUDIOPATH:
  This environment variable specifies a list of directories to be searched when
  opening the input audio files.  Directories in the list are separated by
  colons (semicolons for MS-DOS).

Author / version:
  P. Kabal / v1r12  1996/08/12  Copyright (C) 1996

-------------------------------------------------------------------------*/

static char rcsid[] = "$Id: CopyAudio.c 1.55 1996/08/16 AFsp-V2R1 $";

#include <stdio.h>
#include <stdlib.h>
#include <libtsp.h>
#include <libtsp/AFpar.h>
#include "CopyAudio.h"
#include "AO.h"

#ifndef EXIT_SUCCESS
#  define EXIT_SUCCESS	0	/* Normally in stdlib.h */
#endif

#define MAXV(a, b)	(((a) > (b)) ? (a) : (b))
#define ICEILV(n, m)	(((n) + ((m) - 1)) / (m))	/* int n,m >= 0 */

int
main (argc, argv)

     int argc;
     const char *argv[];

{
  int DformatI[MAXIFILE], Fformat;
  long int Lim[2];
  const char *NHparms[MAXIFILE];
  const char *Hinfo;
  float Chgain[MAXCHANO][MAXCHANI];
  int Nfiles, Nifiles;
  long int NchanO, Nf, Nframe, Nsamp;
  long int Nchan[MAXIFILE], NchanT;
  const char *Fname[MAXFILE];
  const char *FnameO;
  char Fn[FILENAME_MAX+1];
  AFILE *AFp[MAXFILE], *AFpO;
  float Sfreq[MAXIFILE], SfreqO;
  int Spad, Sfdiff, NoChO;
  float Sf;
  int i;

/* Get the input parameters */
  CPoptions (argc, argv, &Fformat, &SfreqO, Lim, NHparms, &Hinfo, Chgain,
	     &NchanO, Fname, &Nfiles);

/* Open the input files and determine the number of channels etc. */
  Nifiles = Nfiles - 1;
  NchanT = 0;
  Nframe = 0;
  Spad = 0;
  for (i = 0; i < Nifiles; ++i) {

    /* Open the input file */
    if (NHparms[i] != NULL)
      AFsetNH (NHparms[i]);
    else
      AFsetNH ("$RAWAUDIOFILE");
    FLpathList (Fname[i], "$AUDIOPATH", Fn);
    AFp[i] = AFopenRead (Fn, &Nsamp, &Nchan[i], &Sfreq[i], stdout);

    /* Number of channels and number of frames */
    NchanT = NchanT + Nchan[i];
    Nf = ICEILV (Nsamp, Nchan[i]);
    if (Nsamp % Nchan[i] != 0 || (i > 0 && Nf != Nframe))
      Spad = 1;
    Nframe = MAXV (Nframe, Nf);

    /* Data format */
    DformatI[i] = AFp[i]->Format;
  }
  printf ("\n");

/* Frame limits */
  if (Lim[0] > Lim[1]) {
    Lim[0] = 0;
    Lim[1] = Nframe - 1;
  }
  else if (Lim[0] < 0 || Lim[1] >= Nframe)
    Spad = 1;
  if (Spad)
    UTwarn ("%s - Input data zero padded", PROGRAM);

/* Sampling frequency */
  Sfdiff = 0;
  Sf = Sfreq[0];
  for (i = 1; i < Nifiles; ++i) {
    if (Sfreq[i] != Sfreq[i-1])
      Sfdiff = 1;
    Sf = Sf + Sfreq[i];
  }
  if (Sfdiff)
    UTwarn ("%s - Input sampling frequencies differ", PROGRAM);
  Sf = Sf / Nifiles;
  if (SfreqO == 0.0)
    SfreqO = Sf;

/* Open the output file */
  if (Hinfo != NULL)
    AFsetHinfo (Hinfo);
  FnameO = Fname[Nfiles-1];
  FLbackup (FnameO);
  NoChO = (NchanO == 0);
  if (NoChO)
    NchanO = NchanT;
  Fformat = AOsetDFormat (Fformat, DformatI, Nifiles);
  AFpO = AFopenWrite (FnameO, Fformat, NchanO, (double) SfreqO, stdout);

/* Scale and copy samples */
  Nchan[Nfiles-1] = NchanO;
  AFp[Nfiles-1] = AFpO;
  if (NoChO)
    CPcopySamp (AFp, Nchan, Nfiles, Lim, Chgain[0][0]);
  else
    CPcombSamp (AFp, Nchan, Nfiles, Lim, Chgain);

/* Close the audio files */
  for (i = 0; i < Nfiles; ++i)
    AFclose (AFp[i]);

  return EXIT_SUCCESS;
}
