/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  void CPdecChan (const char String, float Chgain[])

Purpose:
  Decode channel gain specifications

Description:
  This routine decodes gain factor strings.  The gain factor string takes the
  form
    [+/-] [gain *] chan +/- [gain *] chan ...)
  where chan is A through E.

Parameters:
   -> const char String[]
      Input string
  <-  float Chgain[]
      Output vector of channel gains

Author / revision:
  P. Kabal  Copyright (C) 1996
  $Revision: 1.13 $  $Date: 1996/06/01 02:44:08 $

-------------------------------------------------------------------------*/

static char rcsid[] = "$Id: CPdecChan.c 1.13 1996/06/01 AFsp-V2R1 $";

#include <string.h>
#include <libtsp.h>
#include "CopyAudio.h"

static const char *Chtab[] = {
  "A", "B", "C", "D", "E", "F", "G", "H", "I", "J",
  "K", "L", "M", "N", "O", "P", "Q", "R", "S", "T", NULL };
static const char WS[] = " \t\n";

void
CPdecChan (String, Chgain)

     const char String[];
     float Chgain[];

{
  int nc;
  const char *p;
  char *q;
  char *token;
  float sign, nsign, gain;
  char *Ch;
  int n;

/* Consistency check */
  if (MAXCHANI != (sizeof Chtab / sizeof Chtab[0] - 1))
    UThalt ("CPdecChan: Internal consistency check failed");

/* Allocate temporary space */
  nc = strlen (String);
  token = (char *) UTmalloc (nc + 1);

/* Loop over subexpressions */
  VRfZero (Chgain, MAXCHANI);
  p = String;
  nsign = 1.;
  while (p != NULL) {

    /* Separate the string at + or - signs, p-1 points to the delimiter */
    p = STfindToken (p, "+-", "", token, 1, nc);

    /* Save the sign of the delimiter for the next iteration */
    sign = nsign;
    nsign = 1.;
    if (p != NULL && *(p-1) == '-')
      nsign = -1.;

    /* Special handling for a + or - followed by another + or - */
    if (token[0] == '\0') {
      nsign = sign * nsign;
      continue;
    }

    /* Pick up the gain value */
    q = strchr (token, '*');
    if (q != NULL) {
      *q = '\0';
      if (STdec1float (token, &gain))
	UThalt ("CPdecChan: Invalid channel expression: \"%s\"", String);
      gain = sign * gain;
      Ch = q + 1;
    }
    else {
      gain = sign;
      Ch = token;
    }

    /* Remove leading whitespace from the channel designator */
    Ch = Ch + strspn (Ch, WS);

    /* Find the channel */
    n = STkeyMatch (Ch, Chtab);
    if (n < 0)
      UThalt ("CPdecChan: Invalid channel expression, \"%s\"", String);
    Chgain[n] = Chgain[n] + gain;
  }

  /* Deallocate the storage */
  UTfree (token);

  return;
}
