/***************************************************************************
 ** $Id: voice_recognition.cpp,v 1.9 2004/01/15 23:14:30 wiecko Exp $
                          klearnnotes2
                          voice_recognition.cpp -  description
                             -------------------
    begin                : Thu Oct 23 03:55:46 CEST 2003
    copyright            : (C) 2003 by Marek Wieckowski
    email                : wiecko AT users.sourceforge.net
***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

//#include <signal.h>
#include <stdlib.h>
#include <string.h>
//#include <unistd.h>//fork, _exit

#include <qtranslator.h>
#include <qlistview.h>
#include <kfiledialog.h>
#include <qfiledialog.h>
#include <qapplication.h>

extern "C" {
#include "mixer.h"
#include "audio.h"
#include "preprocess.h"
#include "configuration.h"
#include "model.h"
#include "keypressed.h"
}

const char *model_file_extension = ".cvc";


extern "C" int mixerHasIGain();
extern char orderOfNotes [8] ; //int(name) => char(cname)
//                               definded in klearnnotes2.cpp

#include "voice_recognition.h"
/*****
      fraction of max level (=32768) below which the signal is
      considered to represent silence
*****/
float silence    = 0.02;
/*****
      fraction of max level (=32768) below which the signal is
      considered to represent (too) low volume speech
*****/
float low_volume = 0.15;
/*****
      fraction of max level (=32768) above which the signal is
      considered to represent (too) high volume speech
*****/
float high_volume = 0.75;


MixerDevices *mixer_devices;
AudioDevices *audio_devices;



#include "globals.h"
#include "pics-vr.h"

#include <qvariant.h>
#include <qlabel.h>
#include <qpushbutton.h>
#include <qspinbox.h>
#include <qlayout.h>
#include <qtooltip.h>
#include <qwhatsthis.h>
#include <qvbox.h>
#include <qmessagebox.h>
#include <qtimer.h>
#include <qcombobox.h>


///////////////////////////////////////////////////////
////////////////////////
//////////////////////// dialog MICROPHONE CONFIG:
////////////////////////
///////////////////////////////////////////////////////
/***************************************************************************
 based on microphone_config.h from cvoicecontrol-0.9alpha by Daniel Kiecza
***************************************************************************/



dVoiceMicroSetup::dVoiceMicroSetup(QWidget* parent,  const char* name, 
				   bool modal)
  : KDialogBase( parent, name, modal, 
		 tr("Voice Recognition: microphone setup wizard"),
		 Ok|Cancel|User1|User2|User3,User1, true,
		 KGuiItem(tr("Next >>")),KGuiItem(tr("<< Previous")),
		 KGuiItem(tr("&About")))
{
  if ( !name )
    setName( "micro_setup" );

  dialogMicroBox= makeVBoxMainWidget ();

  title=new QLabel(tr("Step 4: " "Estimate Characteristics of Rec Channel") ,
		   dialogMicroBox); // the longest title
  title->setAlignment(Qt::AlignAuto|Qt::AlignTop);
  QFont qf(title->font());
  qf.setBold(true);
  qf.setStyleHint(QFont::SansSerif,
		  QFont::StyleStrategy(QFont::PreferAntialias|
				       QFont::PreferQuality));
  qf.setPointSize(30);
  title->setFont(qf);

  QHBox* boxForTwoFrames=new QHBox(dialogMicroBox);

  leftFrame=new QFrame (boxForTwoFrames);
  leftFrame->setFrameStyle (QFrame::Box|QFrame::Raised);
  leftFrame->setLineWidth(1);
  leftFrame->setMidLineWidth(2);


  QVBoxLayout * leftLayout = new QVBoxLayout(leftFrame,10,20);


  info=new QLabel(tr("This is two-step procedure: <OL>"
		     "<LI> adjusting Input Gain Level:<BR>"
		     "<B><FONT COLOR=RED>"
		     "   you have to speak at a conversational <BR>"
		     "   volume (speech recognition volume) while <BR>"
		     "   adjusting I-gain <BR><BR>"
		     "</B></FONT></LI>"
		     "<LI> adjusting Microphone Level:<BR>"
		     "<B><FONT COLOR=RED>"
		     "   you have to speak very loudly (laughing <BR>"
		     "   loudly is good) while adjusting Mic level.<BR>"
		     "</B></FONT></LI></OL>"
		     "<BR>"
		     "Click on the first button, then the second one <BR>"
		     "and 'Next' when done."),leftFrame);
  // the tallest label so that resize was not needed!
  info->setTextFormat(RichText);
  info->setAlignment(Qt::AlignAuto|Qt::AlignTop);
  leftLayout->addWidget(info,10,Qt::AlignTop|Qt::AlignLeft);
  boxForTwoFrames->setMinimumSize(500,300);


  //================== let's setup basic frames' features
  for(int i=0; i<MAXPAGE; i++)
    {
      rightFrame[i]=new QFrame (boxForTwoFrames);
      rightFrame[i]->setFrameStyle (QFrame::Box|QFrame::Raised);
      rightFrame[i]->setLineWidth(1);
      rightFrame[i]->setMidLineWidth(2);
      rightFrame[i]->setMinimumSize(QSize ( 250, 250 ));
      rightFrame[i]->hide();

      //      finishedOK[i]=false;
      //      isNotAvailable[i]=false;
    }

  //================== let's setup frame  **  0  ** :
  rightLayout = new QVBoxLayout(rightFrame[0],10,20);
  tmpLabel=new QLabel(" ",rightFrame[0]);
  rightLayout->addWidget(tmpLabel);

  titletxt[0]=new QString(tr("Step 0: introduction"));

  infotxt[0]=new 
    QString(tr("This wizard will guide you through microphone<BR>"
	       "calibration procedure.<BR><BR>"
	       "In every step you will see in this (left) frame<BR>"
	       "some description and information. Following these <BR>"
	       "choose actions from right frame>>>>>>>>>>>>><BR><BR>"
	       "Now click on the 'Next' button to start microphone<BR>"
	       "calibration."));

  warntxt[0]=new QString(tr("You haven't finished Microphone calibration yet! "));

  status[0] = invalid; // mixer
  status[1] = invalid; // audio
  status[2] = inactive;//"Adjust Mixer Levels"
  status[3] = inactive;//"Calculate Recording Thresholds"
  status[4] = inactive;//"Estimate Characteristics of Recording Channel"
  status[5] = inactive;//"Write Configuration"
  status[6] = inactive;  // exit /*wiecko*/

  // ok = set, with right value
  // invalid = available, but not set
  // inactive = not available
  //
  // page 0 is always available;
  // 0&1 refer to page 1
  // then, next statuses refer to the page of the same number
  // i.e. status[3] is the only criterium whether to show page 3
  // status[6] is quite obscure: it is set to ok once status[5] is ok,
  // so it is not really needed

  // other pages depend on available audio/mixer devices etc.
  // therefore they will be defined in delayed part of the constrtucot
  // (so that accept()/reject() was possible)

  currentPage=-1;
  setupPage(0);

  // disable everything for the time of singleShot
  enableButton(User2,false);
  enableButton(User1,false);
  enableButtonOK(false);
  enableButtonCancel(false);

  // wait with the rest of setup for showing the dialog 
  QTimer::singleShot( 1000, this, SLOT( constructorPart2()) );

}

void dVoiceMicroSetup::constructorPart2()
{
  //  Now, this is fun: you can't quit from constructor! At least not by
  // reject(), accept(). :(
  //
  //  Therefore I outsourced some checking code - it is run 1 second 
  // (single Shot) after the main window appears.


  // from Daniel's main():
  /* ***** detect available mixer devices */

  mixer_devices = scanMixerDevices();
  if (mixer_devices == NULL || mixer_devices->count == 0)
    {
      /* ***** no mixer devices available -> exit! */
      QString* msg=new 
	QString( tr("No mixer devices available!\n"
		    "If you have a sound card installed,\n"
		    "please check if some other program is\n"
		    "using the mixer device and turn it off."));
      exitWithError(msg);
    }
  else
    {
      if (mixer_devices->count == 1)/***** exactly one mixer device available*/
	{
	  setMixer(mixer_devices->name[0]); /***** set this mixer */
	  
	  if (initMixer() == MIXER_OK) /***** if mixer is ok, keep it */
	    {
	      status[0] = ok;
	      status[2] = invalid;
	    }
	  else                         /***** otherwise, exit!*/
	    {
	      QString* msg=new 
		QString(tr( "Couldn't init the only available mixer\n"
			    "device: /dev/%1"
			    ).arg(mixer_devices->name[0]));
	      exitWithError(msg);
	    }
	}
      else /* ***** more than one device available! */
	{
	  /* ***** use /dev/mixer as default if it exists */
	  
	  for (int i = 0; i < mixer_devices->count; i++)
	    {
	      if (strcmp(mixer_devices->name[i], "mixer") == 0)
		{
		  setMixer("mixer");
		  if (initMixer() == MIXER_OK)
		    {
		      status[0] = ok;
		      status[2] = invalid;
		    }
		  else
		    noMixer();
		  
		  break;
		}
	    }
	}
    }

  /* ***** detect available audio devices */

  audio_devices = scanAudioDevices();
  if (audio_devices == NULL || audio_devices->count == 0)
    {
      /* ***** no audio devices available! */
      QString* msg=new 
	QString(tr("No audio device available that\n"
		   "supports 16bit recording!\n\n"
		   "If you have a sound card installed,\n"
		   "please check if some other program is\n"
		   "using it (and turn it off!)."
		   ));
      exitWithError(msg);
    }
  else
    {
      if (audio_devices->count == 1)/***** exactly one audio device available*/
	{
	  setAudio(audio_devices->name[0]); /***** set this audio device */
	  
	  if (initAudio() == AUDIO_OK) /***** if audio device is ok, keep it */
	    {
	      status[1] = ok;
	    }
	  else                         /***** otherwise, exit!*/
	    {
	      QString* msg=new 
		QString(tr("Couldn't init the only available\n"
			   "audio device: /dev/%1\n"
			   ).arg(audio_devices->name[0]));
	      exitWithError(msg);
	    }
	}
      else /* ***** more than one device available! */
	{
	  /* ***** use /dev/dspW as default if it exists */
	  
	  for (int i = 0; i < audio_devices->count; i++)
	    {
	      if (strcmp(audio_devices->name[i], "dspW") == 0)
		{
		  setAudio("dspW");
		  if (initAudio() == AUDIO_OK)
		    status[1] = ok;
		  else
		    noAudio();
		  
		  break;
		}
	    }
	}
    }


  ///*****
  // * if mixer and audio device have been selected successfully,
  // * set menu cursor to next available menu item
  // *****/
  //  if (status[0] == ok && status[1] == ok)
  //    current = 2;


  //=============== start of WIDGET SETUP:

  //================== let's setup Page  **  1  ** :
  rightLayout = new QVBoxLayout(rightFrame[1],10,20);

  tmpLabel=new QLabel(tr("Available mixer devices:"),rightFrame[1]);
  rightLayout->addWidget(tmpLabel,0,Qt::AlignTop|Qt::AlignLeft);
  tmpHLayout=new QHBoxLayout(rightLayout);
  CB_mixer=new QComboBox( FALSE, rightFrame[1]);
  CB_mixer->insertItem(tr("(none selected)"));
  for(int i = 0; i<mixer_devices->count;i++)
    CB_mixer->insertItem(QString("/dev/")+QString(mixer_devices->name[i]));
  CB_mixer->setCurrentItem(0);
  comments_mixer=new QLabel(" ",rightFrame[1]);
  if (mixerOK() == MIXER_OK)
    {
      for(int i = 0; i<mixer_devices->count;i++)
	{
	  if((CB_mixer->text(i+1))==(QString(getMixer())))
	    {
	      CB_mixer->setCurrentItem(i+1);
	      comments_mixer->setText(tr("(OK)"));
	    }
	}
    }
  connect(CB_mixer,SIGNAL(activated(int)),SLOT(CB_mixer_activated(int)));
  tmpHLayout->addWidget(CB_mixer,0,Qt::AlignTop|Qt::AlignLeft);
  tmpHLayout->addWidget(comments_mixer,10);

  rightLayout->addSpacing(30);

  tmpLabel=new QLabel(tr("Available audio devices:"),rightFrame[1]);
  rightLayout->addWidget(tmpLabel,0,Qt::AlignTop|Qt::AlignLeft);
  tmpHLayout=new QHBoxLayout(rightLayout);
  CB_audio=new QComboBox( FALSE, rightFrame[1]);
  comments_audio=new QLabel(" ",rightFrame[1]);
  CB_audio->insertItem(tr("(none selected)"));
  for(int i = 0; i<audio_devices->count;i++)
    CB_audio->insertItem(QString("/dev/")+QString(audio_devices->name[i]));
  CB_audio->setCurrentItem(0);
  if(audioOK() == AUDIO_OK)
    for(int i = 0; i<mixer_devices->count;i++)
      if(CB_audio->text(i+1)==QString(getAudio()))
	{
	  CB_audio->setCurrentItem(i+1);
	}
  connect(CB_audio,SIGNAL(activated(int)),SLOT(CB_audio_activated(int)));
  tmpHLayout->addWidget(CB_audio,0,Qt::AlignTop|Qt::AlignLeft);
  tmpHLayout->addWidget(comments_audio,10);

  rightLayout->addStretch(20);

  titletxt[1]=new QString(tr("Step 1: " "Select a Mixer and an Audio Device"));

  if ((mixer_devices->count == 1)&& (audio_devices->count ==1))
    infotxt[1]= new 
      QString(tr("There is only one mixer and one audio device in<BR>"
		 "your system! They have already been selected.<BR><BR>"
		 "Just click 'Next'."));
  else if( mixer_devices->count == 1 )
    infotxt[1]= new 
      QString(tr("There is only one mixer device in your system! <BR>"
		 "It has already been selected.<BR><BR>"
		 "Select audio device and click 'Next'."));
  else if(audio_devices->count ==1)
    infotxt[1]= new 
      QString(tr("There is only one audio device in your system! <BR>"
		 "It has already been selected.<BR><BR>"
		 "Select mixer device and click 'Next'."));
  else 
    infotxt[1]= new 
      QString(tr("Please, choose mixer and audio devices,<BR>"
		 "and then  click 'Next'."));


  warntxt[1]=new QString(tr("You haven't finished Microphone calibration yet! "));

  //================== let's setup Page  **  2  ** :
  rightLayout = new QVBoxLayout(rightFrame[2],10,20);

  tmpHLayout=new QHBoxLayout(rightLayout);
  PBadjust_miclevel=new QPushButton(tr("Adjust Mic Input Gain/Level"),
				    rightFrame[2]);
  comment_miclevel=new QLabel(" ",rightFrame[2]);
  tmpHLayout->addWidget(PBadjust_miclevel,0,Qt::AlignTop|Qt::AlignLeft);
  tmpHLayout->addWidget(comment_miclevel,10,Qt::AlignTop|Qt::AlignLeft);
  connect(PBadjust_miclevel,SIGNAL(clicked()),this,SLOT(adjust_miclevel()));
  label_miclevel_value=new QLabel(" ",rightFrame[2]);
  rightLayout->addWidget(label_miclevel_value);

  rightLayout->addStretch(30);


  titletxt[2]=new QString(tr("Step 2: " "Adjust Mixer Levels"));

  infotxt[2]=new QString(tr("(empty string)")); // this depends on whether 
  //                                           igain is enabled!

  warntxt[2]=new QString(tr("You haven't finished Microphone calibration yet! "));


  //================== let's setup Page  **  3  ** :
  rightLayout = new QVBoxLayout(rightFrame[3],10,20);
  PBthreshold=new QPushButton(tr("Start threshold setup"),rightFrame[3]);
  rightLayout->addWidget(PBthreshold,0,Qt::AlignTop|Qt::AlignLeft);
  comment_threshold=new QLabel(" \n \n ",rightFrame[3]);
  rightLayout->addWidget(comment_threshold,10,Qt::AlignTop|Qt::AlignLeft);
  connect(PBthreshold,SIGNAL(clicked()),this,SLOT(PBthreshold_clicked()));

  titletxt[3]=new QString(tr("Step 3: "  "Calculate Recording Thresholds" ) );
  
  infotxt[3]=new 
    QString(tr("This is a two-step procedure. <OL>"
	       "<LI> first program will record silence level<BR>"
	       "    <FONT COLOR=RED><B>"
	       "    you will have to remain silent during this step"
	       "    </B></FONT></LI>"
	       "<LI> then program will try to estimate when to <BR>"
	       "    start recording <BR>"
	       "    <FONT COLOR=RED><B>"
	       "    you will have to talk at a conversational volume<BR>"
	       "    (speech recognition volume)"
	       "    </B></FONT></LI></OL><BR><BR>"
	       "Click on the 'Start threshold setup' button and<BR>"
	       "'Next' when done."));

  warntxt[3]=new QString(tr("You haven't finished Microphone calibration yet! "));

  //================== let's setup Page  **  4  ** :
  rightLayout = new QVBoxLayout(rightFrame[4],10,20);
  PBcharacter=new QPushButton(tr("Estimate Characteristics"),rightFrame[4]);
  rightLayout->addWidget(PBcharacter,0,Qt::AlignTop|Qt::AlignLeft);
  connect(PBcharacter,SIGNAL(clicked()),this,SLOT(PBcharacter_clicked()));


  titletxt[4]=new QString(tr("Step 4: " "Estimate Characteristics of Rec Channel") );

  infotxt[4]=new 
    //    QString(tr("<FONT COLOR=RED><B>You have to remain silent during"
    QString(tr("<FONT COLOR=RED><B>You have to speak naturally during"
	       " this step.</B></FONT><BR><BR>"
	       "Click 'Estimate Characteristics' to start and \n"
	       "'Next' when done." ));

  warntxt[4]=new QString(tr("You haven't finished Microphone calibration yet! "));

  //================== let's setup Page  **  5  ** :
  rightLayout = new QVBoxLayout(rightFrame[5],10,20);
  PBsave=new QPushButton(tr("Save Configuration"),rightFrame[5]);
  rightLayout->addWidget(PBsave,0,Qt::AlignTop|Qt::AlignLeft);
  connect(PBsave,SIGNAL(clicked()),this,SLOT(PBsave_clicked()));


  titletxt[5]=new QString(tr("Step 5: " "Save Configuration" ));

  infotxt[5]=new
    QString(tr("Click 'Save Config' to save your configuration in"
	       "<BR>~/.cvoicecontrol/config<BR>"
	       "and 'Next' when done."));

  warntxt[5]=new QString(tr("You haven't finished Microphone calibration yet! "));



  //================== let's setup Page  **  6  ** :
  rightLayout = new QVBoxLayout(rightFrame[6],10,20);
  tmpLabel=new QLabel(tr(" Microphone is ready! :) "),rightFrame[6]);
  rightLayout->addWidget(tmpLabel,0,Qt::AlignTop|Qt::AlignLeft);

  titletxt[6]=new QString(tr("Step 6: we are done"));

  infotxt[6]=new 
    QString(tr("This is the last page of microphone calibration."
	       "<BR><BR>"
	       "Note, that for successful voice recognition you <BR>"
	       "have to setup(record) a voice model now ! <BR>"
	       "That is, if you haven't done it before (see Help)"
	       "<BR><BR>"
	       " Click OK to finish microphone configuration."
	       ));

  warntxt[6]=new QString("OK, you are done "); //nobody will see it anyway!


  //================== end of frames' setup

  endPage=6;
  currentPage=endPage;

  // some extra setup, which required all pages' elements to be set up first:
  CB_mixer_activated(CB_mixer->currentItem());
  CB_audio_activated(CB_audio->currentItem());

  setupPage(0);
}

dVoiceMicroSetup::~dVoiceMicroSetup() {
  if (mixer_devices != NULL)
    {
      for (int i = 0; i < mixer_devices->count; i++)
	free(mixer_devices->name[i]);
      free(mixer_devices->name);
      free(mixer_devices);
    }

  if (audio_devices != NULL)
    {
      for (int i = 0; i < audio_devices->count; i++)
	free(audio_devices->name[i]);
      free(audio_devices->name);
      free(audio_devices);
    }
}

void dVoiceMicroSetup::setupPage(int number)
{
  // hiding/showing + button NEXT/PREVIOUS enabling
  int newPage=number;
  if(newPage<0) 
    newPage=0;
  if(newPage>=MAXPAGE)
    newPage=MAXPAGE-1;
  if (currentPage!=-1)
    rightFrame[currentPage]->hide();
  currentPage=newPage;
  rightFrame[currentPage]->show();
  
  info->setText(infotxt[currentPage]->latin1());
  title->setText(titletxt[currentPage]->latin1());

  enableButton(User2,true);
  enableButton(User1,false); //NEXT
  enableButtonOK(false);
  enableButtonCancel(true);

  if (currentPage==endPage)
    {

      enableButton(User1,false);
      if(status[6] == ok )
	{
	  enableButtonOK(true);
	  enableButtonCancel(false);
	}
    }
  if (currentPage==0)
    {
      enableButton(User2,false);
    }

  // setup for specific pages:
  switch(currentPage){
  case 0:
    enableButton(User1,true); 
    break;
  case 1:
    /*****
     * if mixer and audio device have been selected successfully,
     * set menu cursor to next available menu item
     *****/
    enableButton(User1,status[0] == ok && status[1] == ok);

    CB_mixer->setEnabled((status[0] == invalid)||(status[0]==ok));
    CB_audio->setEnabled((status[1] == invalid)||(status[1]==ok));

    break;
  case 2:
    setMicLevel(mic_level);
    if(mixerIgainEnabled)
      {
	debuginfo(tr("igain enabled"));
	delete infotxt[2]; // ? I don't think it is needed...
	infotxt[2]=new 
	  QString(tr("This is two-step procedure: <OL>"
		     "<LI> adjusting Input Gain Level:<BR>"
		     "<B><FONT COLOR=RED>"
		     "you have to speak at a conversational <BR>"
		     "volume (speech recognition volume) while<BR>"
		     "adjusting I-gain <BR><BR>"
		     "</FONT></B></LI>"
		     "<LI> adjusting Microphone Level:<BR>"
		     "<B><FONT COLOR=RED>"
		     "you have to speak very loudly (laughing <BR>"
		     "loudly is good) while adjusting Mic level.<BR>"
		     "</FONT></B></LI></OL>"
		     "<BR>"
		     "Click on 'Adjust Mic Input Gain/Level' to Start"
		     "<BR>and 'Next' when done."));
	info->setText(infotxt[currentPage]->latin1());
	PBadjust_miclevel->setEnabled(true);
	if(status[2]!=ok)
	  {
	    comment_miclevel->setText(" ");
	  }
	else
	  {
	    comment_miclevel->setText(tr("(OK)"));
	    enableButton(User1,true); //NEXT
	  }
      }
    else
      { // no igain!
       	debuginfo(tr("NO IGAIN"));
	delete infotxt[2];
	infotxt[2]= new 
	  QString(tr("Click on 'Adjust Mic Input Gain/Level' button:"
		     "<BR><FONT COLOR=RED><B>"
		     "   you have to speak very loudly (laughing "
		     "<BR>loudly is good) while adjusting Mic "
		     "level.<BR></B></FONT>"
		     "<BR>"
		     "Click on 'Next' when done."));
	info->setText(infotxt[currentPage]->latin1());
	if(status[2]!=ok)
	  comment_miclevel->setText(" ");
	else
	  {
	    comment_miclevel->setText(tr("(OK)"));
	    enableButton(User1,true); //NEXT	    
	  }
      }
    break;
  case 3:
    if(status[3]==ok)
      {
	enableButton(User1,true); //NEXT	    
      }
    break;
  case 4:
    if(status[4]==ok)
      {
	enableButton(User1,true); //NEXT	    
      }
    break;
  case 5:
    /***** if the configuration is done, activate the menu item 
	   "Save Configuration" */
    if (status[0] != ok || status[1] != ok ||
	status[2] != ok || status[3] != ok ||
	status[4] != ok)
      status[5] = inactive;
    else if (status[5] != ok)
      status[5] = invalid;

    PBsave->setEnabled(status[5] != inactive);

    if(status[5]==ok)
      {
	enableButton(User1,true); //NEXT	    
      }
    break;
  }

}

void dVoiceMicroSetup::slotOK()
{
  accept();
}

void dVoiceMicroSetup::slotCancel()
{
  if(QMessageBox::warning(this,tr("Microphone setup WARNING"),
			  tr( "You did not finish the configuration "
			      "process \n and no data has been written "
			      "to your configuration file. \n\nDo you "
			      "really want to quit!?")
			  ,QMessageBox::No | QMessageBox::Default | 
			  QMessageBox::Escape,QMessageBox::Yes)==
     QMessageBox::Yes)
    reject();
}

void dVoiceMicroSetup::slotClose()
{
  slotCancel();
}


void dVoiceMicroSetup::slotUser1() // NEXT
{
  setupPage(currentPage+1);
}

void dVoiceMicroSetup::slotUser2() // PREVIOUS
{
  setupPage(currentPage-1);
}

void dVoiceMicroSetup::slotUser3() // ABOUT
{
  infoPopup(tr("Microphone configuration for "
	       "voice recognition.\n\n"
	       "This is just an interface to "
	       "(and based on) \n "
	       "Daniel Kiecza's cvoicecontrol-0.9alpha\n\n"
	       "Modified for KLearnNotes2\n"
	       "(c) 2003 Marek Wieckowski <wiecko AT users.sourceforge.net>"),
	    tr("About microphone config"));
}

void dVoiceMicroSetup::exitWithError(const QString* txt) 
{
  QMessageBox::warning( this, tr("Microphone setup error!"),
			tr("ERROR!\n")+*txt,
			QMessageBox::Abort | QMessageBox::Default,
			QMessageBox::NoButton);
  reject();
}

void dVoiceMicroSetup::infoPopup(const QString txt,
				 const QString tit)
  //just shorthand; default tit="Microphone setup:"
{
  QMessageBox::information( this, tit,
			    txt);
}

void dVoiceMicroSetup::warningPopup(const QString txt,
				    const QString tit)
  //just shorthand; default tit="Microphone setup:"
{
  QMessageBox::warning( this, tit,
			txt);
}



void dVoiceMicroSetup::CB_mixer_activated(int devNumber)
{
  /********************************************************************
   * select mixer from list of available mixer devices
   *******************************************************************/
  // form: int selectMixer()
  status[0] = invalid;
  status[2] = inactive;
  status[3] = inactive;
  status[4] = inactive;
  status[5] = inactive;
  noMixer();
  comments_mixer->setText(" ");
  mixerIgainEnabled=false;
  // INITIAL VALUES OF SOME mixer VARIABLES:
  mic_level    = 99; /***** initial values for Microphone and */
  igain_level  = 0;  /***** Input Gain level */
  /***** max 16-bit sample value coming from the sound card :*/
  max_sample   = 32500; 


  if (devNumber>0) //0 == no device
    {
      /***** set mixer device to highlighted item */
      setMixer(mixer_devices->name[devNumber-1]); 
      if( initMixer() == MIXER_OK)
	{
     	  status[0] = ok;
	  status[2] = invalid;
	  comments_mixer->setText(tr("(OK)"));
	  if (mixerHasIGain() == MIXER_OK)
	    {
	      igain_level = 1;
	      setIGainLevel(igain_level); /***** set initial levels in mixer */
	      mixerIgainEnabled=true;
	      debuginfo("mixerIgainEnbld=true");
	    }
	}
      else
	comments_mixer->setText(tr("(doesn't work)"));
    }

  setupPage(1);

}

void dVoiceMicroSetup::CB_audio_activated(int devNumber)
{
  /**********************************************************************
   * select audio from list of available audio devices
   *********************************************************************/
  status[1] = invalid;
  status[3] = inactive;
  status[4] = inactive;
  status[5] = inactive;
  noAudio();
  comments_audio->setText(" ");
  if(devNumber>0)
    {
      /***** set audio device to highlighted item */
      setAudio(audio_devices->name[devNumber-1]);
      if (initAudio() == AUDIO_ERR)
	{
	  noAudio();
	  comments_audio->setText(tr("(doesn't work)"));
	}
      else
	{
	  status[1] = ok;
	  comments_audio->setText(tr("(OK)"));
	}
    }
  setupPage(1);
}



void dVoiceMicroSetup::adjust_miclevel()
{
  status[2] = invalid;
  status[3] = inactive;
  status[4] = inactive;

  if (mixerIgainEnabled){
    count    = 0;
    max_gain = 0;
    
    if (openAudio() == AUDIO_ERR)
      {
	/*****
	 * if the audio device could not be opened,
	 * show a warning dialog and exit
	 *****/
	exitWithError(new QString(tr("Failed to open sound device!!")));
      }
    
    infoPopup(tr("Please grab your "
		 "microphone and speak nonsense\n "
		 "at a conversational volume"
		 "(speech recognition volume).\n\n"
		 "Click OK to start"));
    
    /***** repeat until the input gain level has been adjusted properly */

    while (1)
      {
	int max;          /***** maximum sample value in a sequence of audio
				 samples */
	int samples = 10; /***** number of blocks to get from the audio
				 device */

	count++;

	/*****
	 * get the maximum values of 'samples' blocks of data from the sound
	 card
	 * the maximum of these values is stored in 'max'9
	 *****/
	max = 0;
	for (int i = 0; i < samples; i++)
	  {
	    int value = getBlockMax(); /***** should check for value == -1 
					      (case of an error!) */
	    if (value > max)
	      max = value;
	  }

	/***** max_gain holds the highest maximum sample found */

	if (max > max_gain)
	  max_gain = max;

	if (count == 5) /***** after five iterations, check the value of 
			       'max_gain' */
	  {
	    /*****
	     * if max_gain is too low (i.e. it is between silence and 
	     low_volume)
	     * the input gain level is increased
	     *****/
	    if (max_gain >= silence * max_sample && max_gain < low_volume * max_sample)
	      {
		igain_level++;
		if (igain_level > 99)
		  igain_level = 99;
		setIGainLevel(igain_level);

	      }
	    /*****
	     * if the level is above 'low_volume'
	     * we assume that the input gain is high enough and break the
	     current while loop
	    *****/
	    else if (max_gain >= low_volume * max_sample)
	      break;

	    /***** reset count and max_gain */

	    count    = 0;
	    max_gain = 0;
	  }
      }

    closeAudio(); /***** disconnect from microphone */
  
  }








  if (openAudio() == AUDIO_ERR)
    {
      /*****
       * if the audio device could not be opened,
       * show a warning dialog and exit
       *****/
      exitWithError(new QString("Failed to open sound device!!"));
    }

  /***** adjusting microphone level */
  tmpString=tr("Please, grab your microphone and speak very\n"
	       "loudly (laughing loudly is good). \n\n"
	       "Click OK to start...");
  if(mixerIgainEnabled)
    tmpString=tr("OK. Igain adjusted.:) Stop talking "
		 "now.\n\n "
		 "Now I will adjust Mic Level: \n")+
      tmpString;
  infoPopup(tmpString,tr("Adjusting Mic Level..."));

  label_miclevel_value->setText(tr("Current igain value: %1\n"
				   "Current Microphone Level: %2"
				   ).arg(igain_level).arg(mic_level));
  qApp->processEvents () ;

  mic_level    = 99; /***** initial values for Microphone */

  count = 0; /***** reset count */

  /***** repeat until the microphone level has been adjusted properly */
  while (1)
    {
      int max;         /***** maximum sample value in a sequence of audio
			      samples */
      int samples = 4; /***** number of blocks to get from the audio device */

      /*****
       * get the maximum values of 'samples' blocks of data from the sound card
       * the maximum of these values is stored in 'max'9
       *****/
      max = 0;
      for (int i = 0; i < samples; i++)
	{
	  int value = getBlockMax(); /***** should check for value == -1 
					    (case of an error!) */
	  if (value > max)
	    max = value;
	}

      if (max >= high_volume * max_sample)
	{
	  /*****
	   * if max is too high (i.e. above high_volume)
	   * the microphone level is dereased
	   *****/
	  mic_level -= 1;
	  if (mic_level < 5)
	    mic_level = 5;
	  setMicLevel(mic_level);

	  count = 0; /***** reset count */

	  /***** update display of mic level */
	  label_miclevel_value->setText(tr("Current igain value: %1\n"
					   "Current Microphone Level: %2"
					   ).arg(igain_level
						 ).arg(mic_level));
	  qApp->processEvents () ;
	}
      else if (max >= silence * max_sample)
	{
	  /*****
	   * if there is a signal above silence coming in,
	   * increase count
	   * if count >= a specified constant, we assume that
	   * the microphone level is not too high any more and
	   * thus, break the while loop
	   ****/
	  count++;

	  if (count >= 20)
	    break;
	}
      else
	{
	  /*****
	   * this was a silence frame, decrease count if it is > 0
	   *****/
	  if (count > 0)
	    count--;
	}
    }
  closeAudio(); /***** disconnect from microphone */


  if (igain_level >= mic_level ||
      mic_level < MIN_REASONABLE_MIC_LEVEL ||
      igain_level > MAX_REASONABLE_IGAIN_LEVEL)
    {
      tmpString=tr("You have to run this step once again!\n\n"
		   "The estimated level results don't look\n"
		   "reasonable to me!\n\n");
      if (igain_level >= mic_level || igain_level > MAX_REASONABLE_IGAIN_LEVEL)
	{
	  tmpString+=tr("- Input gain level looks too high!\n");
	  comment_miclevel->setText(tr("WRONG; run again"));
	}
      if (igain_level >= mic_level || mic_level < MIN_REASONABLE_MIC_LEVEL)
	{
	  tmpString+=tr("- Microphone level looks too low?!");
	  comment_miclevel->setText(tr("WRONG; run again"));
	}
      infoPopup(tmpString);
    }
  else /***** the mixer values seem ok */
    {
      status[2] = ok;
      status[3] = invalid;
      status[4] = invalid;
      infoPopup(tr("OK. Microphone input levels are set."));
    }
  setupPage(2);
}

void dVoiceMicroSetup::PBthreshold_clicked()
{
  status[3] = invalid;
  int samples;
  int max = 0, value;
  int silence_level_tmp; /***** need int (instead of short) to sum up several 
				values */
  int silence_max;
  infoPopup(tr("I'll calculate some values now. First,I need to\n"
	       "know the silence level of the microphone. \n \n"
	       "Please remain silent until next popup window\n"
	       "says 'silence level is set'.\n\n"
	       "Click OK to start..."),
	    tr("Calculate Recording Thresholds"));

  if (openAudio() == AUDIO_ERR)
    {
      /*****
       * if the audio device could not be opened,
       * show a warning dialog and return to main menu
       *****/
      exitWithError(new QString(tr("Failed to open sound device!!")));
    }
  /*****
   * define the silence_level as the average of a specified number
   * of subsequent block maxima
   *****/
  samples           = 40;
  silence_level_tmp = 0;
  silence_max   = 0;
  for (int i = 0; i < samples; i++)
    {
      int value = getBlockMax(); /***** should check for value == -1 
					(case of an error!) */
      silence_level_tmp += value;
      if (value > silence_max)
	silence_max = value;
    }
  silence_level_tmp /= samples;
  silence_level = silence_level_tmp;

  /***** rec_level and stop_level */

  /***** update dialog */
  infoPopup(tr("Silence level is set :) \n\n"
	       "Now, I'll define the thresholds at which \n"
	       "to start/stop recording. Please talk \n"
	       "at a conversational volume (speech \n"
	       "recognition volume) until I say stop!\n"
	       "\nClick OK to start ..."),
	    tr("Calculate Recording Thresholds"));


  /***** get maximum value 'max' of a prespecified number of subsequent block
	 maxima */

  samples = 80;
  for (int i = 0; i < samples; i++)
    {
      value = getBlockMax(); /***** should check for value == -1 
				    (case of an error!) */
      if (value > max)
	max = value;
    }

  /***** set the rec_level  to be the average of silence_level and 'max' */
  /***** set the stop_level to be three quarters of silence_level plus 
	 one quarter of 'max' */

  rec_level =  (silence_level + max) / 2;
  stop_level = (3*silence_level + max) / 4;

  closeAudio(); /***** disconnect from microphone */


  /***** check that the thresholds are reasonable */

  if (silence_level >= stop_level ||
      silence_max >= stop_level    ||
      silence_level < 0 || stop_level < 0 || rec_level < 0)
    {
    
      warningPopup(tr( "You have to run this step once again!\n"
		       "The calculated thresholds don't look\n"
		       "reasonable to me!"));
    }
  else /***** the values seem ok */
    {
      infoPopup(tr("You may stop talking now! The thresholds\n"
		   "have been defined!"),tr("Success!"));
      ///* Debugging stuff:
      //      comment_threshold->setText(tr("silence_level=%1\n"
      //					"stop_level=%1\n"
      //					"threshscr=%1"
      //					).arg(silence_level
      //					      ).arg(stop_level
      //						    ).arg(rec_level));
      //      //       */
      status[3] = ok;
    }

  setupPage(3);
}


void dVoiceMicroSetup::PBcharacter_clicked()
{
  int result;
  status[4] = invalid;
  infoPopup(tr("I'll calculate the characteristics of\n"
	       //	       "the recording channel now. Please remain\n"
	       //	       "silent until I say I'm done:\n\n"
	       "the recording channel now. Please speak\n"
	       "naturally until I say I'm done:\n\n"
	       "Click OK to start ..."),
	    tr("Estimating Channel Characteristics:"));

  if (openAudio() == AUDIO_ERR)
    {
      /*****
       * if the audio device could not be opened,
       * show a warning dialog and return to main menu
       *****/
      exitWithError(new QString(tr("Failed to open sound device!!")));
    }

  result = calculateChannelMean(); /***** calculate the characteristics 
					  of the recording channel */

  closeAudio(); /***** disconnect from the microphone */


  if (result == AUDIO_ERR)
    {
      /***** if an error occurred during the calculation of the channel 
	     mean ... */

      warningPopup( tr("I didn't manage to estimate the channel\n"
		       "characteristics for some unknown reason!\n"
		       "Make sure that no application uses the\n"
		       "sound card and then try again!\n"),
		    tr("Error!"));
    }
  else /***** channel mean calculation was ok */
    {
      infoPopup(tr("The channel characteristics have\n been estimated!"),
		tr("Success!"));
      status[4] = ok;
    }
  setupPage(4);
}

void dVoiceMicroSetup::PBsave_clicked()
{
  status[5] = invalid;
  status[6] = invalid;
  char *home;        /***** config file related variables */
  char *config_dir;
  char *config_file;
  FILE *f;

  /***** retrieve home directory */

  home = getenv("HOME");
  if (home != NULL)
    {
      FILE *f;

      /***** make sure the config_dir "~/.cvoicecontrol/" exists */

      config_dir = (char*) malloc(strlen(home) + strlen("/.cvoicecontrol/") + 1);
      strcpy(config_dir, home);
      strcat(config_dir, "/.cvoicecontrol/");

      if ((f = fopen(config_dir, "r")) == NULL)
	{
	  char *command = (char*)malloc(strlen("mkdir ") + strlen(config_dir) + 1);
	  strcpy(command, "mkdir ");
	  strcat(command, config_dir);
	  system(command);

	  if ((f = fopen(config_dir, "r")) == NULL)
	    {
	      free(config_dir);
	      config_dir =(char*) malloc(strlen("/tmp/") + 1);
	      strcpy(config_dir, "/tmp/");
	    }

	  free(command);
	}
      fclose(f);

      free(home);
    }
  else /***** couldn't retrieve home directory -> store results in /tmp/ */
    {
      config_dir =(char*) malloc(strlen("/tmp/") + 1);
      strcpy(config_dir, "/tmp/");
    }

  /***** tell user if home directory couldn't be retrieved and /tmp/ is 
	 used instead */

  if (strcmp(config_dir, "/tmp/") == 0)
    {
      warningPopup(tr("Failed to retrieve your home directory,\n"
		      "please contact your local system admin!\n"
		      "Configuration will be stored to /tmp/ instead!"));
    }

  /***** config_file = config_dir+"config" */

  config_file =(char*) malloc(strlen(config_dir) + strlen("config") + 1);
  strcpy(config_file, config_dir);
  strcat(config_file, "config");
  free (config_dir);

  if ((f = fopen(config_file, "w")) == NULL) /***** failed to write config 
						    file */
    {
      exitWithError(new QString(tr("Failed to create your configuration file!\n"
				   "Oops! What's going on?")));
    }

  /***** output configuration information to config file */

  fprintf(f, "Mixer Device    = %s\n", getMixer());
  fprintf(f, "Audio Device    = %s\n", getAudio());
  fprintf(f, "Mic Level       = %d\n", mic_level);
  fprintf(f, "IGain Level     = %d\n", igain_level);
  fprintf(f, "Record Level    = %d\n", rec_level);
  fprintf(f, "Stop Level      = %d\n", stop_level);
  fprintf(f, "Silence Level   = %d\n", silence_level);
  fprintf(f, "Channel Mean    =");
  for (int i = 0; i < FEAT_VEC_SIZE; i++)
    fprintf(f, " %6.5f", channel_mean[i]);
  fprintf(f, "\n");
  fclose(f);
  free(config_file);

  status[5] = ok;
  status[6] = ok;
  setupPage(5);
}	

///////////////////////////////////////////////////////
////////////////////////
//////////////////////// dialog MODEL EDITOR:
////////////////////////
///////////////////////////////////////////////////////
/***************************************************************************
 based on model_editor.h from cvoicecontrol-0.9alpha by Daniel Kiecza
***************************************************************************/

dModelEditor::dModelEditor(const QString filename,KLearnNotes2* parent,  
			   const char* name, bool modal)
  : KDialogBase(Tabbed,tr("Voice Recognition: speaker model editor"),
		Ok|Cancel|Help|User1|User2|User3,Cancel
		, parent, name, modal,true,
		KGuiItem(tr("&About")),KGuiItem(tr("Save as...")), 
		KGuiItem(tr("&Save")))
{
  if ( !name )
    setName( "voiceModelEditor" );
  papa=parent;

  //******************* PAGE1:
  page1 = addPage( tr("General") );
  page1Layout = new QVBoxLayout( page1, 0, 6 );
  tmpLayout=new QHBoxLayout(page1Layout);
  modelLabel=new QLabel(tr("Model:"), page1);
  tmpLayout->addWidget(modelLabel,0,Qt::AlignRight);
  lModelFile=new QLabel(tr("<B>filename</B>"),page1);
  tmpLayout->addWidget(lModelFile,0,Qt::AlignRight);
  tmpLayout->addStretch(10);
  PBnewModel=new QPushButton(tr("New Model"),page1);
  PBloadModel=new QPushButton(tr("Load Model"),page1);
  PBsaveModel=new QPushButton(tr("Save Model"),page1);
  PBsaveModelAs=new QPushButton(tr("Save Model As ..."),page1);
  page1Layout->addWidget(PBnewModel,0,Qt::AlignTop|Qt::AlignLeft);
  page1Layout->addWidget(PBloadModel,0,Qt::AlignTop|Qt::AlignLeft);
  page1Layout->addWidget(PBsaveModel,0,Qt::AlignTop|Qt::AlignLeft);
  page1Layout->addWidget(PBsaveModelAs,0,Qt::AlignTop|Qt::AlignLeft);
  page1Layout->addStretch(10);

  connect(PBnewModel,SIGNAL(clicked()),this,SLOT(startNewModel()));
  connect(PBloadModel,SIGNAL(clicked()),this,SLOT(PBloadModel_clicked()));
  connect(PBsaveModelAs,SIGNAL(clicked()),this,SLOT(PBsaveAs_clicked()));
  connect(PBsaveModel,SIGNAL(clicked()),this,SLOT(PBsave_clicked()));

  //******************** PAGE2:
  page2 = addPage( tr("Notes") );
  page2Layout = new QHBoxLayout( page2, 0, 6 );
  noteList= new QListView(page2);
  noteList->addColumn(tr("Note:"));
  noteList->addColumn(tr("Status:"));
  noteList->setSorting( -1 );
  noteList->setSelectionMode (QListView::Single);
  for(int i =0 ; i<7;i++)
    noteItems[i]=new 
      QListViewItem(noteList, QString(QChar(orderOfNotes [i])),"-");
  // note, that this takes care of English/German notation! :)
  //
  // items present in the model but not corresponding to any items of this
  // and the following lists are not touched by this code; therefore
  // if one chooses German notation and records samples for 'H', and
  // then switches to English notation letter 'B' is simply not-recognized
  // by the cvoicecontrol;
  // now one can edit the model and record samples for 'B';
  // this way there are both: samples for 'H' and for 'B'; both would be
  // reported by cvoicecontrol, but only one would cause some reaction
  // (depending on which notation is active in the moment)

  // note: KLN2 should report unrecognized input from cvoicecontrol!


  page2Layout->addWidget(noteList,0);
  page2Layout->addStretch(10);
  rightLayout2 = new QVBoxLayout(page2Layout);
  lComments2=new 
    QLabel(tr("Choose a note from the left box, then record\n"
	      "your sound samples for this note (i.e. the\n"
	      "sound with which you would like to 'click'\n"
	      "on the note). I need at least %1 samples for\n"
	      "each note.\n"
	      "See help for more suggestions!\n\n"
	      ).arg(MIN_NR_SAMPLES_PER_ITEM),page2);
  rightLayout2->addWidget(lComments2,0,Qt::AlignTop|Qt::AlignLeft);
  samplesList2 = new QListView( page2 );
  samplesList2->setFocusPolicy(QWidget::StrongFocus);
  samplesList2->addColumn(tr("samples for the selected note"));
  rightLayout2->addWidget(samplesList2);
  lRecordLeft2=new 
    QLabel(tr("You need to record %1 samples more for this note!\n"
	      ).arg(MIN_NR_SAMPLES_PER_ITEM), page2);
  rightLayout2->addWidget(lRecordLeft2);

  QGridLayout* buttonGrid2 = new QGridLayout( rightLayout2, 1, 1, 11, 
					      "ButtonLayout2");
  PBrecord2=new QPushButton(tr("Record a sample"),page2);
  PBrecord2->setIconSet(QIconSet(QPixmap(micOn_xpm)));
  QPushButton* PBplay2=new QPushButton(tr("Play selected sample"),page2);
  PBplay2->setIconSet(QIconSet(QPixmap(sound_on_xpm)));
  QPushButton* PBdelete2=new QPushButton(tr("Delete selected sample"),page2);
  PBdelete2->setIconSet(QIconSet(QPixmap(trash2_xpm)));
  QPushButton* PBdeleteAll2=new QPushButton(tr("Delete all samples"),page2);
  PBdeleteAll2->setIconSet(QIconSet(QPixmap(multitrash_xpm)));
  
  buttonGrid2->addWidget(PBrecord2,0,0);
  buttonGrid2->addWidget(PBplay2,1,0);
  buttonGrid2->addWidget(PBdelete2,0,1);
  buttonGrid2->addWidget(PBdeleteAll2,1,1);
  page2Layout->addStretch(10);


  connect(noteList,SIGNAL(selectionChanged ( QListViewItem * )),
	  this,SLOT( selectedNoteChanged(QListViewItem *)));
  connect(PBplay2,SIGNAL(clicked()),this,SLOT(PBplay2_clicked()));
  connect(samplesList2,SIGNAL(selectionChanged ()),
	  this,SLOT(PBplay2_clicked())); // selecting causes auto-play
  connect(PBdelete2,SIGNAL(clicked()),this,SLOT(PBdelete2_clicked()));
  connect(PBdeleteAll2,SIGNAL(clicked()),this,SLOT(PBdeleteAll2_clicked()));
  connect(PBrecord2,SIGNAL(clicked()),this,SLOT(PBrecord2_clicked()));

  //******************** PAGE3:
  page3 = addPage( tr("Program commands") );
  page3Layout = new QHBoxLayout( page3, 0, 6 );
  commandList= new QListView(page3);
  commandList->addColumn(tr("Command:"));
  commandList->addColumn(tr("Status:"));
  commandList->addColumn(tr("#")); // just for this code: number of what was
  //                                  selected
  commandList->setRootIsDecorated ( true);
  commandList->setSorting( -1 );
  commandList->setSelectionMode (QListView::Single);
  commandList->setAllColumnsShowFocus (true);
  head[3]=new QListViewItem(commandList,tr("other"),"");
  commandItems[16]=new QListViewItem(head[3],txt_no,
				     "-",QString("%1").arg(16));
  commandStrings[16][0]=new QString(cmd_no);
  commandStrings[16][1]=new QString(txt_no);
  commandStrings[16][2]=new QString(sug_no);
  commandItems[15]=new QListViewItem(head[3],txt_yes,
				     "-",QString("%1").arg(15));
  commandStrings[15][0]=new QString(cmd_yes);
  commandStrings[15][1]=new QString(txt_yes);
  commandStrings[15][2]=new QString(sug_yes);
  commandItems[14]=new QListViewItem(head[3],txt_quit,
				     "-",QString("%1").arg(14));
  commandStrings[14][0]=new QString(cmd_quit);
  commandStrings[14][1]=new QString(txt_quit);
  commandStrings[14][2]=new QString(sug_quit);
  commandItems[13]=new QListViewItem(head[3],txt_help,
				     "-",QString("%1").arg(13));
  commandStrings[13][0]=new QString(cmd_help);
  commandStrings[13][1]=new QString(txt_help);
  commandStrings[13][2]=new QString(sug_help);
  head[2]=new QListViewItem(commandList,tr("sound"),"");
  commandItems[12]=new QListViewItem(head[2],txt_quieter,
				     "-",QString("%1").arg(12));
  commandStrings[12][0]=new QString(cmd_quieter);
  commandStrings[12][1]=new QString(txt_quieter);
  commandStrings[12][2]=new QString(sug_quieter);
  commandItems[11]=new QListViewItem(head[2],txt_louder,
				     "-",QString("%1").arg(11));
  commandStrings[11][0]=new QString(cmd_louder);
  commandStrings[11][1]=new QString(txt_louder);
  commandStrings[11][2]=new QString(sug_louder);
  commandItems[10]=new QListViewItem(head[2],txt_sndSetup,
				     "-",QString("%1").arg(10));
  commandStrings[10][0]=new QString(cmd_sndSetup);
  commandStrings[10][1]=new QString(txt_sndSetup);
  commandStrings[10][2]=new QString(sug_sndSetup);
  commandItems[9]=new QListViewItem(head[2],txt_sndOn,
				    "-",QString("%1").arg(9));
  commandStrings[9][0]=new QString(cmd_sndOn);
  commandStrings[9][1]=new QString(txt_sndOn);
  commandStrings[9][2]=new QString(sug_sndOn);
  commandItems[8]=new QListViewItem(head[2],txt_sndOff,
				    "-",QString("%1").arg(8));
  commandStrings[8][0]=new QString(cmd_sndOff);
  commandStrings[8][1]=new QString(txt_sndOff);
  commandStrings[8][2]=new QString(sug_sndOff);
  head[1]=new QListViewItem(commandList,tr("exercise"),"");
  commandItems[7]=new QListViewItem(head[1],txt_none,
				    "-",QString("%1").arg(7));
  commandStrings[7][0]=new QString(cmd_none);
  commandStrings[7][1]=new QString(txt_none);
  commandStrings[7][2]=new QString(sug_none);
  commandItems[6]=new QListViewItem(head[1],txt_all,
				    "-",QString("%1").arg(6));
  commandStrings[6][0]=new QString(cmd_all);
  commandStrings[6][1]=new QString(txt_all);
  commandStrings[6][2]=new QString(sug_all);
  commandItems[5]=new QListViewItem(head[1],txt_length,
				    "-",QString("%1").arg(5));
  commandStrings[5][0]=new QString(cmd_length);
  commandStrings[5][1]=new QString(txt_length);
  commandStrings[5][2]=new QString(sug_length);
  commandItems[4]=new QListViewItem(head[1],txt_next,
				    "-",QString("%1").arg(4));
  commandStrings[4][0]=new QString(cmd_next);
  commandStrings[4][1]=new QString(txt_next);
  commandStrings[4][2]=new QString(sug_next);
  commandItems[3]=new QListViewItem(head[1],txt_treble,
				    "-",QString("%1").arg(3));
  commandStrings[3][0]=new QString(cmd_treble);
  commandStrings[3][1]=new QString(txt_treble);
  commandStrings[3][2]=new QString(sug_treble);
  commandItems[2]=new QListViewItem(head[1],txt_bass,
				    "-",QString("%1").arg(2));
  commandStrings[2][0]=new QString(cmd_bass);
  commandStrings[2][1]=new QString(txt_bass);
  commandStrings[2][2]=new QString(sug_bass);
  head[0]=new QListViewItem(commandList,tr("test"),"");
  commandItems[1]=new QListViewItem(head[0],txt_stop,
				    "-",QString("%1").arg(1));
  commandStrings[1][0]=new QString(cmd_stop);
  commandStrings[1][1]=new QString(txt_stop);
  commandStrings[1][2]=new QString(sug_stop);
  commandItems[0]=new QListViewItem(head[0],txt_start,
				    "-",QString("%1").arg(0));
  commandStrings[0][0]=new QString(cmd_start);
  commandStrings[0][1]=new QString(txt_start);
  commandStrings[0][2]=new QString(sug_start);

  for(int i =0;i<4;i++)
    head[i]->setOpen(true);
  page3Layout->addWidget(commandList,0);

  page3Layout->addSpacing(10);
  page3Layout->addStretch(10);
  rightLayout3 = new QVBoxLayout(page3Layout);
  lComments3=new 
    QLabel(tr("Choose a command from the left box, then record\n"
	      "your sound samples for this command (i.e. the\n"
	      "sound with which you would like to activate\n"
	      "this command). I need at least %1 samples for\n"
	      "each command.\n"
	      "See help for more suggestions!\n\n"
	      ).arg(MIN_NR_SAMPLES_PER_ITEM),page3);
  rightLayout3->addWidget(lComments3);//,0,Qt::AlignTop|Qt::AlignLeft
  rightLayout3->addStretch(10);
  samplesList3 = new QListView( page3 );
  samplesList3->setFocusPolicy(QWidget::StrongFocus);
  samplesList3->addColumn(tr("samples for the selected command"));
  rightLayout3->addWidget(samplesList3);
  lRecordLeft3=new 
    QLabel(tr("You need to record %1 samples more for this command!\n"
	      ).arg(MIN_NR_SAMPLES_PER_ITEM), page3);
  rightLayout3->addWidget(lRecordLeft3);

  QGridLayout* buttonGrid3 = new QGridLayout( rightLayout3, 1, 1, 11, 
					      "ButtonLayout3");
  PBrecord3=new QPushButton(tr("Record a sample"),page3);
  PBrecord3->setIconSet(QIconSet(QPixmap(micOn_xpm)));
  PBplay3=new QPushButton(tr("Play selected sample"),page3);
  PBplay3->setIconSet(QIconSet(QPixmap(sound_on_xpm)));
  PBdelete3=new QPushButton(tr("Delete selected sample"),page3);
  PBdelete3->setIconSet(QIconSet(QPixmap(trash2_xpm)));
  QPushButton* PBdeleteAll3=new QPushButton(tr("Delete all samples"),page3);
  PBdeleteAll3->setIconSet(QIconSet(QPixmap(multitrash_xpm)));

  buttonGrid3->addWidget(PBrecord3,0,0);
  buttonGrid3->addWidget(PBplay3,1,0);
  buttonGrid3->addWidget(PBdelete3,0,1);
  buttonGrid3->addWidget(PBdeleteAll3,1,1);
  page3Layout->addStretch(10);


  connect(commandList,SIGNAL(selectionChanged ( QListViewItem * )),
	  this,SLOT( selectedCommandChanged(QListViewItem *)));
  connect(PBplay3,SIGNAL(clicked()),this,SLOT(PBplay3_clicked()));
  connect(samplesList3,SIGNAL(selectionChanged ()),
	  this,SLOT(PBplay3_clicked())); // selecting causes auto-play
  connect(PBdelete3,SIGNAL(clicked()),this,SLOT(PBdelete3_clicked()));
  connect(PBdeleteAll3,SIGNAL(clicked()),this,SLOT(PBdeleteAll3_clicked()));
  connect(PBrecord3,SIGNAL(clicked()),this,SLOT(PBrecord3_clicked()));




  model_file_name = NULL;  /***** name of speaker model file */
  modified = 0;            /***** tells whether the model has been modified */

  /***** setup speaker model */
  model = (Model *) malloc(sizeof(Model));
  initModel(model);
  modelFileName=filename;
  lastNoteItem=NULL;
  lastCommandItem=NULL;
  
  QTimer::singleShot( 1000, this, SLOT( constructorPart2()) );
}

void dModelEditor::constructorPart2()
{
  /***** load configuration */

  if (loadConfiguration () == 0)
    {
      exitWithError(new 
		    QString(tr( "Couldn't load configuration! \n"
				"Have you already run Microphone Calibration?")));
    }
  loadFile(modelFileName);
}

dModelEditor::~dModelEditor()
{
  resetModel(model);
  if (model_file_name != NULL) /***** if a file name had been 
				      specified, ... */
    free(model_file_name);     /***** free any memory used by it */
  model_file_name = NULL;
}

void dModelEditor::exitWithError(const QString* txt) 
{
  QMessageBox::warning( this, tr("Microphone setup error!"),
			QString(tr("ERROR!\n"))+*txt,
			QMessageBox::Abort | QMessageBox::Default,
			QMessageBox::NoButton);
  reject();
}

void dModelEditor::infoPopup(const QString txt,
			     const QString tit)
  //just shorthand; default tit="Voice speaker model editor:"
{
  QMessageBox::information( this, tit,
			    txt);
}

void dModelEditor::warningPopup(const QString txt,
				const QString tit)
  //just shorthand; default tit="Voice speaker model editor:"
{
  QMessageBox::warning( this, tit,
			txt);
}

void dModelEditor::slotUser1() // ABOUT
{
  infoPopup(tr("Voice Model Editor for "
	       "voice recognition.\n\n"
	       "This is just an interface to "
	       "(and based on) \n "
	       "Daniel Kiecza's cvoicecontrol-0.9alpha\n\n"
	       "Modified for KLearnNotes2\n"
	       "(c) 2003 Marek Wieckowski\n<wiecko AT users.sourceforge.net>"),
	    tr("About Voice Model Editor"));
}

void dModelEditor::slotUser2() // Save as...
{
  PBsaveAs_clicked();
}

void dModelEditor::slotUser3() // Save
{
  PBsave_clicked();
}

// NOTE: OK/CANCEL: shouldn't it free some malloced memory???????

void dModelEditor::slotOk() // SAVE?
{// the difference: slotOk sets parent's (KLN2) model filename;
  // slotCancel - doesn't
  if(modified)
    {
      int inforesult=QMessageBox::Cancel;
      inforesult=
	QMessageBox::information(this,
				 tr("Speaker Model Editor WARNING"),
				 tr("Your voice model (speaker model)\n"
				    "has not been saved! \n\n"
				    "Do you want to save it now?")
				 ,QMessageBox::No,
				 QMessageBox::Yes|QMessageBox::Default,
				 QMessageBox::Cancel|QMessageBox::Escape);
      if(inforesult==QMessageBox::Yes)
	{
	  PBsave_clicked();//SAVE or "save as" if no filename
	}
      else if(inforesult ==QMessageBox::Cancel)
	{
	  return;
	}
    }
  // not modified OR user didn't want to save OR "Okeyed and Saved"
  // i.e. all but QMB::Cancel
  papa->setVoiceModelFilename(modelFileName);
  accept();
}

void dModelEditor::slotCancel() // Cancel
{
  if(modified)
    {
      if(QMessageBox::warning(this,tr("Speaker Model Editor WARNING"),
			      tr("You did not save the current voice\n"
				 "model (speaker model)! If you abort\n"
				 "now all changes in the model will \n"
				 "be lost!!\n"
				 "\nDo you really want to quit!?")
			      ,QMessageBox::No | QMessageBox::Default | 
			      QMessageBox::Escape,QMessageBox::Yes)==
	 QMessageBox::Yes)
	{
	  modified = 0;     /***** we reset the speaker model */	  
	  reject();
	}
    }
  else // not modified
    reject();
}

void dModelEditor::slotClose() // window Closes
{
  slotCancel();
}

int dModelEditor::safeResetModel()
{// quite similar to "Cancel" just without reject();
  // used e.g. when a new model is loaded/created

  /*****
   * if speaker model has been modified,
   * display a "warning" dialog to avoid an accidental loss of data
   *****/

  if (modified)
    {
      if(QMessageBox::warning(this,tr("Speaker Model Editor WARNING"),
			      tr( "You did not save the current voice\n"
				  "model (speaker model)! If you \n"
				  "continue now all changes in the\n"
				  "model will be lost!!\n"
				  "\nDo you really want to continue!?")
			      ,QMessageBox::No | QMessageBox::Default | 
			      QMessageBox::Escape,QMessageBox::Yes)==
	 QMessageBox::Yes)
	{
	  return(1);
	}
      else                /***** any other key -> keep speaker model */
	return(0);
    }
  return(1);   /***** if speaker model hasn't been modified, return ok */
}

void dModelEditor::startNewModel()
{
  if(safeResetModel())
    {
      modified = 0;     /***** we reset the speaker model */
      resetModel(model);
      if (model_file_name != NULL) /***** if a file name had been
					  specified, ... */
	free(model_file_name);     /***** free any memory used by it */
      model_file_name = NULL;
      addMissingItems();
      lModelFile->setText(QString("<B> </B>"));
      modelFileName=QString("");
    }
  updateModelViews();
}


/*****************************************************************************
 * play a sample's wave
 *****************************************************************************/

int  dModelEditor::playSample(ModelItemSample *sample)
{
  /***** play sample->wav_data */
  if (initAudio() == AUDIO_ERR) /***** make sure the audio device is
				       initialized properly */
    return AUDIO_ERR;
  /***** call playUtterance in audio.c to actually play the utterance */
  return (playUtterance(sample->wav_data, sample->wav_length));
}

/*****************************************************************************
 * record a sample utterance
 *****************************************************************************/

ModelItemSample * dModelEditor::recordSample()
{
  /***** allocate memory for the new sample */

  ModelItemSample *new_sample = 
    (ModelItemSample *) malloc(sizeof(ModelItemSample));

  char *tmp_string;
  time_t timer;     /***** used to get the current time, which will become */
  time(&timer);     /***** the main part of the new utterance's ID */

  /***** initialize the audio device */

  if (initMixer() == MIXER_ERR) /***** if mixer error, return nothing */
    {
      free(new_sample);
      return NULL;
    }
  if (igain_level > 0)
    setIGainLevel(igain_level); /***** set IGain and Mic level according */
  setMicLevel(mic_level);     /***** to configuration */

  if (initAudio() == AUDIO_ERR) /***** if audio error, return nothing */
    {
      free(new_sample);
      return NULL;
    }

  /***** connect to microphone, get utterance, disconnect */

  openAudio();
  new_sample->wav_data = getUtterance(&new_sample->wav_length);
  closeAudio();


  if (new_sample->wav_data == NULL) /***** if nothing was recorded, 
					   return nothing */
    {
      new_sample->wav_length = 0;
      new_sample->has_wav    = 0;
      free(new_sample);
      return NULL;
    }
  else
    /***** flag says that this sample utterance also contains its original
     * wave data, not only the preprocessed feature vectors
     *****/
    new_sample->has_wav    = 1;

  /***** preprocess the wave data */

  new_sample->data = preprocessUtterance(new_sample->wav_data, 
					 new_sample->wav_length, 
					 &new_sample->length);

  if (new_sample->data == NULL) /***** if preprocessing failed, return 
				       nothing */
    {
      new_sample->length = 0;
      free(new_sample->wav_data);
      free(new_sample);
      return NULL;
    }

  /***** set ID */

  tmp_string = ctime(&timer); /***** get current time */

  /***** set sample ID looks like:  [Thu Feb 10 12:10:53 2000] */

  new_sample->id = (char*) malloc(strlen(tmp_string)+2);
  new_sample->id[0] = '[';
  strcpy(new_sample->id+1, tmp_string);
  new_sample->id[strlen(tmp_string)] = ']';

  new_sample->next   = NULL; /***** next sample pointer is NULL */
  {
    int i;
    for (i = 0; i < 3; i++)
      new_sample->matrix[i] = NULL;
  }

  modified = 1; /***** speaker model has been modified now */

  return(new_sample);
}


void dModelEditor::loadFile(const QString qsFileName)
{
  if(qsFileName!=QString(""))
    {
      char *file_name;
      file_name=(char*)malloc(strlen(qsFileName.latin1())+1);
      strcpy(file_name,qsFileName.latin1());
      /***** display information that model is being loaded */
      if (loadModel(model,file_name, 1) == 1) 
	/***** loading model is successful */
	{
	  model_file_name = (char*)malloc(strlen(file_name) + 1);
	  strcpy(model_file_name, file_name);
	  lModelFile->setText(QString("<B>")+qsFileName+"</B>");
	  modelFileName=qsFileName;
	  // MESS WITH MULTIPLE THINGS KEEPING FILENAMES!!!!!!!!!!!!!!!!!!!
	}
      else /***** loading model has failed */
	{ 
	  warningPopup(tr("Loading model file failed!"));
	  free (file_name);
	}
    }
  addMissingItems();
  updateModelViews();
}


int dModelEditor::noteItemNumber(QListViewItem * it)
{
  // returns postion of item it in model or -1 if not found

  if(it == NULL) 
    return(-1);
  QString txt=it->text (0);
  int result=-1;
  for(int i =0; i<model->number_of_items;i++)
    if(QString(getModelItem(model,i)->label)==txt)
      result=i;
  return(result);
}

int dModelEditor::commandItemNumber(QListViewItem * it)
{
  // returns postion of item it in model or -1 if not found
  if(it == NULL) 
    return(-1);
  QString txt=*(commandStrings[it->text(2).toInt()][0]);
  int result=-1;
  for(int i =0; i<model->number_of_items;i++)
    if(QString(getModelItem(model,i)->label)==txt)
      result=i;
  return(result);
}


void dModelEditor::addMissingItems()
{
  for(int i=0;i<7;i++)
    if(noteItemNumber(noteItems[i])==-1)
      {
	appendEmptyModelItem(model, noteItems[i]->text(0).latin1(), "");
	noteItems[i]->setText(1,"-");
      }
  for(int i=0;i<NUMCOMMANDS;i++)
    if(commandItemNumber(commandItems[i])==-1)
      {
	appendEmptyModelItem(model,(commandStrings[i][0])->latin1(), "");
      }
}

void dModelEditor::deleteEmptyItems()
{
  for(int i=0;i<7;i++)
    if(noteItems[i]->text(1)=="-")
      {
	int j=noteItemNumber(noteItems[i]);
	if (j!=-1)
	  deleteModelItem(model,j );
      }
  
  for(int i=0;i<NUMCOMMANDS;i++)
    if(commandItems[i]->text(1)=="-")
      {
	int j=commandItemNumber(commandItems[i]);
	if (j!=-1)
	  deleteModelItem(model,j );
      }
}

void dModelEditor::debugModel()
{
  // dumps model info to stdout

  debuginfo("============= model dump: =================");
  for(int i=0;i<model->number_of_items;i++)
    debuginfo(QString("%1:    Samples: %2       Label:%3"
		      ).arg(i
			    ).arg(getModelItem(model,i)->number_of_samples
				  ).arg(getModelItem(model,i)->label));
}




void dModelEditor::selectedNoteChanged(QListViewItem * newItem)
{
  if((lastNoteItem!= NULL) && (lastNoteItem!=newItem))
    {
      int k =noteItemNumber(lastNoteItem);
      if (k>=0)
 	if (getModelItem(model,k)->number_of_samples < 
 	    MIN_NR_SAMPLES_PER_ITEM) /***** not enoug samples to save! */
 	  {
	    if(getModelItem(model,k)->number_of_samples ==0)
	      lastNoteItem->setText(1,"-");
	    else
	      {
		warningPopup(tr("We need at least %1 sample utterances\n"
				"for each speaker model item!!!\n\n"
				"Please donate the minimum number of \n"
				"samples for this item or delete all \n"
				"samples do deactivate this item!"
				).arg(MIN_NR_SAMPLES_PER_ITEM));
		//		lastNoteItem->setSelected ( true);
		//		noteList->repaintItem(lastNoteItem);
		//		newItem->setSelected ( false);
		//		noteList->repaintItem(newItem);
		noteList->setSelected ( lastNoteItem, true);
		return;
	      }
	  }
    }
  // a new selection; setup right list:
  if(newItem==0)// nothing selected
    {
      samplesList2->clear();
      lRecordLeft2->setText("");
      lComments2->
	setText(tr("Choose a note from the left box, then record\n"
		   "your sound samples for this note (i.e. the\n"
		   "sound with which you would like to 'click'\n"
		   "on the note). I need at least %1 samples for\n"
		   "each note.\n"
		   "See help for more suggestions!\n\n"
		   ).arg(MIN_NR_SAMPLES_PER_ITEM));
    }
  else // there is a note selected
    {
      lComments2-> 
	setText(tr("Notes' names are very short utterances \n"
		   "(sounds). Therefore, they are particularly\n"
		   "hard to record and recognize. \n"
		   "MAKE THEM LONGER! Say 'see-ee' not 'si' for C,\n"
		   "'aeaeaeff', not 'aff' for F and so on. \n\n"
		   "If in problems - see HELP for other hints!"));
      samplesList2->clear();
      ModelItem* model_item= getModelItem(model,noteItemNumber(newItem));
      for(int i=0; i<model_item->number_of_samples;i++)
	new QListViewItem( samplesList2,
			   getModelItemSample(model_item,i)->id);
      if(model_item->number_of_samples<MIN_NR_SAMPLES_PER_ITEM)
	{
	  lRecordLeft2->setText(tr("You need to record at least %1"
				   " samples more for this note"
				   ).arg(MIN_NR_SAMPLES_PER_ITEM -
					 model_item->number_of_samples)
				);
	  newItem->setText(1,tr("%1 lacking"
				).arg(MIN_NR_SAMPLES_PER_ITEM-
				      model_item->number_of_samples));
	}
      else
	{
	  lRecordLeft2->setText(tr("OK. You have at least %1 samples. :)"
				   "\nStill, you may record more if you "
				   "wish."
				   ).arg(MIN_NR_SAMPLES_PER_ITEM)

				);
	  newItem->setText(1,tr("(OK)"));
	}
    }
  lastNoteItem=newItem;
}



void dModelEditor::PBsave_clicked()
{
  deleteEmptyItems();
  if (model->number_of_items > 0)
    {
      if(modelFileName==QString(""))
	PBsaveAs_clicked();
      else
	save(modelFileName);
    }
  else
    {
      infoPopup(tr("This voice model is still empty!"));
    }
  addMissingItems();
}

void dModelEditor::PBsaveAs_clicked()
{
  /***** save the speaker model */
  // delete items with zero samples:
  deleteEmptyItems();
  if (model->number_of_items > 0)
    {
      //      QString 
	//qsFileName(KFileDialog::getSaveFileName(QDir::homeDirPath ()+
	//				"/.klearnnotes2_d/"
	//				"voice_models/",
	//				QString("*.cvc"),
	//				this,
	//				tr("Save voice model as...")));
      QString 
	qsFileName(QFileDialog::getSaveFileName(QDir::homeDirPath ()+
						"/.klearnnotes2_d/"
						"voice_models/",
						QString("*.cvc"),
						//+*model_file_extension,
						this,"save as dialog",
						tr("Save voice model as...")));
      
      if(qsFileName!=QString::null)
	save(qsFileName);
    }
  else
    {
      infoPopup(tr("This voice model is still empty!"));
    }
  addMissingItems();
}

void dModelEditor::save(QString qsFileName)
{
  char *file_name;
  if(QString(model_file_extension)!=
     qsFileName.right(strlen(model_file_extension)))
    qsFileName+=QString(model_file_extension);
  file_name = (char*) malloc( strlen(qsFileName.latin1())+1);
  strcpy(file_name,qsFileName.latin1());
  if (saveModel(model, file_name) == 1) /***** saving successful */
    {
      /***** update model_file_name if necessary */
      if (model_file_name != NULL && strcmp(model_file_name, file_name) != 0)
	{
	  free(model_file_name);
	  model_file_name = (char*)malloc(strlen(file_name) + 1);
	  strcpy(model_file_name, file_name);
	}
	
      modified = 0; /***** switch modified to '0' */
      lModelFile->setText(QString("<B>")+qsFileName+"</B>");
      modelFileName=qsFileName;
    }
  else /***** saving failed */
    {
      warningPopup(tr("Saving file failed!"));
    }
  free (file_name);
  updateModelViews();
}
  

void dModelEditor::PBloadModel_clicked()
{
  if (!safeResetModel())
    return;
  QString 
    qsFileName(KFileDialog::getOpenFileName(QDir::homeDirPath ()+
					    "/.klearnnotes2_d/voice_models/",
					    QString("*")+*model_file_extension,
					    this,
					    tr("Select a model file to load")));
  if(qsFileName!=QString::null)
    loadFile(qsFileName);
}


void dModelEditor::updateModelViews()
{
  // used after saves/loads and NOT after individual changes!!!!
  for (int i =0; i<7;i++)
    {
      int k = noteItemNumber(noteItems[i]);
      if (k == -1)
	noteItems[i]->setText(1,"-");
      else
	{
	  int j = getModelItem(model,k)->number_of_samples;
	  if(j==0)
	    noteItems[i]->setText(1,"-");
	  else if (j<MIN_NR_SAMPLES_PER_ITEM )
	    noteItems[i]->setText(1,"?");
	  else
	    noteItems[i]->setText(1,tr("(OK)"));
	}
    }
  for (int i =0; i<NUMCOMMANDS;i++)
    {
      int k = commandItemNumber(commandItems[i]);
      if (k == -1)
	commandItems[i]->setText(1,"-");
      else
	{
	  int j = getModelItem(model,k)->number_of_samples;
	  if(j==0)
	    commandItems[i]->setText(1,"-");
	  else if (j<MIN_NR_SAMPLES_PER_ITEM )
	    commandItems[i]->setText(1,"?");
	  else
	    commandItems[i]->setText(1,tr("(OK)"));
	}
    }

  lastNoteItem=NULL;
  QListViewItem * tmpItem=noteList->selectedItem ();
  if(tmpItem!=0)
    noteList->setSelected (tmpItem, false);
  lastCommandItem=NULL;
  tmpItem=commandList->selectedItem ();
  if(tmpItem!=0)
    commandList->setSelected (tmpItem, false);
  for(int i =0; i<4; i++)
    commandList->setOpen ( head[i], true );
}

void dModelEditor::PBplay2_clicked()
{
  QListViewItem* note= noteList->selectedItem ();
  if(note==0)
    return;
  int k =noteItemNumber(note);
  if(k==-1)
    return;
  QListViewItem* sample= samplesList2->selectedItem ();
  if(sample==0)
    return;
  ModelItem* model_item= getModelItem(model,k);
  if(model_item->number_of_samples==0)
    return; //sth wrong: this shouldn't happen; if there are no
  //          samples in model there shouldn't be any in sampleList2!
  int sampleNum=sampleNumber(model_item, sample->text(0));
  if (sampleNum==-1)
    return;
  /***** play wave if present */
  if (getModelItemSample(model_item, sampleNum)->has_wav)
    {
      /***** if play fails or was canceled ....*/
      if (playSample(getModelItemSample(model_item, sampleNum))!= AUDIO_OK)
	{
	  /***** display information */
	  warningPopup(tr("Either playback has been canceled \n"
			  "or wave couldn't be sent to your   \n"
			  "soundcard !                         "));
	}
    }
  else /***** no wave data present! */
    {
      warningPopup(tr("Utterance can't be played! No wave  \n"
		      "data available! Note: CVoiceControl \n"
		      "did NOT save the original wave data!"));
    }   
}

void dModelEditor::PBdelete2_clicked()
{
  QListViewItem* note= noteList->selectedItem ();
  if(note==0)
    return;
  int k =noteItemNumber(note);
  if(k==-1)
    return;
  QListViewItem* sample= samplesList2->selectedItem ();
  if(sample==0)
    return;
  ModelItem* model_item= getModelItem(model,k);
  if(model_item->number_of_samples==0)
    return; //sth wrong: this shouldn't happen; if there are no
  //          samples in model there shouldn't be any in sampleList2!
  int sampleNum=sampleNumber(model_item, sample->text(0));
  if (sampleNum==-1)
    return;

  /*****
   * delete currently selected sample utterance from model item
   * and switch 'modified' to 1
   *****/
  deleteModelItemSample(model_item, sampleNum);
  modified = 1;
  // and delete it from sampleList2:
  delete sample;
  // update note info on the list
  if(model_item->number_of_samples<MIN_NR_SAMPLES_PER_ITEM)
    {
      lRecordLeft2->setText(tr("You need to record at least %1"
			       " samples more for this note"
			       ).arg(MIN_NR_SAMPLES_PER_ITEM -
				     model_item->number_of_samples)
			    );
      note->setText(1,tr("%1 lacking"
			 ).arg(MIN_NR_SAMPLES_PER_ITEM-
			       model_item->number_of_samples));
    }
  else
    {
      lRecordLeft2->setText(tr("OK. You have at least %1 samples. :)"
			       "\nStill, you may record more if you "
			       "wish."
			       ).arg(MIN_NR_SAMPLES_PER_ITEM));
      note->setText(1,tr("(OK)"));
    }
}

void dModelEditor::PBdeleteAll2_clicked()
{
  QListViewItem* note= noteList->selectedItem ();
  if(note==0)
    return;
  int k =noteItemNumber(note);
  if(k==-1)
    return;

  QListViewItem* sample=samplesList2->lastItem ();
  if( sample==0)
    return;

  if( QMessageBox::warning ( this, "Delete ALL samples for this note?",
			     QString("Are you sure you want to delete all\n"
				     "samples for note %1 ?"
				     ).arg(note->text(0)), 
			     QMessageBox::Yes, 
			     QMessageBox::No|QMessageBox::Default)==
      QMessageBox::No)
    return;
  
  ModelItem* model_item= getModelItem(model,k);
  while(sample != 0)
    {
      if(model_item->number_of_samples==0)
	return; //sth wrong: this shouldn't happen; if there are no
      //          samples in model there shouldn't be any in sampleList2!
      int sampleNum=sampleNumber(model_item, sample->text(0));
      if (sampleNum==-1)
	return;

      /*****
       * delete currently selected sample utterance from model item
       * and switch 'modified' to 1
       *****/
      deleteModelItemSample(model_item, sampleNum);
      modified = 1;
      // and delete it from sampleList2:
      delete sample;

      sample=samplesList2->lastItem ();
    }

  // update note info on the list
  if(model_item->number_of_samples<MIN_NR_SAMPLES_PER_ITEM)
    {
      lRecordLeft2->setText(tr("You need to record at least %1"
			       " samples more for this note"
			       ).arg(MIN_NR_SAMPLES_PER_ITEM -
				     model_item->number_of_samples)
			    );
      note->setText(1,tr("%1 lacking"
			 ).arg(MIN_NR_SAMPLES_PER_ITEM-
			       model_item->number_of_samples));
    }
  else
    {
      lRecordLeft2->setText(tr("OK. You have at least %1 samples. :)"
			       "\nStill, you may record more if you "
			       "wish."
			       ).arg(MIN_NR_SAMPLES_PER_ITEM));
      note->setText(1,tr("(OK)"));
    }
}


int dModelEditor::sampleNumber(ModelItem* model_item, QString id)
{
  int result=-1;
  for(int i =0; i<model_item->number_of_samples; i++)
    if(id==QString(getModelItemSample(model_item,i)->id))
      result=i;
  return(result);
}

void dModelEditor::PBrecord2_clicked()
{
  QListViewItem* note= noteList->selectedItem ();
  if(note==0)
    return;
  int k =noteItemNumber(note);
  if(k==-1)
    return;

  cancelDialog=new KeyPressDialog(tr("Recording!"),
				  tr("Utterance is recorded automatically!\n"
				     "Please say what you want to say ...\n\n"
				     "Or click 'Cancel' to cancel recording"),
				  this);
  cancelDialog->show(); 
  qApp->processEvents () ;

  ModelItem* model_item= getModelItem(model,k);
    
  ModelItemSample *new_sample; /***** temporary pointer to new utterance */
    
  new_sample = recordSample(); /***** auto record a sample utterance */

  cancelDialog->hide();
  delete cancelDialog;

  if (new_sample != NULL) /***** if the recording was successful ... */
    {
      appendModelItemSample(model_item, new_sample); /***** add the sample 
							    to the model */
      QListViewItem* item = new QListViewItem( samplesList2, new_sample->id);
      samplesList2->setSelected(item,true); //this should play too!
    }
  else /***** recording has failed or was canceled */
    {
      infoPopup(tr("Cancel!    \n"
		   "Nothing has been recorded! Either  \n "
		   "you canceled or no data could be   \n"
		   "recorded !                          "));
    }

  if(model_item->number_of_samples<MIN_NR_SAMPLES_PER_ITEM)
    {
      lRecordLeft2->setText(tr("You need to record at least %1"
			       " samples more for this note"
			       ).arg(MIN_NR_SAMPLES_PER_ITEM -
				     model_item->number_of_samples)
			    );
      note->setText(1,tr("%1 lacking"
			 ).arg(MIN_NR_SAMPLES_PER_ITEM-
			       model_item->number_of_samples));
    }
  else
    {
      lRecordLeft2->setText(tr("OK. You have at least %1 samples. :)"
			       "\nStill, you may record more if you "
			       "wish."
			       ).arg(MIN_NR_SAMPLES_PER_ITEM));
      note->setText(1,tr("(OK)"));
    }
}




/*****************************************************************************
 * get an utterance from the sound card via auto recording
 *****************************************************************************/

unsigned char * dModelEditor::getUtterance(int *length)
{
  int i;
  signed short max = 0;

  /***** set prefetch buffer size to 5, and allocate memory */

  int           prefetch_N    = 5;
  int           prefetch_pos  = 0;
  unsigned char prefetch_buf[FRAG_SIZE*prefetch_N];
  void         *prefetch      = prefetch_buf;

  int count     = 0;

  /***** space for one audio block */

  unsigned char buffer_raw[FRAG_SIZE];

  /***** store whole wav data in a queue-like buffer */

  struct Buffer
  {
    unsigned char buffer[FRAG_SIZE];
    struct Buffer *next;
  };

  int nr_of_blocks = 0;

  struct Buffer *first = NULL;
  struct Buffer *last  = NULL;

  unsigned char *return_buffer;

  initKeyPressed();

  memset (prefetch_buf, 0x00, FRAG_SIZE*prefetch_N);

  /***** prefetch data in a circular buffer, and check it for speech content */

  while (count < CONSECUTIVE_SPEECH_BLOCKS_THRESHOLD)
    {
      if (keyPressed()) 
	{ 
	  return_buffer = NULL; 
	  goto getUtteranceReturn; 
	} 

      /* fprintf(stderr, "%d ", fgetc_unlocked(stdin)); */

      if (read(fd_audio, buffer_raw, FRAG_SIZE) != FRAG_SIZE)
	{
	  return_buffer = NULL;
	  goto getUtteranceReturn;
	}

      memcpy((prefetch+prefetch_pos*(FRAG_SIZE)), buffer_raw, FRAG_SIZE);
      prefetch_pos = (prefetch_pos+1)%prefetch_N;

      /***** check for speech */

      max = 0;
      for (i = 0; i < FRAG_SIZE/2 - 1; i += 2)
	{
	  signed short value = abs((signed short)(buffer_raw[i]|(buffer_raw[i+1]<<8)));

	  if (value > max)
	    max = value;
	}
      if (max > rec_level)
	count++;
      else
	count = 0;
    }

  /***** store prefetch buffer in queue and do recording until level falls below threshold */

  for (i = prefetch_pos; i < prefetch_pos+prefetch_N; i++)
    {
      struct Buffer *data = (struct Buffer *)malloc(sizeof(struct Buffer));

      memcpy(data->buffer, prefetch+(i%prefetch_N)*FRAG_SIZE, FRAG_SIZE);
      data->next = NULL;
      nr_of_blocks++;

      if (last != NULL)
	last->next = data;
      last = data;
      if (first == NULL)
	first = data;
    }

  count = 0;
  while (count < CONSECUTIVE_NONSPEECH_BLOCKS_THRESHOLD)
    {
      struct Buffer *data = (struct Buffer *)malloc(sizeof(struct Buffer));

      if (keyPressed())
	{
	  return_buffer = NULL; 
	  goto getUtteranceReturn; 
	} 

      if (read(fd_audio, data->buffer, FRAG_SIZE) != FRAG_SIZE)
	{
	  free(data);
	  return_buffer = NULL;
	  goto getUtteranceReturn;
	}
      data->next = NULL;
      nr_of_blocks++;

      if (last != NULL)
	last->next = data;
      last = data;

      /***** check for nonspeech */

      max = 0;
      for (i = 0; i < FRAG_SIZE/2 - 1; i += 2)
	{
	  signed short value = abs((signed short)(data->buffer[i]|(data->buffer[i+1]<<8)));

	  if (value > max)
	    max = value;
	}
      if (max < stop_level)
	count++;
      else
	count = 0;
    }

  /***** assemble data into one buffer, return it */

  return_buffer = (unsigned char *)malloc(nr_of_blocks*FRAG_SIZE);

  *length = nr_of_blocks*FRAG_SIZE;

  {
    struct Buffer *tmp_buffer = first;

    for (i = 0; i < nr_of_blocks; i++)
      {
	memcpy(return_buffer+(i*FRAG_SIZE), tmp_buffer->buffer, FRAG_SIZE);
	tmp_buffer = tmp_buffer->next;
      }
  }

 getUtteranceReturn:

  for (i = 0; i < nr_of_blocks; i++)
    {
      struct Buffer *tmp_buffer = first;
      first = first->next;
      free(tmp_buffer);
    }

  endKeyPressed(); 
  return return_buffer;
}

void dModelEditor::setKeypressed(bool pressed)
{
  if(pressed)
    keypressed=1;
  else
    keypressed=0;
}


void dModelEditor::initKeyPressed()
{
  keypressed=0;
  qApp->processEvents () ;
}

int dModelEditor::keyPressed()
{
  qApp->processEvents () ; // allow user to cancel!
  return keypressed;
}

void dModelEditor::endKeyPressed()
{
  qApp->processEvents ();
}

bool dModelEditor::isHead(QListViewItem * listItem)
{
  // heads are defined by:
  //  head[3]=new QListViewItem(commandList,"other","");

  for(int i =0; i< NUMHEADS;i++)
    if (listItem==head[i])
      return true;
  return false;
}

void dModelEditor::selectedCommandChanged(QListViewItem *
					  newCommandItem)
{
  if(!isHead(lastCommandItem))
    {// check if we should ever leave the last command:
      if((lastCommandItem!= NULL) && (lastCommandItem!=newCommandItem))
	{
	  int k =commandItemNumber(lastCommandItem);
	  if (k>=0)
	    if (getModelItem(model,k)->number_of_samples < 
		MIN_NR_SAMPLES_PER_ITEM) /***** not enoug samples to save! */
	      {
		if(getModelItem(model,k)->number_of_samples ==0)
		  lastCommandItem->setText(1,"-");
		else
		  {
		    warningPopup(tr("We need at least %1 sample utterances\n"
				    "for each speaker model item!!!\n\n"
				    "Please donate the minimum number of \n"
				    "samples for this item or delete all \n"
				    "samples do deactivate this item!"
				    ).arg(MIN_NR_SAMPLES_PER_ITEM));
		    // bail back to previous selection without triggering
		    // selectionChanged():
		    //		    lastCommandItem->setSelected ( true);
		    //		    commandList->repaintItem(lastCommandItem);
		    //		    newCommandItem->setSelected ( false);
		    //		    commandList->repaintItem(newCommandItem);
		    commandList->setSelected ( lastCommandItem, true);
		    return;
		  }
	      }
	}
    }
  // OK: sth changed and leaving last command was accepted

  samplesList3->clear();

  if(newCommandItem==0)// nothing selected
    {
      lRecordLeft3->setText("");
      lComments3->
	setText(tr("Choose a command from the left box, then record\n"
		   "your sound samples for this command (i.e. the\n"
		   "sound with which you would like to activate\n"
		   "this command). I need at least %1 samples for\n"
		   "each command.\n"
		   "See help for more suggestions!\n\n"
		   ).arg(MIN_NR_SAMPLES_PER_ITEM));
    }
  else if(isHead(newCommandItem)) // selected head
    {
      newCommandItem->setOpen(!(newCommandItem->isOpen())); // open<->close
      newCommandItem->setSelected(false);// so that second click 
      //                                    changed the state too
      commandList->repaintItem(newCommandItem);// but we don't want to trigger
      //                                          selChanged again right now!
      lRecordLeft3->setText("");
      lComments3->
	setText(tr("Choose a command from the left box, then record\n"
		   "your sound samples for this command (i.e. the\n"
		   "sound with which you would like to activate\n"
		   "this command). I need at least %1 samples for\n"
		   "each command.\n"
		   "See help for more suggestions!\n\n"
		   ).arg(MIN_NR_SAMPLES_PER_ITEM));
    }
  else // selected a command:
    {
      lComments3->
	setText(tr("<B>Command <U>%1</U>:\n</B><BR>"
		   "%2<BR><BR>"
		   "See help for more suggestions!<BR>"
		   ).arg(*commandStrings[newCommandItem->text(2).toInt()][1]
			 ).arg(*commandStrings[newCommandItem->
					       text(2).toInt()][2]));
      ModelItem* model_item= getModelItem(model,
					  commandItemNumber(newCommandItem));
      for(int i=0; i<model_item->number_of_samples;i++)
	new QListViewItem( samplesList3,
			   getModelItemSample(model_item,i)->id);
      if(model_item->number_of_samples<MIN_NR_SAMPLES_PER_ITEM)
	{
	  lRecordLeft3->setText(tr("You need to record at least %1"
				   " samples more for this command"
				   ).arg(MIN_NR_SAMPLES_PER_ITEM -
					 model_item->number_of_samples)
				);
	  newCommandItem->setText(1,
				  tr("%1 lacking"
				     ).arg(MIN_NR_SAMPLES_PER_ITEM-
					   model_item->number_of_samples)
				  );
	}
      else
	{
	  lRecordLeft3->setText(tr("OK. You have at least %1 samples. :)"
				   "\nStill, you may record more if you "
				   "wish."
				   ).arg(MIN_NR_SAMPLES_PER_ITEM)
				);
	  newCommandItem->setText(1,tr("(OK)"));
	}
    }
  lastCommandItem=newCommandItem;
}
  

void dModelEditor::PBplay3_clicked()
{
  QListViewItem* command= commandList->selectedItem ();
  if(command==0)
    return;
  int k =commandItemNumber(command);
  if(k==-1)
    return;
  QListViewItem* sample= samplesList3->selectedItem ();
  if(sample==0)
    return;
  ModelItem* model_item= getModelItem(model,k);
  if(model_item->number_of_samples==0)
    return; //sth wrong: this shouldn't happen; if there are no
  //          samples in model there shouldn't be any in sampleList3!
  int sampleNum=sampleNumber(model_item, sample->text(0));
  if (sampleNum==-1)
    return;
  /***** play wave if present */
  if (getModelItemSample(model_item, sampleNum)->has_wav)
    {
      /***** if play fails or was canceled ....*/
      if (playSample(getModelItemSample(model_item, sampleNum))!= AUDIO_OK)
	{
	  /***** display information */
	  warningPopup(tr("Either playback has been canceled \n"
			  "or wave couldn't be sent to your   \n"
			  "soundcard !                         "));
	}
    }
  else /***** no wave data present! */
    {
      warningPopup(tr("Utterance can't be played! No wave  \n"
		      "data available! Note: CVoiceControl \n"
		      "did NOT save the original wave data!"));
    }   
}
void dModelEditor::PBdelete3_clicked()
{
  QListViewItem* command= commandList->selectedItem ();
  if(command==0)
    return;
  int k =commandItemNumber(command);
  if(k==-1)
    return;
  QListViewItem* sample= samplesList3->selectedItem ();
  if(sample==0)
    return;
  ModelItem* model_item= getModelItem(model,k);
  if(model_item->number_of_samples==0)
    return; //sth wrong: this shouldn't happen; if there are no
  //          samples in model there shouldn't be any in sampleList3!
  int sampleNum=sampleNumber(model_item, sample->text(0));
  if (sampleNum==-1)
    return;

  /*****
   * delete currently selected sample utterance from model item
   * and switch 'modified' to 1
   *****/
  deleteModelItemSample(model_item, sampleNum);
  modified = 1;
  // and delete it from sampleList3:
  delete sample;
  // update command info on the list
  if(model_item->number_of_samples<MIN_NR_SAMPLES_PER_ITEM)
    {
      lRecordLeft3->setText(tr("You need to record at least %1"
			       " samples more for this command"
			       ).arg(MIN_NR_SAMPLES_PER_ITEM -
				     model_item->number_of_samples)
			    );
      command->setText(1,tr("%1 lacking"
			    ).arg(MIN_NR_SAMPLES_PER_ITEM-
				  model_item->number_of_samples));
    }
  else
    {
      lRecordLeft3->setText(tr("OK. You have at least %1 samples. :)"
			       "\nStill, you may record more if you "
			       "wish."
			       ).arg(MIN_NR_SAMPLES_PER_ITEM));
      command->setText(1,tr("(OK)"));
    }
}

void dModelEditor::PBdeleteAll3_clicked()
{
  QListViewItem* command= commandList->selectedItem ();
  if(command==0)
    return;
  int k =commandItemNumber(command);
  if(k==-1)
    return;

  QListViewItem* sample= samplesList3->lastItem ();
  if(sample==0)
    return;
  
  if( QMessageBox::warning ( this, "Delete ALL samples for this command?",
			     QString("Are you sure you want to delete all\n"
				     "samples for command %1 ?"
				     ).arg(*(commandStrings[command->
							    text(2).toInt()][1]
					     )),
			     QMessageBox::Yes, 
			     QMessageBox::No|QMessageBox::Default)==
      QMessageBox::No)
    return;
  

  ModelItem* model_item= getModelItem(model,k);
  while(sample != 0)
    {
      if(model_item->number_of_samples==0)
	return; //sth wrong: this shouldn't happen; if there are no
      //          samples in model there shouldn't be any in sampleList3!
      int sampleNum=sampleNumber(model_item, sample->text(0));
      if (sampleNum==-1)
	return;

      /*****
       * delete currently selected sample utterance from model item
       * and switch 'modified' to 1
       *****/
      deleteModelItemSample(model_item, sampleNum);
      modified = 1;
      // and delete it from sampleList3:
      delete sample;
      sample=samplesList3->lastItem ();
    }


  // update command info on the list
  if(model_item->number_of_samples<MIN_NR_SAMPLES_PER_ITEM)
    {
      lRecordLeft3->setText(tr("You need to record at least %1"
			       " samples more for this command"
			       ).arg(MIN_NR_SAMPLES_PER_ITEM -
				     model_item->number_of_samples)
			    );
      command->setText(1,tr("%1 lacking"
			    ).arg(MIN_NR_SAMPLES_PER_ITEM-
				  model_item->number_of_samples));
    }
  else
    {
      lRecordLeft3->setText(tr("OK. You have at least %1 samples. :)"
			       "\nStill, you may record more if you "
			       "wish."
			       ).arg(MIN_NR_SAMPLES_PER_ITEM));
      command->setText(1,tr("(OK)"));
    }
}


void dModelEditor::PBrecord3_clicked()
{
  QListViewItem* command= commandList->selectedItem ();
  if(command==0)
    return;
  int k =commandItemNumber(command);
  if(k==-1)
    return;

  cancelDialog=new KeyPressDialog(tr("Recording!"),
				  tr("Utterance is recorded automatically!\n"
				     "Please say what you want to say ...\n\n"
				     "Or click 'Cancel' to cancel recording"),
				  this);
  cancelDialog->show(); 
  qApp->processEvents () ;

  ModelItem* model_item= getModelItem(model,k);
    
  ModelItemSample *new_sample; /***** temporary pointer to new utterance */
    
  new_sample = recordSample(); /***** auto record a sample utterance */

  cancelDialog->hide();
  delete cancelDialog;

  if (new_sample != NULL) /***** if the recording was successful ... */
    {
      appendModelItemSample(model_item, new_sample); /***** add the sample 
							    to the model */
      QListViewItem* item = new QListViewItem( samplesList3, new_sample->id);
      samplesList3->setSelected(item,true); //this should play too!
    }
  else /***** recording has failed or was canceled */
    {
      infoPopup(tr("Cancel!    \n"
		   "Nothing has been recorded! Either  \n "
		   "you canceled or no data could be   \n"
		   "recorded !                          "));
    }

  if(model_item->number_of_samples<MIN_NR_SAMPLES_PER_ITEM)
    {
      lRecordLeft3->setText(tr("You need to record at least %1"
			       " samples more for this command"
			       ).arg(MIN_NR_SAMPLES_PER_ITEM -
				     model_item->number_of_samples)
			    );
      command->setText(1,tr("%1 lacking"
			    ).arg(MIN_NR_SAMPLES_PER_ITEM-
				  model_item->number_of_samples));
    }
  else
    {
      lRecordLeft3->setText(tr("OK. You have at least %1 samples. :)"
			       "\nStill, you may record more if you "
			       "wish."
			       ).arg(MIN_NR_SAMPLES_PER_ITEM));
      command->setText(1,tr("(OK)"));
    }
}

