/****************************************************************************

    DRC: Digital Room Correction
    Copyright (C) 2002 Denis Sbragion

		This file is part of DRC, Digital Room Correction

		DRC  is  distributed  with  NO WARRANTY OF ANY KIND. No author or
		distributor accepts any responsibility for  the  consequences  of
		using  it,  or  for  whether  it serves any particular purpose or
		works at all, unless he or she says so in writing.  Refer to  the
		Aladdin Free Public License (the "License") for full details.

		Every copy of DRC must include a copy of the License, normally in
		a plain ASCII text file named PUBLIC. The License grants you  the
		right  to  copy, modify and redistribute DRC, but only under cer
		tain conditions described in the License.   Among  other  things,
		the License requires that the copyright notice and this notice be
		preserved on all copies.

		You can contact the author on Internet at the following address:

				d.sbragion@infotecna.it

		This program uses the parsecfg library from Yuuki  NINOMIYA.  De
		tails  on  this  library  can be found in the parsecfg.c and par
		secfg.h files.  Many thanks to Yuuki NINOMIYA for this useful li
		brary.

****************************************************************************/

/* Spline routines */

/* Inclusioni */
#include "spline.h"
#include <stdlib.h>
#include <math.h>

/* Prepara la base per la spline cubica interpolante, ponendola in SPD2YN */
Boolean SplinePrepare(const DRCFloat * XN,const DRCFloat * YN,
	const int N,const DRCFloat SPD1Y1,const DRCFloat SPD1YN,
	const SplineType SPType,DRCFloat * SPD2YN)
	{
		int IM1;
		int I;
		int IP1;
		DRCFloat P;
		DRCFloat Q;
		DRCFloat S;
		DRCFloat U;
		DRCFloat * UN;
		
		/* Alloca l'array intermedio calcolo spline */
		if ((UN = (DRCFloat *) malloc((N - 1) * sizeof(DRCFloat))) == NULL)
			return False;

		/* Verifica il tipo di condizioni al contorno */
		switch (SPType)
			{
				case SplineBDef:
					SPD2YN[0] = (DRCFloat) -0.5;
					UN[0] = (((DRCFloat) 3.0) / (XN[1] - XN[0]))
							* ((YN[1] - YN[0]) / (XN[1] - XN[0]) - SPD1Y1);
				break;

				case SplineBNat:
					SPD2YN[0] = (DRCFloat) 0.0;
					UN[0] = (DRCFloat) 0.0;
				break;
			}

		/* Decomposizione sistema tridiagonale */
		for (IM1 = 0,I = 1,IP1 = 2;I < N - 1;IM1++,I++,IP1++) 
			{
				S = (XN[I] - XN[IM1]) / (XN[IP1] - XN[IM1]);
				P = S * SPD2YN[IM1] + (DRCFloat) 2.0;
				SPD2YN[I] = (S - (DRCFloat) 1.0) / P;
				UN[I] = (YN[IP1] - YN[I]) / (XN[IP1] - XN[I]) - 
					(YN[I] - YN[IM1]) / (XN[I] - XN[IM1]);
				UN[I] = (((DRCFloat) 6.0) * UN[I] / (XN[IP1] - XN[IM1]) - 
					S * UN[IM1]) / P;
			}

		/* Verifica il tipo di condizioni al contorno */
		switch (SPType)
			{
				case SplineBDef:
					Q = (DRCFloat) 0.5;
					U = (((DRCFloat) 3.0) / (XN[N - 1] - XN[N - 2])) *
						(SPD1YN - (YN[N - 1] - YN[N - 2]) / (XN[N - 1] - XN[N - 2]));

				break;

				case SplineBNat:
					Q = (DRCFloat) 0.0;
					U = (DRCFloat) 0.0;
				break;
			}

		/* Sostituzione all'indietro per recupero derivate */
		SPD2YN[N - 1] = (U - Q * UN[N - 2]) / 
			(Q * SPD2YN[N - 2] + ((DRCFloat) 1.0));			
		for (I = N - 2,IP1 = N - 1;I >= 0;I--,IP1--) 
			SPD2YN[I] = SPD2YN[I] * SPD2YN[IP1] + UN[I];
	
		/* Dealloca il vettore temporaneo */
		free(UN);

		/* Operazione completata */
		return True;
	}

/* Valore della spline cubica interpolante nel punto X */
DRCFloat SplineValue(const DRCFloat * XN,const DRCFloat * YN,
	const int N, const DRCFloat * SPD2YN, const DRCFloat X)
	{
		int II;
		int IS;
		int IP;
		DRCFloat DX;
		DRCFloat W;
		DRCFloat Q;
				
		/* Recupera l'intervallo interessato */
		II = 0; 
		IS = N - 1;
		while (IS - II > 1) 
			{
				IP = (IS + II) >> 1;
				if (XN[IP] > X) 
					IS = IP;
				else 
					II = IP;
			}

		/* Calcola il valore nel punto */
		DX = XN[IS] - XN[II];
		W = (XN[IS] - X) / DX;
		Q = (X - XN[II]) / DX; 
		return W * YN[II] + Q * YN[IS] + 
			((W * W * W - W) * SPD2YN[II] + (Q * Q * Q - Q) * SPD2YN[IS]) * 
			(DX * DX) / (DRCFloat) 6.0;
	}

/* Valore della B spline cubica uniforme nel punto X */
DRCFloat B3SplineValue(const DRCFloat * XN,const DRCFloat * YN,
	const int N, const DRCFloat X)
	{
		DRCFloat BV;
		DRCFloat RX;
		DRCFloat Y;
		int IP;		
		int IL;
		int IR;
		
		/* Recupera l'intervallo interessato */
		IL = 0; 
		IR = N - 1;
		while (IR - IL > 1) 
			{
				IP = (IR + IL) >> 1;
				if (XN[IP] > X) 
					IR = IP;
				else 
					IL = IP;
			}
	
		/* Rimappa X secondo l'intervallo corrente */
		RX = (X - XN[IL]) / (XN[IR] - XN[IL]);

		/* Inizializza il valore */
		Y = (DRCFloat) 0.0;
		
		/* Fattore di blending per IL - 1 */
		/* (((-t+3)*t-3)*t+1)/6 */
		BV = (((-RX + (DRCFloat) 3.0) * RX - (DRCFloat) 3.0) * RX + (DRCFloat) 1.0) / 
			(DRCFloat) 6.0;
		if (IL > 0)
			Y += YN[IL - 1] * BV;
		else
			Y += (((DRCFloat) 2.0) * YN[0] - YN[1]) * BV;
			
		/* Fattore di blending per IL */
		/* (((3*t-6)*t)*t+4)/6 */
		BV = (((((DRCFloat) 3.0) * RX - (DRCFloat) 6.0) * RX) * RX + (DRCFloat) 4.0) /
			(DRCFloat) 6.0; 
		Y += YN[IL] * BV;
	
		/* Fattore di blending per IR */	
		/* (((-3*t+3)*t+3)*t+1)/6 */
		BV = (((((DRCFloat) -3.0) * RX + (DRCFloat) 3.0) * RX + (DRCFloat) 3.0) * RX + (DRCFloat) 1.0) / 
			(DRCFloat) 6.0;
		Y += YN[IR] * BV;
		
		/* Fattore di blending per IR + 1 */
		/* (t*t*t)/6 */
		BV = (RX * RX * RX) / (DRCFloat) 6.0;
		if (IR + 1 < N)
			Y += YN[IR + 1] * BV;
		else
			Y += (((DRCFloat) 2.0) * YN[N - 1] - YN[N - 2]) * BV;
			
		/* Ritorna il valore calcolato */
		return Y;
	}

/* Valore della B spline cubica di tipo Catmull Rom nel punto X */
DRCFloat B3CRSplineValue(const DRCFloat * XN,const DRCFloat * YN,
	const int N, const DRCFloat X)
	{
		DRCFloat BV;
		DRCFloat RX;
		DRCFloat Y;
		int IP;		
		int IL;
		int IR;
		
		/* Recupera l'intervallo interessato */
		IL = 0; 
		IR = N - 1;
		while (IR - IL > 1) 
			{
				IP = (IR + IL) >> 1;
				if (XN[IP] > X) 
					IR = IP;
				else 
					IL = IP;
			}
	
		/* Rimappa X secondo l'intervallo corrente */
		RX = (X - XN[IL]) / (XN[IR] - XN[IL]);

		/* Inizializza il valore */
		Y = (DRCFloat) 0.0;
		
		/* Fattore di blending per IL - 1 */
		/* ((-t+2)*t-1)*t/2 */
		BV = (((-RX + (DRCFloat) 2.0) * RX - (DRCFloat) 1.0) * RX) / 
			(DRCFloat) 2.0;
		if (IL > 0)
			Y += YN[IL - 1] * BV;
		else
			Y += (((DRCFloat) 2.0) * YN[0] - YN[1]) * BV;
			
		/* Fattore di blending per IL */
		/* (((3*t-5)*t)*t+2)/2 */
		BV = (((((DRCFloat) 3.0) * RX - (DRCFloat) 5.0) * RX) * RX + (DRCFloat) 2.0) /
			(DRCFloat) 2.0;
		Y += YN[IL] * BV;
	
		/* Fattore di blending per IR */	
		/* ((-3*t+4)*t+1)*t/2 */
		BV = (((((DRCFloat) -3.0) * RX + (DRCFloat) 4.0) * RX + (DRCFloat) 1.0) * RX) / 
			(DRCFloat) 2.0;
		Y += YN[IR] * BV;
		
		/* Fattore di blending per IR + 1 */
		/* ((t-1)*t*t)/2 */
		BV = ((RX - (DRCFloat) 1.0) * RX * RX) / (DRCFloat) 2.0;
		if (IR + 1 < N)
			Y += YN[IR + 1] * BV;
		else
			Y += (((DRCFloat) 2.0) * YN[N - 1] - YN[N - 2]) * BV;
			
		/* Ritorna il valore calcolato */
		return Y;
	}

/* Valore della spline lineare nel punto X */
DRCFloat L1SplineValue(const DRCFloat * XN,const DRCFloat * YN,
	const int N, const DRCFloat X)
	{
		int IP;		
		int IL;
		int IR;
		
		/* Recupera l'intervallo interessato */
		IL = 0; 
		IR = N - 1;
		while (IR - IL > 1) 
			{
				IP = (IR + IL) >> 1;
				if (XN[IP] > X) 
					IR = IP;
				else 
					IL = IP;
			}
				
		/* Ritorna il valore calcolato */
		return YN[IL] + ((X - XN[IL]) / (XN[IR] - XN[IL])) * (YN[IR] - YN[IL]);
	}
