/****************************************************************************

    DRC: Digital Room Correction
    Copyright (C) 2002, 2003 Denis Sbragion

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

		You can contact the author on Internet at the following address:

				d.sbragion@infotecna.it

		This program uses the parsecfg library from Yuuki  NINOMIYA.  De
		tails  on  this  library  can be found in the parsecfg.c and par
		secfg.h files.  Many thanks to Yuuki NINOMIYA for this useful li
		brary.

****************************************************************************/

/****************************************************************************
  Progetto    : DSP Library.
  File        : Fir.cpp
  Autore      : Sbragion Denis
  Descrizione : Funzioni per il calcolo di filtri fir.
  Revisioni   :
  16/10/93    : Prima stesura.
****************************************************************************/

#include "fir.h"
#include "fft.h"
#include "spline.h"
#include "baselib.h"
#include <stddef.h>

void LowPassFir(DLReal * Filter,unsigned int Order,DLReal Freq)
  {
    unsigned int I,HalfOrder = Order/2;
    DLReal C = (DLReal) (M_PI*Freq);

    if (Order > 1)
      if (Order%2 == 0)
        for(I = 1; I <= HalfOrder;I++)
          {
            Filter[HalfOrder-I] = (DLReal) (sin((I-0.5)*C)/((I-0.5)*M_PI));
            Filter[HalfOrder+I-1] = Filter[HalfOrder-I];
          }
      else
        {
          for(I = 1; I <= HalfOrder;I++)
            {
              Filter[HalfOrder-I] = (DLReal) (sin(I*C)/(I*M_PI));
              Filter[HalfOrder+I] = Filter[HalfOrder-I];
            }
          Filter[HalfOrder] = Freq;
        }
  }

void HighPassFir(DLReal * Filter,unsigned int Order,DLReal Freq)
  {
    unsigned int I,HalfOrder;
    DLReal C = (DLReal) (M_PI*Freq);

    if (Order > 2)
      {
        if (Order%2 == 0)
          {
            Order--;
            Filter[Order] = 0;
          }

        HalfOrder = Order/2;

        for(I = 1; I <= HalfOrder;I++)
          {
            Filter[HalfOrder-I] = (DLReal) (-sin(I*C)/(I*M_PI));
            Filter[HalfOrder+I] = Filter[HalfOrder-I];
          }
        Filter[HalfOrder] = 1-Freq;
      }
  }

void BandPassFir(DLReal * Filter,unsigned int Order,DLReal Low,DLReal High)
  {
    unsigned int I,HalfOrder = Order/2;
    DLReal D = (DLReal) (M_PI*(High-Low)/2), S = (DLReal) (M_PI*(High+Low)/2);

    if (Order > 1)
      if (Order%2 == 0)
				{
					for(I = 1;I <= HalfOrder;I++)
						{
							Filter[HalfOrder-I] = (DLReal) (M_2PI*sin((I-0.5)*D)*
								cos((I-0.5)*S)/(I-0.5));
							Filter[HalfOrder+I-1] = Filter[HalfOrder-I];
						}
				}
      else
        {
          for(I = 1;I <= HalfOrder;I++)
            {
              Filter[HalfOrder-I] = (DLReal) (M_2PI*sin(I*D)*
                cos(I*S)/I);
              Filter[HalfOrder+I] = Filter[HalfOrder-I];
            }
          Filter[HalfOrder] = M_PI*M_PI*(High-Low);
        }
  }

void BandStopFir(DLReal * Filter,unsigned int Order,DLReal Low,DLReal High)
  {
    unsigned int I,HalfOrder;
    DLReal D = (DLReal) (M_PI*(High-Low)/2),S = (DLReal) (M_PI*(High+Low)/2);

    if (Order > 2)
      {
        if (Order%2 == 0)
          {
            Order--;
            Filter[Order] = 0;
          }

        HalfOrder = Order/2;

        for(I = 1; I <= HalfOrder;I++)
          {
            Filter[HalfOrder-I] = (DLReal) (-M_2PI*sin(I*D)*
              cos(I*S)/I);
            Filter[HalfOrder+I] = Filter[HalfOrder-I];
          }
        Filter[HalfOrder] = 1-(High-Low);
      }
  }

void HilbertFir(DLReal * Filter,unsigned int Order)
  {
    unsigned int I,HalfOrder;

    if (Order > 2)
      {
        if (Order%2 == 0)
          {
            Order--;
            Filter[Order] = 0;
          }

        HalfOrder = Order/2;

        for(I = 1;I <= HalfOrder;I++)
          {
            Filter[HalfOrder+I] = (DLReal) (M_2PI*(I%2)/I);
            Filter[HalfOrder-I] = -Filter[HalfOrder+I];
          }

        Filter[HalfOrder] = 0;
      }
  }

void LinInterpolate(DLComplex * A,unsigned int Size,DLReal * X,
  DLReal * YM, DLReal * YP)
  {
    unsigned int I,P;
    DLReal Mag;
		DLReal DeltaMag;
		DLReal Arg;
		DLReal DeltaArg;
    DLReal StopI;
		DLReal LastI;
		DLReal Pc;
		DLReal DeltaF;

    P = 0;
    StopI = 0;

    for(I = 0;I < Size;)
      {
        if (I >= StopI)
          {
						Mag = YM[P];						
						DeltaMag = YM[P+1]-Mag;
            Arg = YP[P];
            DeltaArg = YP[P+1]-Arg;
            LastI = StopI;
            StopI = X[P+1]*Size;
            DeltaF = StopI-LastI;
            P++;
          }
        else
          {
            Pc = (I-LastI)/DeltaF;
            A[I] = std::polar(Mag+DeltaMag*Pc,Arg+DeltaArg*Pc);
            I++;
          }
      }
  }

void LogInterpolate(DLComplex * A,unsigned int Size,DLReal * X,
  DLReal * YM, DLReal * YP)
  {
    unsigned int I,P;
    DLReal Mag;
		DLReal DeltaMag;
		DLReal Arg;
		DLReal DeltaArg;
    DLReal StopI;
		DLReal LastI;
		DLReal Pc;
		DLReal LI;
		DLReal LStopI;
		DLReal LLastI;
		DLReal LDeltaF;
		DLReal LSize;

    P = 0;
    StopI = 0;
    LSize = (DLReal) (Size / log10(Size));

    for(I = 0;I < Size;)
      {      	
        if (I >= StopI)
          {
						Mag = (DLReal) log10(YM[P]);
						DeltaMag = ((DLReal) log10(YM[P+1]))-Mag;
            Arg = YP[P];
            DeltaArg = YP[P+1]-Arg;
            LastI = StopI;            
            StopI = X[P+1]*Size;
            
            LLastI = (DLReal) (LSize * log10(1.0 + LastI));
            LStopI = (DLReal) (LSize * log10(1.0 + StopI));
            LDeltaF = LStopI - LLastI;
            
            P++;
          }
        else
          {           	
          	LI = (DLReal) (LSize * log10(1.0 + I));
            Pc = (LI-LLastI)/LDeltaF;            
						A[I] = std::polar((DLReal) pow(10.0,Mag+DeltaMag*Pc),Arg+DeltaArg*Pc);
            I++;
          }
      }
  }

/* Interpolazione su scala lineare tramite B spline cubica 
con riparametrizzazione arc length tramite splice cubica 
naturale interpolante */
Boolean SplineLinInterpolate(DLComplex * A,unsigned int Size,DLReal * X,
  DLReal * YM, DLReal * YP, unsigned int Np)
	{
		/* Array gestione spline rimappatura */
		DLReal * SPD2YNR;		
		DLReal * BSX;
		DLReal * CSX;

		/* Posizioni e riferimento calcolo valore spline */
		DLReal CX;
		DLReal CXR;
		DLReal CXM;
		DLReal CXD;
		DLReal BX;
		DLReal LBX;
		DLReal BXM;		

		/* Posizione riferimento approssimazione */
		DLReal FCX;
		
		/* Indici generici */
		unsigned int I;
		unsigned int J;
		unsigned int P;

		/* Dimensione effettiva spline cubica */
		unsigned int BSSize;

		/* Alloca gli array gestione spline */
		if ((SPD2YNR = new DLReal[Size]) == NULL)
			return(False);
		if ((BSX = new DLReal[Size]) == NULL)
			{
				delete SPD2YNR;
				return(False);
			}
		if ((CSX = new DLReal[Size]) == NULL)
			{
				delete SPD2YNR;
				delete BSX;
				return(False);
			}

		/* Calcola la configurazione spline */
		P = 0;
		CXR = X[P];
		CXM = X[P + 1];
		CXD = CXM - CXR;		
		BXM = B3SplineValue(X,X,Np,CXM);
		BSSize = 0;
		LBX = (DLReal) -1.0;
		for (J = 0;J < Size;J++)
			{
				CX = CXR + (CXD * J) / (Size - 1);
				BX = B3SplineValue(X,X,Np,CX);
				if ((BX - LBX) > DRCEpsFloat)
					{
						CSX[BSSize] = CX;
						BSX[BSSize] = BX;
						LBX = BX;
						BSSize++;
					}
			}
		CSX[BSSize - 1] = CXM;
		BSX[BSSize - 1] = BXM;
		
		/* Prepara le spline */
		if (SplinePrepare(BSX,CSX,BSSize,(DLReal) 0.0,(DLReal) 0.0,
			SplineBNat,SPD2YNR) == False)
			{
				delete SPD2YNR;
				delete BSX;
				delete CSX;
				return(False);
			}

		/* Ritorna i valori interpolati */		
		for (I = 0;I < Size;I++)
			{
				/* Calcola la posizione corrente */
				FCX = ((DLReal) I) / (Size - 1);				

				/* Verifica di non aver superato la posizione massima */
				if ((FCX - BXM) > DRCEpsFloat)
					{
						/* Ricalcola la configurazione spline */
						P++;						
						CXR = X[P];
						CXM = X[P + 1];
						CXD = CXM - CXR;		
						BXM = B3SplineValue(X,X,Np,CXM);
						BSSize = 0;
						LBX = (DLReal) -1.0;
						for (J = 0;J < Size;J++)
							{
								CX = CXR + (CXD * J) / (Size - 1);								
								BX = B3SplineValue(X,X,Np,CX);
								if ((BX - LBX) > DRCEpsFloat)
									{
										CSX[BSSize] = CX;
										BSX[BSSize] = BX;
										LBX = BX;
										BSSize++;
									}
							}
						CSX[BSSize - 1] = CXM;
						BSX[BSSize - 1] = BXM;
							
						/* Prepara le spline */
						if (SplinePrepare(BSX,CSX,BSSize,(DLReal) 0.0,(DLReal) 0.0,
							SplineBNat,SPD2YNR) == False)
							{
								delete SPD2YNR;
								delete BSX;
								delete CSX;
								return(False);
							}
					}

				/* Rimappa la posizione secondo la B Spline */
				BX = SplineValue(BSX,CSX,BSSize,SPD2YNR,FCX);

				/* Calcola il valore corrente*/
				A[I] = std::polar(B3SplineValue(X,YM,Np,BX),
					B3SplineValue(X,YP,Np,BX));
			}

		/* Dealloca gli array gestione spline */
		delete SPD2YNR;
		delete BSX;
		delete CSX;

		/* Operazione completata */
		return True;
	}

/* Interpolazione su scala logaritmica tramite B spline cubica 
con riparametrizzazione arc length tramite splice cubica 
naturale interpolante */
Boolean SplineLogInterpolate(DLComplex * A,unsigned int Size,DLReal * X,
  DLReal * YM, DLReal * YP, unsigned int Np)
	{
		/* Array gestione spline rimappatura */
		DLReal * SPD2YNR;		
		DLReal * BSX;
		DLReal * CSX;

		/* Posizioni e riferimento calcolo valore spline */
		DLReal CX;				
		DLReal CXR;
		DLReal CXM;
		DLReal CXD;
		DLReal BX;
		DLReal LBX;
		DLReal BXR;
		DLReal BXM;
		DLReal BXD;		

		/* Array conversione logaritmica */
		DLReal * LX;
		DLReal * LM;

		/* Posizione riferimento approssimazione */
		DLReal FCX;

		/* Coefficiente correzione scala logaritmica */
		DLReal LCS;
		
		/* Indici generici */
		unsigned int I;
		unsigned int J;
		unsigned int P;

		/* Dimensione effettiva spline cubica */
		unsigned int BSSize;

		/* Alloca gli array gestione spline */
		if ((SPD2YNR = new DLReal[Size]) == NULL)
			return(False);
		if ((BSX = new DLReal[Size]) == NULL)
			{
				delete SPD2YNR;
				return(False);
			}
		if ((CSX = new DLReal[Size]) == NULL)
			{
				delete SPD2YNR;
				delete BSX;
				return(False);
			}
		if ((LX = new DLReal[Np]) == NULL)
			{
				delete SPD2YNR;
				delete BSX;
				delete CSX;
				return(False);
			}
		if ((LM = new DLReal[Np]) == NULL)
			{
				delete SPD2YNR;
				delete BSX;
				delete CSX;
				delete LX;
				return(False);
			}

		/* Calcola la scala logaritmica */
		LCS = (DLReal) log10(1.0 + (Size - 1) * X[Np - 1]);
		for (I = 0;I < Np;I++)
			{
				LX[I] = (DLReal) (log10(1.0 + (Size - 1) * X[I]) / LCS);
				LM[I] = (DLReal) log10(YM[I]);
			}

		/* Calcola la configurazione spline */
		P = 0;
		CXR = LX[P];
		CXM = LX[P + 1];
		CXD = (double) CXM - CXR;		
		BXR = B3SplineValue(LX,LX,Np,CXR);
		BXM = B3SplineValue(LX,LX,Np,CXM);
		BXD = (double) BXM - BXR;		
		BSSize = 0;
		LBX = (DLReal) -1.0;
		for (J = 0;J < Size;J++)
			{
				/* CX = CXR + (CXD * J) / (Size - 1); */
				CX = ((double) J) / (Size - 1);
				BX = (B3SplineValue(LX,LX,Np,(double) CXR + CXD * CX) - (double) BXR) / BXD;
				if ((BX - LBX) >= DRCEpsFloat)
					{
						CSX[BSSize] = CX;
						BSX[BSSize] = BX;
						LBX = BX;
						BSSize++;
					}
			}
		CSX[BSSize - 1] = (DLReal) 1.0;
		BSX[BSSize - 1] = (DLReal) 1.0;
		
		/* Prepara le spline */
		if (SplinePrepare(BSX,CSX,BSSize,(DLReal) 0.0,(DLReal) 0.0,
			SplineBNat,SPD2YNR) == False)
			{
				delete SPD2YNR;
				delete BSX;
				delete CSX;
				delete LX;
				delete LM;
				return(False);
			}

		/* Ritorna i valori interpolati */		
		LCS = (DLReal) log10(Size);
		for (I = 0;I < Size;I++)
			{
				/* Calcola la posizione corrente */
				FCX = (DLReal) (log10(1.0 + I) / LCS);
				
				/* Verifica di non aver superato la posizione massima */
				if ((FCX - BXM) >= DRCEpsFloat)
					{
						/* Ricalcola la configurazione spline */
						P++;
						CXR = LX[P];
						CXM = LX[P + 1];
						CXD = (double) CXM - CXR;		
						BXR = B3SplineValue(LX,LX,Np,CXR);
						BXM = B3SplineValue(LX,LX,Np,CXM);
						BXD = (double) BXM - BXR;		
						BSSize = 0;
						LBX = (DLReal) -1.0;
						for (J = 0;J < Size;J++)
							{
								/* CX = CXR + (CXD * J) / (Size - 1); */
								CX = ((double) J) / (Size - 1);
								BX = (B3SplineValue(LX,LX,Np,(double) CXR + CXD * CX) - (double) BXR) / BXD;
								if ((BX - LBX) >= DRCEpsFloat)
									{
										CSX[BSSize] = CX;
										BSX[BSSize] = BX;
										LBX = BX;
										BSSize++;
									}
							}
						CSX[BSSize - 1] = (DLReal) 1.0;
						BSX[BSSize - 1] = (DLReal) 1.0;
							
						/* Prepara le spline */
						if (SplinePrepare(BSX,CSX,BSSize,(DLReal) 0.0,(DLReal) 0.0,
							SplineBNat,SPD2YNR) == False)
							{
								delete SPD2YNR;
								delete BSX;
								delete CSX;
								delete LX;
								delete LM;
								return(False);
							}
					}

				/* Rimappa la posizione secondo la B Spline */				
				BX = (double) CXR + CXD * SplineValue(BSX,CSX,BSSize,SPD2YNR,(FCX - BXR) / (double) BXD);

				/* Calcola il valore corrente*/
				/* A[I] = std::polar((DLReal) pow(10.0,B3SplineValue(LX,LM,Np,BX)),
					B3SplineValue(LX,YP,Np,BX));				*/
				A[I] = DLComplex((DLReal) BX,
					B3SplineValue(LX,LX,Np,BX) - FCX);				
			}

		/* Dealloca gli array gestione spline */
		delete SPD2YNR;
		delete BSX;
		delete CSX;
		delete LX;
		delete LM;

		/* Operazione completata */
		return True;
	}

Boolean GenericFir(DLReal * Filter,unsigned int Order,DLReal * F,
  DLReal * M, DLReal * P,unsigned int Np,unsigned int Is,
	InterpolationType It)
  {
    DLComplex * Ht;		
    unsigned int I,HalfFilter,HtSize,HtSize2;

    if ((Order < 2) || (Np < 2))
      return(False);

    if ((F[0] != 0) || (F[Np-1] != 1))
      return(False);

		if (It == Linear || It == Logarithmic)			
			{
				for(I = 0;I < Np-1;I++)
					if (F[I] > F[I+1])
						return(False);
			}
		else
			{
				for(I = 0;I < Np-1;I++)
					if (F[I] >= F[I+1])
						return(False);
			}

    if (Is == 0)
      for (HtSize = 2;HtSize < Order;HtSize <<= 1);
    else
      HtSize = Is;

    if (HtSize < Order)
      return(False);

    HtSize2 = 2*HtSize;

    if ((Ht = new DLComplex[HtSize2]) == NULL)
      return(False);		

		switch (It)
			{
				case Linear:
					LinInterpolate(Ht,HtSize,F,M,P);
				break;
				case Logarithmic:
					LogInterpolate(Ht,HtSize,F,M,P);
				break;
				case SplineLinear:
					if (SplineLinInterpolate(Ht,HtSize,F,M,P,Np) == False)
						{
							delete Ht;
							return(False);
						}
				break;
				case SplineLogarithmic:
					if (SplineLogInterpolate(Ht,HtSize,F,M,P,Np) == False)
						{
							delete Ht;
							return(False);
						}
				break;
			}

		DLReal * HM;
		HM = new DLReal[HtSize];
		for(I = 1;I < HtSize;I++)
			HM[I] = std::real(Ht[I]);
		WriteSignal("htm.pcm",HM,HtSize,PcmFloat32Bit);
		for(I = 1;I < HtSize;I++)
			HM[I] = std::imag(Ht[I]);
		WriteSignal("htx.pcm",HM,HtSize,PcmFloat32Bit);

    if (Order%2 == 0)
      {
        for(I = 1;I < HtSize;I++)
          Ht[HtSize2-I] = -std::conj(Ht[I]);
        Ht[HtSize] = std::polar(M[Np-1],P[Np-1]);
        for (I = 0;I < HtSize2;I++)
          Ht[I] *= UnitRoot(I,2*HtSize2);
      }
    else
      {
        for(I = 1;I < HtSize;I++)
          Ht[2*HtSize-I] = std::conj(Ht[I]);
        Ht[HtSize] = std::polar(M[Np-1],P[Np-1]);
      }

    if (IFft(Ht,HtSize2) == False)
      {
        delete(Ht);
        return(False);
      }

    HalfFilter = Order/2;

    for(I = 0;I < HalfFilter;I++)
      Filter[I] = std::real(Ht[HtSize2-HalfFilter+I]);
    for(I = HalfFilter;I < Order;I++)
      Filter[I] = std::real(Ht[I-HalfFilter]);

    delete(Ht);
    return(True);
  }

/***************************************************************************/
