## Copyright (C) 2005 Denis Sbragion
##
## This program is free software; you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 2 of the License, or
##(at your option) any later version.
##
## This program is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
## GNU General Public License for more details.
##
## You should have received a copy of the GNU General Public License
## along with this program; if not, write to the Free Software
## Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
##
## This function is part of DRC - Digital room correction

## usage: tf = tfir(is, ic, cp, fp, fl, at, uf, ft)
##
## Time-frequency impulse response
##
## is: input signal
## is: input signal center
## cp: time axis cicles points
## fp: filtering frequency points
## fl: filter length used for ETC filtering
## at: filter stopband attenuation
## uf: integer upsampling factor for internal computation
## ft: filter type, 0 = lowpass, 1 = bandpass, 2 = morlet
## wm: wavelet frequency multiplier for Morlet wavelet, usually 2

function tf = tfir(is, ic, cp, fp, fl, at, uf, ft, wm);

	# Extracts the minimum frequency point
	mfp = min(fp);
	
	# Computes the upsampling range
	ib = ic + floor(2 * min(cp) / mfp);
	ie = ic + ceil(2 * max(cp) / mfp);
	
	# Rescales the input signal
	is = is(ib:ie);
	ic = ic - ib;
	
	# Rescales the filter length
	uf = round(uf);
	fl = fl * uf;
	
	# Perform the initial upsampling
	ul = length(is) * uf;
	us = zeros(ul,1);
	us(1:uf:ul) = is * uf;
	
	# Allocates the cumulative IR output matrix
	nf = length(fp);
	nc = length(cp);
	tf = zeros(nf,nc);
		
	# Check if morlet
	if (ft == 2)
		# Compute the sampling frequency needed to get the
		# desired base wavelet frequency
		wf = (wm * uf) / mfp;
				
		# Computes the wavelet length needed to ge the desired
		# base filter length
		wl  = (fl - 1) / (2 * wf);
	endif
	
	# Check the number of points for bandpass filtering 
	if (ft == 0 || ft == 1)
		# Add the last frequency point if missing
		if (fp(nf) < 1.0)
			fp = [fp 1.0];
		else
			# Decrease the number of points
			nf = nf - 1;
		endif		
	endif
	
	# Cumulative IR computation loop
	for I = 1:nf		
		# Check the filter type
		if (ft == 0)
			# Computes the filter frequency
			cf = sqrt(fp(I) * fp(I + 1));
			
			# Computes the effective filter len
			efl = round(fl * mfp / cf);
			
			# Lowpass
			ifir = lowpassfir(efl,cf/uf)' .* chebwin(efl,at);
		elseif (ft == 1)
			# Computes the effective filter len
			efl = round(fl * mfp / fp(I));
			
			# Bandpass
			ifir = bandpassfir(efl,fp(I)/uf,fp(I+1)/uf)' .* chebwin(efl,at);
			cf = fp(I+1) - fp(I);
		elseif (ft == 2)					
			# Morlet
			ifir = morletw(wf,mfp / fp(I),0,wl,1,wm * pi);			
			
			# Estract the filter len
			efl = length(ifir);
			cf = fp(I) ^ 0.5;
		else
			# Default, lowpass
			ifir = lowpassfir(efl,fp(I)/uf)' .* chebwin(efl,at);
			cf = fp(I);
		endif
		
		# Applies the filter
		ec = real(fftconv(us,ifir));
	
		# Computes the references
		uc = (ic * uf) + ((1 + efl) / 2);
		el = ul + efl - 1;
		
		# Computes the cicle rescaling
		cs = (-uc * fp(I)) / (2 * uf);
		ce = ((el - uc) * fp(I)) / (2 * uf);
		
		# Interpolate at requested cycle points
		tf(I,:) = interp1(linspace(cs,ce,el), ec, cp, '*pchip') / cf;
	endfor

endfunction