## Copyright (C) 2005 Denis Sbragion
##
## This program is free software; you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 2 of the License, or
##(at your option) any later version.
##
## This program is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
## GNU General Public License for more details.
##
## You should have received a copy of the GNU General Public License
## along with this program; if not, write to the Free Software
## Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
##
## This function is part of DRC - Digital room correction

## usage: createbasicplots(outdir,fprefix);
##
## createbasicplots - Creates the basic DRC plots
##
## outdir = output directory for eps files
## fprefix = prefix for eps files

function createbasicplots(outdir,fprefix);

	# Set the frequency dependent windowing basic parameters
	sf = 20;
	sw = 500;
	ef = 20000;
	ew = 0.5;
	np = 32768;
	
	# Creates the windows array
	wa = linspace(ew,sw,np);
	
	# Computes the gabor limit
	gl = 1000 ./ (2*wa);
	
	# Disable automatic replot
	automatic_replot = 0;
	
	# Set terminal to unknown for initial simulated plotting postscript eps
	__gnuplot_raw__('set terminal unknown;\n');
	
	# Report the stage
	disp('Frequency dependent windowing, linear scale...');
	fflush(stdout);
	
	# Plot the frequency dependent windowing curves
	grid('on');
	__gnuplot_raw__('set xlabel "Time (ms)" 0;\n');
	__gnuplot_raw__('set ylabel "Frequency (Hz)" 0;\n');
	__gnuplot_raw__('set xrange [0.3:550];\n');
	__gnuplot_raw__('set xtics 0,50,550;\n');
	__gnuplot_raw__('set yrange [10:24000];\n');
	__gnuplot_raw__('set ytics 0,1000,24000;\n');
	__gnuplot_raw__('set key right top;\n');
	for gn = 1:10
		plot(wa,fslprefilt(sf,sw,ef,ew,0.2 * gn,np),sprintf(';WE = %.1f;',0.2 * gn));
		hold on;
	endfor
	hold off;
	
	# Reset the gnuplot status
	__gnuplot_raw__('reset;\n');

	# Set terminal to postscript eps for the real plotting
	__gnuplot_raw__('set terminal postscript eps color solid linewidth 1 rounded;\n');
	
	# Set output file
	__gnuplot_raw__([ 'set output ' '"' outdir fprefix '-Linear.eps' '"' '\n' ]);
	
	# Performs the real plot
	replot;
	
	# Reset output file
	__gnuplot_raw__('set output;\n');
	
	# Reset the terminal to unknown
	__gnuplot_raw__('set terminal unknown;\n');
	
	# Report the stage
	disp('Frequency dependent windowing, semilogy scale...');
	fflush(stdout);
	
	# Plot the frequency dependent windowing curves
	grid('on');
	__gnuplot_raw__('set xlabel "Time (ms)" 0;\n');
	__gnuplot_raw__('set ylabel "Frequency (Hz)" 0;\n');
	__gnuplot_raw__('set xrange [0.3:550];\n');
	__gnuplot_raw__('set xtics 0,50,550;\n');
	__gnuplot_raw__('set yrange [10:24000];\n');
	__gnuplot_raw__('set ytics (10,20,30,50,100,200,300,500,1000,2000,3000,5000,10000,20000);\n');
	__gnuplot_raw__('set key right top;\n');
	for gn = 1:10
		semilogy(wa,fslprefilt(sf,sw,ef,ew,0.2 * gn,np),sprintf(';WE = %.1f;',0.2 * gn));
		hold on;
	endfor
	hold off;
	
	# Set terminal to postscript eps for the real plotting
	__gnuplot_raw__('set terminal postscript eps color solid linewidth 1 rounded;\n');
	
	# Set output file
	__gnuplot_raw__([ 'set output ' '"' outdir fprefix '-SemiLogY.eps' '"' '\n' ]);
	
	# Performs the real plot
	replot;
	
	# Reset output file
	__gnuplot_raw__('set output;\n');
	
	# Reset the terminal to unknown
	__gnuplot_raw__('set terminal unknown;\n');
	
	# Report the stage
	disp('Frequency dependent windowing, semilogx scale...');
	fflush(stdout);
	
	# Plot the frequency dependent windowing curves
	grid('on');
	__gnuplot_raw__('set xlabel "Time (ms)" 0;\n');
	__gnuplot_raw__('set ylabel "Frequency (Hz)" 0;\n');
	__gnuplot_raw__('set xrange [0.3:550];\n');
	__gnuplot_raw__('set xtics (0.3,0.5,1,2,3,5,10,20,30,50,100,200,300,500);\n');
	__gnuplot_raw__('set yrange [10:24000];\n');
	__gnuplot_raw__('set ytics 0,1000,24000;\n');
	__gnuplot_raw__('set key right top;\n');
	for gn = 1:10
		semilogx(wa,fslprefilt(sf,sw,ef,ew,0.2 * gn,np),sprintf(";WE = %.1f;",0.2 * gn));
		hold on;
	endfor
	hold off;
	
	# Set terminal to postscript eps for the real plotting
	__gnuplot_raw__('set terminal postscript eps color solid linewidth 1 rounded;\n');
	
	# Set output file
	__gnuplot_raw__([ 'set output ' '"' outdir fprefix '-SemiLogX.eps' '"' '\n' ]);
	
	# Performs the real plot
	replot;
	
	# Reset output file
	__gnuplot_raw__('set output;\n');
	
	# Reset the terminal to unknown
	__gnuplot_raw__('set terminal unknown;\n');
	
	# Report the stage
	disp('Frequency dependent windowing, loglog scale...');
	fflush(stdout);
	
	# Plot the frequency dependent windowing curves
	grid('on');
	__gnuplot_raw__('set xlabel "Time (ms)" 0;\n');
	__gnuplot_raw__('set ylabel "Frequency (Hz)" 0;\n');
	__gnuplot_raw__('set xrange [0.3:550];\n');
	__gnuplot_raw__('set xtics (0.3,0.5,1,2,3,5,10,20,30,50,100,200,300,500);\n');
	__gnuplot_raw__('set yrange [10:24000];\n');
	__gnuplot_raw__('set ytics (10,20,30,50,100,200,300,500,1000,2000,3000,5000,10000,20000);\n');
	__gnuplot_raw__('set key right top;\n');
	for gn = 1:10
		loglog(wa,fslprefilt(sf,sw,ef,ew,0.2 * gn,np),sprintf(';WE = %.1f;',0.2 * gn));
		hold on;
	endfor
	hold off;
	
	# Set terminal to postscript eps for the real plotting
	__gnuplot_raw__('set terminal postscript eps color solid linewidth 1 rounded;\n');
	
	# Set output file
	__gnuplot_raw__([ 'set output ' '"' outdir fprefix '-LogLog.eps' '"' '\n' ]);
	
	# Performs the real plot
	replot;
	
	# Reset output file
	__gnuplot_raw__('set output;\n');
	
	# Reset the terminal to unknown
	__gnuplot_raw__('set terminal unknown;\n');
	
	# Report the stage
	disp('Frequency dependent windowing, loglog scale with gabor limit...');
	fflush(stdout);
	
	# Plot the frequency dependent windowing curves
	grid('on');
	__gnuplot_raw__('set xlabel "Time (ms)" 0;\n');
	__gnuplot_raw__('set ylabel "Frequency (Hz)" 0;\n');
	__gnuplot_raw__('set xrange [0.3:550];\n');
	__gnuplot_raw__('set xtics (0.3,0.5,1,2,3,5,10,20,30,50,100,200,300,500);\n');
	__gnuplot_raw__('set yrange [10:24000];\n');
	__gnuplot_raw__('set ytics (10,20,30,50,100,200,300,500,1000,2000,3000,5000,10000,20000);\n');
	__gnuplot_raw__('set key right top;\n');
	for gn = 1:10
		loglog(wa,fslprefilt(sf,sw,ef,ew,0.2 * gn,np),sprintf(';WE = %.1f;',0.2 * gn));
		hold on;
	endfor
	loglog(wa,gl,';Gabor;');
	hold off;
	
	# Set terminal to postscript eps for the real plotting
	__gnuplot_raw__('set terminal postscript eps color solid linewidth 1 rounded;\n');
	
	# Set output file
	__gnuplot_raw__([ 'set output ' '"' outdir fprefix '-Gabor.eps' '"' '\n' ]);
	
	# Performs the real plot
	replot;
	
	# Reset output file
	__gnuplot_raw__('set output;\n');
	
	# Reset the terminal to unknown
	__gnuplot_raw__('set terminal unknown;\n');
	
	# Report the stage
	disp('Frequency dependent windowing, time-frequency jail...');
	fflush(stdout);
	
	# Plot the frequency dependent windowing curves
	grid('on');
	__gnuplot_raw__('set xlabel "Time (ms)" 0;\n');
	__gnuplot_raw__('set ylabel "Frequency (Hz)" 0;\n');
	__gnuplot_raw__('set xrange [-50:550];\n');
	__gnuplot_raw__('set xtics -50,50,550;\n');
	__gnuplot_raw__('set yrange [10:24000];\n');
	__gnuplot_raw__('set ytics (10,20,30,50,100,200,300,500,1000,2000,3000,5000,10000,20000);\n');
	__gnuplot_raw__('set key right top;\n');
	semilogy(-wa/20,fslprefilt(sf,sw/20,ef,ew,1.0,np),'1;Pre-echo truncation;');
	hold on;
	semilogy(wa,fslprefilt(sf,sw,ef,ew,1.0,np),'3;Ringing truncation;');
	hold off;
	
	# Set terminal to postscript eps for the real plotting
	__gnuplot_raw__('set terminal postscript eps color solid linewidth 1 rounded;\n');
	
	# Set output file
	__gnuplot_raw__([ 'set output ' '"' outdir fprefix '-Jail.eps' '"' '\n' ]);
	
	# Performs the real plot
	replot;
	
	# Reset output file
	__gnuplot_raw__('set output;\n');
	
	# Reset the terminal to unknown
	__gnuplot_raw__('set terminal unknown;\n');
	
	# Report the stage
	disp('Frequency dependent windowing, psychoacoustic bandwidth comparison...');
	fflush(stdout);
	
	# Plot the frequency dependent windowing curves
	grid('on');
	__gnuplot_raw__('set xlabel "Frequency (KHz)" 0;\n');
	__gnuplot_raw__('set ylabel "Frequency (Hz)" 0;\n');
	__gnuplot_raw__('set xrange [0.01:24];\n');
	__gnuplot_raw__('set xtics (0.01,0.02,0.03,0.05,0.1,0.2,0.3,0.5,1,2,3,5,10,20);\n');
	__gnuplot_raw__('set yrange [0.5:10000];\n');
	__gnuplot_raw__('set ytics (0.5,1,2,3,5,10,20,30,50,100,200,300,500,1000,2000,3000,5000,10000);\n');
	__gnuplot_raw__('set key left top;\n');
	loglog(linspace(sf,ef,np)/1000,(1000 * 0.5) ./ (wslprefilt(sf,sw,ef,ew,1.0,np) + wslprefilt(sf,sw/20,ef,ew,1.0,np)),'1;normal.drc;');
	hold on;
	loglog(linspace(sf,ef,np)/1000,(1000 * 2.0) ./ (wbslprefilt(sf,65,ef,0.46,1.87,np) + wbslprefilt(sf,10,ef,0.46,1.87,np)),'2;erb.drc;');
	hold on;
	loglog(linspace(sf,ef,np)/1000,barkbwidth(linspace(sf,ef,np)),'3;Bark;');
	hold on;
	loglog(linspace(sf,ef,np)/1000,erbbwidth(linspace(sf,ef,np)),'4;ERB;');
	hold on;
	loglog(linspace(sf,ef,np)/1000,foctbwidth(linspace(sf,ef,np),1/3),'5;1-3 oct;');
	hold on;
	loglog(linspace(sf,ef,np)/1000,foctbwidth(linspace(sf,ef,np),1/6),'6;1-6 oct;');
	hold off;
	
	# Set terminal to postscript eps for the real plotting
	__gnuplot_raw__('set terminal postscript eps color solid linewidth 1 rounded;\n');
	
	# Set output file
	__gnuplot_raw__([ 'set output ' '"' outdir fprefix '-BWidthCmp.eps' '"' '\n' ]);
	
	# Performs the real plot
	replot;
	
	# Reset output file
	__gnuplot_raw__('set output;\n');
	
	# Reset the terminal to unknown
	__gnuplot_raw__('set terminal unknown;\n');
	
	# Report the stage
	disp('Frequency dependent windowing, proportional and bilinear comparison...');
	fflush(stdout);
	
	# Plot the frequency dependent windowing curves
	grid('on');
	__gnuplot_raw__('set xlabel "Time (ms)" 0;\n');
	__gnuplot_raw__('set ylabel "Frequency (Hz)" 0;\n');
	__gnuplot_raw__('set xrange [0.3:550];\n');
	__gnuplot_raw__('set xtics (0.3,0.5,1,2,3,5,10,20,30,50,100,200,300,500);\n');
	__gnuplot_raw__('set yrange [10:24000];\n');
	__gnuplot_raw__('set ytics (10,20,30,50,100,200,300,500,1000,2000,3000,5000,10000,20000);\n');
	__gnuplot_raw__('set key right top;\n');
	for gn = 1:7
		loglog(wa,fslprefilt(sf,sw,ef,ew,0.2 + 0.2 * gn,np),sprintf('1;Prop. WE = %.1f;',0.2 + 0.2 * gn));
		hold on;
	endfor
	for gn = 1:7
		loglog(wa,fbslprefilt(sf,sw,ef,ew,0.2 + 0.2 * gn,np),sprintf('3;Bil. WE = %.1f;',0.2 + 0.2 * gn));
		hold on;
	endfor
	hold off;
	
	# Set terminal to postscript eps for the real plotting
	__gnuplot_raw__('set terminal postscript eps color solid linewidth 1 rounded;\n');
	
	# Set output file
	__gnuplot_raw__([ 'set output ' '"' outdir fprefix '-BPComparison.eps' '"' '\n' ]);
	
	# Performs the real plot
	replot;
	
	# Reset output file
	__gnuplot_raw__('set output;\n');
	
	# Reset the terminal to unknown
	__gnuplot_raw__('set terminal unknown;\n');
	
	# Remove variables no longer used
	clear sf sw ef ew wa gl;
	
	# Reset the terminal to unknown
	__gnuplot_raw__('set terminal unknown;\n');
	
	# Report the stage
	disp('Transfer functions...');
	fflush(stdout);
	
	# Prepare the target functions
	bkf = [ 0 10 20 400 12800 20000 21500 22050 ];
	bkm = [ -20.0 -10.0 0.00 0.00 -5.00 -6.00 -10.0 -20.0 ];
	
	bk2f = [ 0 10 20 200 12800 20000 21500 22050 ];
	bk2m = [ -20.0 -10.0 0.00 0.00 -3.0 -3.5 -10.0 -20.0 ];
	
	bk2sf = [ 0 16 18 200 12800 20000 21500 22050 ];
	bk2sm = [ -200.0 -120.0 0.00 0.00 -3.0 -3.5 -10.0 -20.0 ];
	
	bk3f = [ 0 10 20 100 12800 20000 21500 22050 ];
	bk3m = [ -20.0 -10.0 0.00 0.00 -3.50 -4.00 -10.0 -20.0 ];
	
	bk3sf = [ 0 16 18 100 12800 20000 21500 22050 ];
	bk3sm = [ -200.0 -120.0 0.00 0.00 -3.50 -4.00 -10.0 -20.0 ];
	
	bk3sfsp = [ 0 14 15 16 17 18 19 20 50 100 150 19900 20000 20100 20200 20300 20400 20500 22050 ];
	bk3smsp = [ -200.0 -200.0 -200.0 -200.0 -100.0 0.00 0.00 0.00 0.00 0.00 0.00 -3.95 -4.00 -4.00 -98.0 -200.0 -200.0 -200.0 -200.0 ];
	
	# Plot the target functions
	grid('on');
	__gnuplot_raw__('set xlabel "Frequency (KHz)" 0;\n');
	__gnuplot_raw__('set ylabel "Magnitude (dB)" 0;\n');
	__gnuplot_raw__('set xrange [0.01:22.5];\n');
	__gnuplot_raw__('set xtics (0.01,0.02,0.03,0.05,0.1,0.2,0.3,0.5,1,2,3,5,10,20);\n');
	__gnuplot_raw__('set yrange [-30:9];\n');
	__gnuplot_raw__('set ytics -30,3,9;\n');
	__gnuplot_raw__('set key right top;\n');
	
	semilogx(bkf/1000,bkm,'1;bk;');
	hold on;
	
	semilogx(bk2f/1000,bk2m,'2;bk-2;');
	hold on;
	
	semilogx(bk3f/1000,bk3m,'5;bk-3;');
	hold on;
	
	semilogx(bk2sf/1000,bk2sm,'4;bk-2-sub;');
	hold on;
	
	semilogx(bk3sf/1000,bk3sm,'3;bk-3-sub;');
	hold on;
	
	lbk3sfsp = log10(1 + bk3sfsp);
	spx = log10(logspace(log10(1),log10(22051),np));
	semilogx((-1.0 + (10 .^ ub3splinev(lbk3sfsp,lbk3sfsp,spx))) / 1000,-10 + ub3splinev(lbk3sfsp,bk3smsp,spx),'8;bk-3-subultra-spline (-10 dB);');
	clear spx;
	hold off;
	
	# Set terminal to postscript eps for the real plotting
	__gnuplot_raw__('set terminal postscript eps color solid linewidth 1 rounded;\n');
	
	# Set output file
	__gnuplot_raw__([ 'set output ' '"' outdir fprefix '-DTFCmp.eps' '"' '\n' ]);
	
	# Performs the real plot
	replot;
	
	# Reset output file
	__gnuplot_raw__('set output;\n');
	
	# Reset the terminal to unknown
	__gnuplot_raw__('set terminal unknown;\n');
	
	# Remove variables no longer used
	clear sf sw ef ew wa gl;
	
	# Report the stage
	disp('Completed, wait for GNUPlot to complete before opening output files.');
	fflush(stdout);

endfunction