/*
    This file is part of CDLoop
    Copyright (C) 1997, 1998, 1999  Claus Brunzema (chb@ossi.fho-emden.de)

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  -
  -  cdplayer.h
  -
  -  the cdplayer-engine.
  -
  - $Id: cdplayer.h,v 1.1.1.1 1999/06/04 13:05:41 chb Exp $
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/

#ifndef CDPLAYER_H_INCLUDED
#define CDPLAYER_H_INCLUDED


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - constants indicating the status of the cdplayer
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#define STATUS_NO_TOC   1  /* no toc available (no disc) */
#define STATUS_STOP     2  /* cdplayer is stopped */
#define STATUS_DATA     3  /* disc contains only datatracks */
#define STATUS_PLAY     4  /* cdplayer plays */
#define STATUS_PAUSE    5  /* cdplayer is in pause mode */

#define LOOP_NONE  0       /* no loop defined */
#define LOOP_BEGIN 1       /* only the beginning of the loop is defined */
#define LOOP_AB    2       /* cdplayer is playing between locator a and b */
#define LOOP_GAP   3       /* cdplayer is in the gap */


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - bitmask-constants for cdplayer_ask_changes 
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#define STATUS_CHANGE    (1<<0)  /* status (STOP, PLAY etc.) has changed */
#define TOC_CHANGE       (1<<1)  /* new toc */ 
#define POS_CHANGE       (1<<2)  /* new position */
#define LOCATOR_A_CHANGE (1<<3)  /* locator a changed */
#define LOCATOR_B_CHANGE (1<<4)  /* locator b changed */
#define GAP_CHANGE       (1<<5)  /* gap changed */


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - void cdplayer_init(char *device)
  -  initializes the cdplayer.
  -  in: char *device   - name of the cd-device (e.g /dev/cdrom)
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
void cdplayer_init(char *device);


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - void cdplayer_exit(void)
  -  shuts down the cdplayer.
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
void cdplayer_exit(void);


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - int cdplayer_ask_changes(void)
  -  gets a bitmask indicating changes in the cdplayer.
  -  out: int             - the bitmask indicating changes (see #defines).
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int cdplayer_ask_changes(void);


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - void cdplayer_get_status(int *st,int *loop_st)
  -  gets the status of the cdplayer.
  -  out: int *st         - status of the cdplayer (STATUS_PLAY,
  -                         STATUS_NO_TOC etc.).
  -       int *loop_st    - loop status (LOOP_BEGIN, LOOP_GAP etc.).
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
void cdplayer_get_status(int *st,int *loop_st);


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - int cdplayer_get_n_track(void)
  -  gets the number of tracks of the current disc.
  -  out: int *st         - status of the cdplayer (STATUS_PLAY,
  -                         STATUS_NO_TOC etc.).
  -       int *loop_st    - loop status (LOOP_BEGIN, LOOP_GAP etc.).
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int cdplayer_get_n_track(void);


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - void cdplayer_get_pos(int *track,int *m,int *s,int *f)
  -  gets the current relative position of the cdplayer.
  -  out: int *track      - current track.
  -       int *m          - current minute.
  -       int *s          - current second.
  -       int *f          - current frame.
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
void cdplayer_get_pos(int *track,int *m,int *s,int *f);


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - void cdplayer_play(void)
  -  starts playing.
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
void cdplayer_play(void);


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - void cdplayer_play_at(int track)
  -  starts playing the specified track.
  -  in:  int track       - the track to begin playing with. The first
  -                         track has the number 1.
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
void cdplayer_play_at(int track);


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - void cdplayer_play_prev(void)
  -  sets the playing position to the beginning of the previous track. If
  -  already playing the first track, the first track is restarted.
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
void cdplayer_play_prev(void);


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - void cdplayer_play_next(void)
  -  sets the playing position to the beginning of the next track. Does
  -  nothing if already playing the last track.
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
void cdplayer_play_next(void);


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - void cdplayer_restart(void)
  -  restarts the cdplayer if in pause mode.
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
void cdplayer_restart(void);


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - void cdplayer_pause(void)
  -  puts the cdplayer in pause mode.
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
void cdplayer_pause(void);


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - void cdplayer_stop(void)
  -  stops the cdplayer.
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
void cdplayer_stop(void);


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - void cdplayer_search(long offset)
  -  changes the playing position according to offset.
  -  in:  long offset     - the (positive or negative) offset in frames.    
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
void cdplayer_search(long offset);


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - void cdplayer_eject(void)
  -  ejects the disc.
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
void cdplayer_eject(void);


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - void cdplayer_close_tray(void)
  -  closes the tray of the cdrom-drive.
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
void cdplayer_close_tray(void);


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - void cdplayer_set_locator_a(void)
  -  sets locator a to the current position.
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
void cdplayer_set_locator_a(void);


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - void cdplayer_set_locator_b(void)
  -  sets locator b to the current position.
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
void cdplayer_set_locator_b(void);


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - void cdplayer_get_locator_a(void)
  -  gets the position of locator a.
  -  out: int *track      - track of locator a.
  -       int *m          - minute of locator a.
  -       int *s          - second of locator a.
  -       int *f          - frame of locator a.
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
void cdplayer_get_locator_a(int *track,int *m,int *s,int *f);


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - void cdplayer_get_locator_b(void)
  -  gets the position of locator a.
  -  out: int *track      - track of locator b.
  -       int *m          - minute of locator b.
  -       int *s          - second of locator b.
  -       int *f          - frame of locator b.
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
void cdplayer_get_locator_b(int *track,int *m,int *s,int *f);


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - void cdplayer_clear_locator_b(void)
  -  clears locator b.
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
void cdplayer_clear_locator_b(void);


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - void cdplayer_clear_locators(void)
  -  clears both locators (and the gap, a gap without locators makes
  -  no sense).
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
void cdplayer_clear_locators(void);


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - void cdplayer_move_locator_a(long offset)
  -  moves locator a according to offset.
  -  in:  long offset     - the (positive or negative) offset in frames.
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
void cdplayer_move_locator_a(long offset);


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - void cdplayer_move_locator_b(long offset)
  -  moves locator b according to offset.
  -  in:  long offset     - the (positive or negative) offset in frames.
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
void cdplayer_move_locator_b(long offset);


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - void cdplayer_set_gap(long gap)
  -  sets the length of the gap.
  -  in:  long gap        - the length of the gap.
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
void cdplayer_set_gap(long gap);


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - long cdplayer_get_gap(void)
  -  gets the length of the gap.
  -  out: long             - the length of the gap
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
long cdplayer_get_gap(void);

#endif

