/*
    This file is part of CDLoop
    Copyright (C) 1997, 1998, 1999  Claus Brunzema (chb@ossi.fho-emden.de)

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  -
  -  cdplayer.c
  -
  -  the cdplayer-engine.
  -
  - $Id: cdplayer.c,v 1.1.1.1 1999/06/04 13:05:41 chb Exp $
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/


#include "cdplayer.h"
#include "cdhw.h"
#include "err.h"

#include <stdio.h>
#include <unistd.h>
#include <wait.h>
#include <limits.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#include <sys/time.h>
#include <sys/types.h>


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - return codes for the cdplayer control process
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#define INIT                -1    /* startup phase */
#define READY                0    /* ready to receive commands */


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - command codes for the cdplayer control process
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#define CMD_EXIT             1
#define CMD_GET_STATUS       2
#define CMD_GET_N_TRACK      3
#define CMD_GET_POS          4
#define CMD_PLAY             5
#define CMD_PLAY_AT          6
#define CMD_PLAY_PREV        7
#define CMD_PLAY_NEXT        8
#define CMD_RESTART          9
#define CMD_PAUSE           10
#define CMD_STOP            11
#define CMD_SEARCH          12
#define CMD_EJECT           13
#define CMD_CLOSE_TRAY      14
#define CMD_SET_LOCATOR_A   15
#define CMD_SET_LOCATOR_B   16
#define CMD_GET_LOCATOR_A   17
#define CMD_GET_LOCATOR_B   18
#define CMD_CLEAR_LOCATOR_B 19
#define CMD_CLEAR_LOCATORS  20
#define CMD_MOVE_LOCATOR_A  21
#define CMD_MOVE_LOCATOR_B  22
#define CMD_SET_GAP         23
#define CMD_GET_GAP         24


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - misc constants
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#define MIN_LOOP   30       /* minimum number of frames in a loop */
#define GAP_FACTOR 10000L   /* scaling factor for the gap */


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - variables global to this module
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
pid_t cd_player_control_process;  /* process id for the cdplayer controller */

int shmid;                        /* id of the shared memory area */

struct shmblock                   /* structure of the shared memory area */
{  volatile int     command;
   volatile long    offset_buffer;
   volatile int     track_buffer;
   volatile cd_time pos_buffer;
   volatile int     status_buffer;
   volatile int     loop_status_buffer;
   volatile long    gap_buffer;
   volatile int     change;
};

struct shmblock *shm;             /* pointer to the shared memory area */


/*========================================================================
  - static functions
  ========================================================================*/

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - void set_simple_command(int cmd)
  -  sends a command without extra arguments to the cdplayer controller
  -  process.
  -  in:  int cmd         - the command code.
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
static void set_simple_command(int cmd)
{  while(shm->command!=READY)
   { }
   shm->command=cmd;
}


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - void set_command_with_offset(int cmd,long offset)
  -  sends a command with offset argument to the cdplayer controller
  -  process.
  -  in:  int cmd         - the command code.
  -       long offset     - the offset.
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
static void set_command_with_offset(int cmd,long offset)
{  while(shm->command!=READY)
   { }
   shm->offset_buffer=offset;
   shm->command=cmd;
}


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - void set_command_get_pos(int cmd,int *track,int *m,int *s,int *f)
  -  sends a command without extra arguments to the cdplayer controller
  -  and returns position data received from the controller.
  -  in:  int cmd         - the command code.
  -  out: int *track      - the track number
  -       int *m          - minute.
  -       int *s          - second.
  -       int *f          - frame.
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
static void set_command_get_pos(int cmd,int *track,int *m,int *s,int *f)
{  set_simple_command(cmd);
   while(shm->command!=READY)
   { }
   *track=shm->track_buffer;   
   *m=shm->pos_buffer.m;   
   *s=shm->pos_buffer.s;   
   *f=shm->pos_buffer.f;
}


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - void set_unknown_pos_rel(void)
  -  makes the position info in the shared memory area invalid.
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
static void set_unknown_pos_rel(void)
{  shm->track_buffer=-1;
   shm->pos_buffer.m=shm->pos_buffer.s=shm->pos_buffer.f=-1;
   shm->command=READY;
}


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - long timeval_diff(struct timeval *first,struct timeval *last)
  -  calculates the difference between two timeval structures.
  -  in:  struct timeval *first - the earlier point in time.
  -       struct timeval *last  - the later point in time.
  -  out: long                  - the difference in microseconds.
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
static long timeval_diff(struct timeval *first,struct timeval *last)
{  return (last->tv_usec-first->tv_usec)+(last->tv_sec-first->tv_sec)*1000000;
}


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - void cd_player_control(void)
  -  This is the cdplayer controller. It runs in a child process and
  -  communicates with the rest of the application via the shared memory
  -  area.
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
static void cd_player_control(void)
{  /* some general purpose buffers */
   int buffer;
   cd_time timebuffer;
   ULONG framebuffer;
   struct timeval tvbuffer;
   
   int status;                          /* status of the cdplayer */
   int old_status;
   int loop_status;                     /* loop status of the cdplayer */
   int old_loop_status;
   cd_time locatorA;                    /* position of locator a */
   cd_time locatorB;                    /* position of locator b */
   ULONG locatorAframe;                 /* frame position of locator a */
   ULONG locatorBframe;                 /* frame position of locator b */
   cd_time firstpos;                    /* first position on the disc */
   cd_time lastpos;                     /* last position on the disc */
   cd_time old_pos;
   long gap;                            /* length of the gap */
   struct timeval gap_start;            /* when in the gap: time when gap
                                           started */


   status=STATUS_NO_TOC;
   loop_status=LOOP_NONE;
   gap=0L;
   shm->command=READY;
   /* this is the main loop */
   while(shm->command!=CMD_EXIT)
   {  old_status=status;
      old_loop_status=loop_status;
      old_pos=cd_status.pos_abs;

      if(cd_read_status())
      {  status=STATUS_NO_TOC;
      }
      else
      {  if(old_status==STATUS_NO_TOC)
         {  if(!cd_read_toc())
            {  if(cd_toc.n_track>0)
               {  /* a new audio disc is in the drive */
	          status=STATUS_STOP;
                  firstpos=cd_toc.track[0].start;
                  lastpos=cd_toc.track[cd_toc.n_track-1].end;
               }
               else
               {  /* we have a data disc */
	          status=STATUS_DATA;
               }
               loop_status=LOOP_NONE;
               shm->change|=TOC_CHANGE;
            }
         }
      }   

      /* are we playing in a loop ? */
      if(status==STATUS_PLAY && loop_status==LOOP_AB)
      {  /* is the loop completed ? */
         if(cd_time_to_frame(&cd_status.pos_abs)>=locatorBframe)
         {  /* do we have a gap ? */
	    if(gap==0L)
            {  /* restart the loop */
	       cd_play(&locatorA,&locatorB);
            }
            else
            {  /* go into the gap */
	       cd_pause();
               gettimeofday(&gap_start,NULL);
               loop_status=LOOP_GAP;
	       shm->change|=POS_CHANGE;
            }
         }
      }

      /* are we in a gap ? */
      if(status==STATUS_PLAY && loop_status==LOOP_GAP)
      {  gettimeofday(&tvbuffer,NULL);
         /* is th gap completed ? */
         if(timeval_diff(&gap_start,&tvbuffer)>=gap)
         {  cd_play(&locatorA,&locatorB);
            loop_status=LOOP_AB;
         }
      }

      if(status==STATUS_PLAY && loop_status==LOOP_NONE)
      {  /* is the disc completed ? */
         if(cd_time_to_frame(&cd_status.pos_abs)>=
            cd_time_to_frame(&lastpos))
         {  cd_stop();
            status=STATUS_STOP;
            shm->change|=POS_CHANGE;
	 }
      }
      
      /* position changed ? */
      if(status==STATUS_PLAY &&
         (old_pos.f!=cd_status.pos_abs.f ||
          old_pos.s!=cd_status.pos_abs.s ||
          old_pos.m!=cd_status.pos_abs.m))
      {  shm->change|=POS_CHANGE;
      }

      switch(shm->command)
      {  case READY:        break;
         case CMD_GET_STATUS:
                            shm->status_buffer=status;
                            shm->loop_status_buffer=loop_status;
                            shm->change&=~STATUS_CHANGE;
                            shm->command=READY;
                            break;

         case CMD_GET_N_TRACK:
                            if(status==STATUS_NO_TOC)
                            {  shm->track_buffer=-1;
                            }
                            else
                            {  shm->track_buffer=cd_toc.n_track;
                            }
                            shm->change&=~TOC_CHANGE;
                            shm->command=READY;
                            break;
                               
         case CMD_GET_POS:  if(status==STATUS_NO_TOC ||
                               status==STATUS_STOP || 
			       status==STATUS_DATA ||
			       loop_status==LOOP_GAP)
                            {  set_unknown_pos_rel();
                            }
                            else      
                            {  shm->track_buffer=cd_status.track+1;
                               shm->pos_buffer=cd_status.pos_rel;
                            }
                            shm->change&=~POS_CHANGE;
                            shm->command=READY;
                            break;      

         case CMD_PLAY:     switch(status)
	                    {  case STATUS_DATA:
			       case STATUS_NO_TOC:
			              break;
			       case STATUS_STOP:  
                                      cd_play(&firstpos,&lastpos);
				      status=STATUS_PLAY;
				      break;
			       case STATUS_PLAY:
			              switch(loop_status)
				      {  case LOOP_NONE:  
                                                break;
					 case LOOP_BEGIN: 
					        cd_play(&locatorA,&lastpos);
						break;
					 case LOOP_AB:
					        cd_play(&locatorA,&locatorB);
						break;
					 case LOOP_GAP:
					        cd_play(&locatorA,&locatorB);
						loop_status=LOOP_AB;
						break;
				      }
				      break;
			       case STATUS_PAUSE:
			              if(loop_status==LOOP_GAP)
				      {  cd_play(&locatorA,&locatorB);
				         status=STATUS_PLAY;
					 loop_status=LOOP_AB;
				      }
				      else
				      {  cd_resume();
					 status=STATUS_PLAY;
				      }
				      break;
			    }
                            shm->command=READY;
                            break;
                               
         case CMD_PLAY_AT:  if(status!=STATUS_NO_TOC && status!=STATUS_DATA &&
                               (shm->track_buffer-1)>=0 &&
                               (shm->track_buffer-1)<cd_toc.n_track)
                            {  if(!cd_play(&cd_toc.track
			                       [shm->track_buffer-1].start,
                                           &lastpos))
                               {  status=STATUS_PLAY;
                                  loop_status=LOOP_NONE;
				  gap=0L;
                                  shm->change|=
				      (LOCATOR_A_CHANGE | 
				       LOCATOR_B_CHANGE |
				       GAP_CHANGE);
                               }
                            }
                            shm->command=READY;
                            break;

         case CMD_PLAY_PREV:
	                    if(status==STATUS_PLAY || status==STATUS_PAUSE)
                            {  if(!cd_play(&cd_toc.track
			                     [(cd_status.track<1) ?
					          0 :
						  (cd_status.track-1)].start,
                                           &lastpos))
                               {  status=STATUS_PLAY;
                                  loop_status=LOOP_NONE;
				  gap=0L;
                                  shm->change|=
				      (LOCATOR_A_CHANGE | 
				       LOCATOR_B_CHANGE |
				       GAP_CHANGE);
                               }
                            }
                            shm->command=READY;
                            break;

         case CMD_PLAY_NEXT:
	                    if(status==STATUS_PLAY || status==STATUS_PAUSE)
                            {  if(cd_status.track<cd_toc.n_track-1)
			       {  if(!cd_play(&cd_toc.track
			                          [(cd_status.track+1)].start,
					      &lastpos))
				  {  status=STATUS_PLAY;
				     loop_status=LOOP_NONE;
				     gap=0L;
				     shm->change|=
					 (LOCATOR_A_CHANGE | 
					  LOCATOR_B_CHANGE |
					  GAP_CHANGE);
				  }
			       }
                            }
                            shm->command=READY;
                            break;

         case CMD_RESTART:  if(status==STATUS_PLAY || status==STATUS_PAUSE)
			    {  if(loop_status==LOOP_NONE)
			       {  if(!cd_play(&cd_toc.track
			                         [cd_status.track].start,
					     &lastpos))
			         {  status=STATUS_PLAY;
			         }
			       }
			       else
			       {  if(loop_status==LOOP_BEGIN)
			          {  if(!cd_play(&locatorA,&lastpos))
			             {  status=STATUS_PLAY;
			             }
				  }
				  else
				  {  if(!cd_play(&locatorA,&locatorB))
			             {  status=STATUS_PLAY;
				        loop_status=LOOP_AB;
			             }
				  }
			       }
			    }
                            shm->command=READY;
                            break;

         case CMD_PAUSE:    if(status==STATUS_PLAY)
                            {  if(loop_status!=LOOP_GAP)
			       {  cd_pause();
			       }
                               status=STATUS_PAUSE;
			    }	 
			    else
                            {  if(status==STATUS_PAUSE)
                               {   if(loop_status==LOOP_GAP)
			           {  cd_play(&locatorA,&locatorB);
			              status=STATUS_PLAY;
  			 	      loop_status=LOOP_AB;
				   }
				   else
				   {  cd_resume();
				      status=STATUS_PLAY;
				   }
			       }	   
                            }
                            shm->command=READY;
                            break;
                             
         case CMD_STOP:     if(status!=STATUS_NO_TOC && status!=STATUS_DATA)
                            {  cd_stop();
                               status=STATUS_STOP;
                               loop_status=LOOP_NONE;
			       gap=0L;
                               shm->change|=
				   (POS_CHANGE |
				    LOCATOR_A_CHANGE | 
				    LOCATOR_B_CHANGE |
				    GAP_CHANGE);
                            }
                            shm->command=READY;
                            break;
                               
         case CMD_SEARCH:   if(status==STATUS_PLAY && loop_status!=LOOP_GAP)
                            {  timebuffer=cd_status.pos_abs;
                               add_to_cd_time(&timebuffer,shm->offset_buffer);
                               framebuffer=cd_time_to_frame(&timebuffer);
                               if(framebuffer>=
                                      cd_time_to_frame(&firstpos) &&
                                  framebuffer<=
                                      cd_time_to_frame(&lastpos))
                               {  if(loop_status==LOOP_NONE)
                                  {  cd_play(&timebuffer,&lastpos);
                                  }
                                  else
                                  {  if(framebuffer>=locatorAframe &&
                                        framebuffer<=locatorBframe)
                                     {  cd_play(&timebuffer,&locatorB);
                                     }
                                  }
                               }
                            }
                            shm->command=READY;
                            break;
                               
         case CMD_EJECT:    if(status!=STATUS_NO_TOC && status!=STATUS_DATA)
                            {  cd_stop();
                            }
                            cd_eject();
                            status=STATUS_NO_TOC;
                            loop_status=LOOP_NONE;
			    gap=0L;
                            shm->change|=
			        (TOC_CHANGE |
				 POS_CHANGE | 
				 LOCATOR_A_CHANGE |
				 LOCATOR_B_CHANGE |
				 GAP_CHANGE);
                            shm->command=READY;
                            break;
                               
         case CMD_CLOSE_TRAY:
                            if(status==STATUS_NO_TOC)
                            {  cd_close_tray();
                            }
                            shm->command=READY;
                            break;
                               
         case CMD_SET_LOCATOR_A:
                            if(status==STATUS_PLAY && loop_status!=LOOP_GAP)
                            {  if(loop_status==LOOP_NONE || 
                                  loop_status==LOOP_BEGIN)
                               {  locatorA=cd_status.pos_abs;
                                  locatorAframe=cd_time_to_frame(&locatorA);
                                  loop_status=LOOP_BEGIN;
                                  shm->change|=LOCATOR_A_CHANGE;
                               }
                               else
                               {  framebuffer=cd_time_to_frame(&cd_status.pos_abs);
                                  if(locatorBframe>framebuffer && 
                                     (locatorBframe-framebuffer)>=MIN_LOOP)
                                  {  locatorA=cd_status.pos_abs;
                                     locatorAframe=cd_time_to_frame(&locatorA);
                                     shm->change|=LOCATOR_A_CHANGE;
                                  }
                               }
                            }
                            shm->command=READY;
                            break;
                               
         case CMD_SET_LOCATOR_B:
                            if(status==STATUS_PLAY && 
                               (loop_status==LOOP_BEGIN || loop_status==LOOP_AB))
                            {  framebuffer=cd_time_to_frame(&cd_status.pos_abs);
                               if(framebuffer>locatorAframe && 
                                  (framebuffer-locatorAframe)>MIN_LOOP)
                               {  locatorB=cd_status.pos_abs;
                                  locatorBframe=cd_time_to_frame(&locatorB);
                                  cd_play(&locatorA,&locatorB);
				  loop_status=LOOP_AB;
                                  shm->change|=LOCATOR_B_CHANGE;
                               }
                            }
                            shm->command=READY;
                            break;
           
         case CMD_GET_LOCATOR_A:
                            if((status!=STATUS_PLAY && 
			          status!=STATUS_PAUSE)|| 
                                loop_status==LOOP_NONE)
                            {  set_unknown_pos_rel();
                            }
                            else
                            {  if(cd_abs_to_rel(&locatorA,
                                                &buffer,
                                                &timebuffer))
                               {  set_unknown_pos_rel();
                               }
                               else
                               {  shm->track_buffer=buffer+1;
                                  shm->pos_buffer=timebuffer;
                               }
                            }
                            shm->change&=~LOCATOR_A_CHANGE;
                            shm->command=READY;
                            break;      

         case CMD_GET_LOCATOR_B:
                            if((status!=STATUS_PLAY &&
			         status!=STATUS_PAUSE) || 
			       (loop_status!=LOOP_AB &&
			        loop_status!=LOOP_GAP))
                            {  set_unknown_pos_rel();
                            }
                            else
                            {  if(cd_abs_to_rel(&locatorB,
                                                   &buffer,
                                                   &timebuffer))
                               {  set_unknown_pos_rel();
                               }
                               else
                               {  shm->track_buffer=buffer+1;
                                  shm->pos_buffer=timebuffer;
                               }
                            }
                            shm->change&=~LOCATOR_B_CHANGE;
                            shm->command=READY;
                            break;

         case CMD_CLEAR_LOCATOR_B:
                            if(status==STATUS_PLAY)
			    {  if(loop_status==LOOP_AB || loop_status==LOOP_GAP)
			       {  cd_play(&locatorA,&lastpos);
			          loop_status=LOOP_BEGIN;
				  gap=0L;
			          shm->change|=(LOCATOR_B_CHANGE |
				                GAP_CHANGE);
			       }
			    }
			    shm->command=READY;
			    break;

         case CMD_CLEAR_LOCATORS:
                            if(status==STATUS_PLAY)
                            {  if(loop_status!=LOOP_NONE)
                               {  cd_play(&locatorA,&lastpos);
                               }
                               loop_status=LOOP_NONE;
			       gap=0L;
                               shm->change|=
			           (LOCATOR_A_CHANGE | 
				    LOCATOR_B_CHANGE |
				    GAP_CHANGE);
                            }
                            shm->command=READY;
                            break;

         case CMD_MOVE_LOCATOR_A:
                            if(status==STATUS_PLAY && loop_status!=LOOP_NONE)
                            {  framebuffer=cd_time_to_frame(&locatorA);
                               framebuffer+=shm->offset_buffer;
                               if(framebuffer>=
                                     cd_time_to_frame(&firstpos) &&
                                  framebuffer<=
                                     cd_time_to_frame(&lastpos)  &&
                                  framebuffer<locatorBframe      &&
                                  (locatorBframe)-framebuffer >=MIN_LOOP)
                               {  frame_to_cd_time(framebuffer,&locatorA);
                                  locatorAframe=cd_time_to_frame(&locatorA);
                                  if(loop_status==LOOP_GAP)
                                  {  loop_status=LOOP_AB;
                                  }
				  if(loop_status==LOOP_BEGIN)
				  {  cd_play(&locatorA,&lastpos);
				  }
				  else
                                  {  cd_play(&locatorA,&locatorB);
				  }
                                  shm->change|=LOCATOR_A_CHANGE;
                               }
                            }
                            shm->command=READY;
                            break;
                               
         case CMD_MOVE_LOCATOR_B:
                            if(status==STATUS_PLAY && 
			        (loop_status==LOOP_AB|| loop_status==LOOP_GAP))
                            {  framebuffer=cd_time_to_frame(&locatorB);
                               framebuffer+=shm->offset_buffer;
                               if(framebuffer>=
                                     cd_time_to_frame(&firstpos) &&
                                  framebuffer<=
                                     cd_time_to_frame(&lastpos)  &&
                                  framebuffer>locatorAframe      &&
                                  (framebuffer-locatorAframe)>=MIN_LOOP)   
                               {  frame_to_cd_time(framebuffer,&locatorB);
                                  locatorBframe=cd_time_to_frame(&locatorB);
                                  cd_play(&locatorA,&locatorB);
                                  if(loop_status==LOOP_GAP)
                                  {  loop_status=LOOP_AB;
                                  }
                                  shm->change|=LOCATOR_B_CHANGE;
                               }
                            }
                            shm->command=READY;
                            break;

         case CMD_SET_GAP:  if(status==STATUS_PLAY &&
	                       (loop_status==LOOP_AB || loop_status==LOOP_GAP))
			    {  if(LONG_MAX/GAP_FACTOR>shm->gap_buffer)
			       {  if(shm->gap_buffer<0)
			          {  gap=0L;
				  }
				  else
				  {  gap=shm->gap_buffer*GAP_FACTOR;
				  }
				  if(loop_status==LOOP_GAP)
				  {  loop_status=LOOP_AB;
			             cd_play(&locatorA,&locatorB);
				  }                  
			          shm->change|=GAP_CHANGE;
			       }
			    }
                            shm->command=READY;
                            break;

         case CMD_GET_GAP:  if(status!=STATUS_PLAY ||
	                       (loop_status!=LOOP_AB && loop_status!=LOOP_GAP))
			    {  shm->gap_buffer=-1;
			    }
			    else
			    {  shm->gap_buffer=gap/GAP_FACTOR;
			    }
			    shm->change&=~GAP_CHANGE;
			    shm->command=READY;
			    break;

      }

      if(old_status!=status || old_loop_status!=loop_status)
      {  shm->change|=STATUS_CHANGE;
      }
   }
   cd_stop();
   _exit(0);
}


/*========================================================================
  - exported functions
  ========================================================================*/

void cdplayer_init(char *device)
{  shmid=shmget(IPC_PRIVATE,sizeof(struct shmblock),IPC_CREAT | 0700);
   if(shmid==-1)
   {  ERROR("Coudn't get shared memory");
   }
   shmctl(shmid,SHM_LOCK,NULL);

   shm=(struct shmblock *)shmat(shmid,NULL,0);
   shm->command=INIT;
   shm->change=0;

   if(cd_init(device))
   {  ERROR("Couldn't open %s",device);
   }

   if(!(cd_player_control_process=fork()))
   {  cd_player_control();
   }
}


void cdplayer_exit(void)
{  int dummy;

   set_simple_command(CMD_EXIT);
   wait(&dummy);
   cd_exit();
   shmdt((char *)shm);
   shmctl(shmid,IPC_RMID,NULL);
}


int cdplayer_ask_changes(void)
{  return shm->change;
}                    

                    
void cdplayer_get_status(int *st,int *loop_st)
{  set_simple_command(CMD_GET_STATUS);
   while(shm->command!=READY)
   { }
   *st=shm->status_buffer;
   *loop_st=shm->loop_status_buffer;
}


int cdplayer_get_n_track(void)
{  set_simple_command(CMD_GET_N_TRACK);
   while(shm->command!=READY)
   { }   
   return shm->track_buffer;
}


void cdplayer_get_pos(int *track,int *m,int *s,int *f)
{  set_simple_command(CMD_GET_POS);
   while(shm->command!=READY)
   { }
   *track=shm->track_buffer;
   *m=shm->pos_buffer.m;   
   *s=shm->pos_buffer.s;   
   *f=shm->pos_buffer.f;
}


void cdplayer_play(void)
{   set_simple_command(CMD_PLAY);
}


void cdplayer_play_at(int track)
{  while(shm->command!=READY)
   { }
   shm->track_buffer=track;
   shm->command=CMD_PLAY_AT;
}


void cdplayer_play_prev(void)
{   set_simple_command(CMD_PLAY_PREV);
}


void cdplayer_play_next(void)
{   set_simple_command(CMD_PLAY_NEXT);
}


void cdplayer_restart(void)
{   set_simple_command(CMD_RESTART);
}


void cdplayer_pause(void)
{   set_simple_command(CMD_PAUSE);
}


void cdplayer_stop(void)
{   set_simple_command(CMD_STOP); 
}


void cdplayer_search(long offset)
{  set_command_with_offset(CMD_SEARCH,offset);
}


void cdplayer_eject(void)
{   set_simple_command(CMD_EJECT);
}


void cdplayer_close_tray(void)
{   set_simple_command(CMD_CLOSE_TRAY);
}


void cdplayer_set_locator_a(void)
{  set_simple_command(CMD_SET_LOCATOR_A);
}


void cdplayer_set_locator_b(void)
{  set_simple_command(CMD_SET_LOCATOR_B);
}


void cdplayer_get_locator_a(int *track,int *m,int *s,int *f)
{  set_command_get_pos(CMD_GET_LOCATOR_A,track,m,s,f);
}


void cdplayer_get_locator_b(int *track,int *m,int *s,int *f)
{  set_command_get_pos(CMD_GET_LOCATOR_B,track,m,s,f);
}


void cdplayer_clear_locator_b(void)
{  set_simple_command(CMD_CLEAR_LOCATOR_B);
}


void cdplayer_clear_locators(void)
{  set_simple_command(CMD_CLEAR_LOCATORS);
}


void cdplayer_move_locator_a(long offset)
{  set_command_with_offset(CMD_MOVE_LOCATOR_A,offset);
}


void cdplayer_move_locator_b(long offset)
{  set_command_with_offset(CMD_MOVE_LOCATOR_B,offset);
}


void cdplayer_set_gap(long gap)
{  while(shm->command!=READY)
   { }
   shm->gap_buffer=gap;
   shm->command=CMD_SET_GAP;
}


long cdplayer_get_gap(void)
{  set_simple_command(CMD_GET_GAP);
   while(shm->command!=READY)
   { }   
   return shm->gap_buffer;
}
