/*
    This file is part of CDLoop
    Copyright (C) 1997, 1998, 1999  Claus Brunzema (chb@ossi.fho-emden.de)

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  -
  -  cdloop.c
  -
  -  main application (an extended Tcl/Tk-shell with a built-in script).
  -
  - $Id: cdloop.c,v 1.1.1.1 1999/06/04 13:05:41 chb Exp $
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <tcl.h>
#include <tk.h>

#include "cdplayer.h"
#include "tclscript.h"


#define TclRegister(name,proc)  \
                    Tcl_CreateCommand(interp,#name, \
                    (Tcl_CmdProc *)proc,(ClientData)NULL, \
                    (Tcl_CmdDeleteProc *)NULL)

#define POS_FORMAT "%02d-%02d:%02d:%02d"  /* for displaying positions */
#define NO_POSITION "-----------"         /* when there is no position ... */

#define TICKER_MS   100                   /* ticker interval 100ms */
#define TICKER_RUN  1                     /* ticker is running */
#define TICKER_STOP 2                     /* ticker is stopped */
#define TICKER_DONE 3                     /* ticker is shut down */


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - variables global to this module 
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
Tcl_Interp *theTCL;                       /* our Tcl-interpreter */
volatile int ticker_status;               /* the status of the ticker */


/*========================================================================
  - static functions
  ========================================================================*/

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - void tickerProc(ClientData clientData)
  -  This is the routine which gets called by the Tcl-interpreter about
  -  every 100ms (defined in the constant TICKER_MS). This is done via
  -  Tcl_CreateTimerHandler().
  -  in:  ClientData clientData     - dummy argument
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
static void tickerProc(ClientData clientData)
{  static char c[2]={ 0,0 };

   if(ticker_status==TICKER_RUN) /* are we running ? */
   {  /* change the variable */
      c[0]++;
      /* send the variable to Tcl, so tkwait can return */
      Tcl_SetVar(theTCL,"Ticker",c,TCL_GLOBAL_ONLY);
      /* register ourself to be called again */
      Tcl_CreateTimerHandler(TICKER_MS,(Tcl_TimerProc *)tickerProc,NULL);
   }
   else
   {  if(ticker_status==TICKER_STOP)  /* we must stop */
      {  ticker_status=TICKER_DONE;
      }
   }
}


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - Tk_Window get_win(Tcl_Interp *interp,char *name)
  -  returns the window with the name name.
  -  in:  Tcl_Interp *interp        - the Tcl-interpreter to ask.
  -       char *name                - the name of the window.
  -  out: Tk_Window                 - the window, NULL if error.
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
static Tk_Window get_win(Tcl_Interp *interp,char *name)
{  Tk_Window mainwin;
   
   mainwin=Tk_MainWindow(interp);
   if(!mainwin)
   {  return NULL;
   }
   return Tk_NameToWindow(interp,name,mainwin);
}


/*========================================================================
  - functions corresponding to Tcl-commands
  ========================================================================*/

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - int XWinInteriorWidthProc(ClientData clientdata,Tcl_Interp *interp,
  -                           int argc,char *argv[])
  -  returns the window width of the window named in argv[1].
  -  For documentation on arguments and return code see Tcl-documentation.
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int XWinInteriorWidthProc(ClientData clientdata,Tcl_Interp *interp,
                          int argc,char *argv[])
{  Tk_Window win;
   
   if(argc!=2)
   {  return TCL_ERROR;
   }
   win=get_win(interp,argv[1]);
   if(!win)
   {  return TCL_ERROR;
   }
   sprintf(interp->result,"%d",Tk_Width(win));
   return TCL_OK;
}


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - int XWinInteriorHeightProc(ClientData clientdata,Tcl_Interp *interp,
  -                            int argc,char *argv[])
  -  returns the window height of the window named in argv[1].
  -  For documentation on arguments and return code see Tcl-documentation.
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int XWinInteriorHeightProc(ClientData clientdata,Tcl_Interp *interp,
                           int argc,char *argv[])
{  Tk_Window win;
   
   if(argc!=2)
   {  return TCL_ERROR;
   }
   win=get_win(interp,argv[1]);
   if(!win)
   {  return TCL_ERROR;
   }
   sprintf(interp->result,"%d",Tk_Height(win));
   return TCL_OK;
}


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - int CD_initProc(ClientData clientdata,Tcl_Interp *interp,
  -                 int argc,char *argv[])
  -  initializes the cdrom-drive and starts the ticker.
  -  For documentation on arguments and return code see Tcl-documentation.
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int CD_initProc(ClientData clientdata,Tcl_Interp *interp,int argc,char *argv[])
{
    if(argc!=2)
    {  return TCL_ERROR;
    }
    cdplayer_init(argv[1]);
    ticker_status=TICKER_RUN;
    Tcl_CreateTimerHandler(TICKER_MS,(Tcl_TimerProc *)tickerProc,NULL);
    return TCL_OK;
}


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - int CD_exitProc(ClientData clientdata,Tcl_Interp *interp,
  -                 int argc,char *argv[])
  -  shuts down the ticker and releases the cdrom-drive.
  -  For documentation on arguments and return code see Tcl-documentation.
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int CD_exitProc(ClientData clientdata,Tcl_Interp *interp,int argc,char *argv[])
{  ticker_status=TICKER_STOP;
   while(ticker_status!=TICKER_DONE)
   {  Tk_DoOneEvent(TK_ALL_EVENTS);
   }
   cdplayer_exit();
   return TCL_OK;
}


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - int CD_ask_changesProc(ClientData clientdata,Tcl_Interp *interp,
  -                        int argc,char *argv[])
  -  polls changes from the cdplayer.
  -  For documentation on arguments and return code see Tcl-documentation.
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int CD_ask_changesProc(ClientData clientdata,Tcl_Interp *interp,
                       int argc,char *argv[])
{  sprintf(interp->result,"%d",cdplayer_ask_changes());
   return TCL_OK;
}


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - int CD_get_statusProc(ClientData clientdata,Tcl_Interp *interp,
  -                       int argc,char *argv[])
  -  gets status from the cdplayer.
  -  For documentation on arguments and return code see Tcl-documentation.
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int CD_get_statusProc(ClientData clientdata,Tcl_Interp *interp,
                       int argc,char *argv[])
{  int status,loop_status;

   cdplayer_get_status(&status,&loop_status);
   switch(status)
   {  case STATUS_NO_TOC: sprintf(interp->result,"NO DISC");
                          break;
      case STATUS_STOP:   sprintf(interp->result,"STOP");
                          break;
      case STATUS_DATA:   sprintf(interp->result,"DATA");
                          break;
      case STATUS_PLAY:   switch(loop_status)
                          {  case LOOP_NONE:
			     case LOOP_BEGIN: sprintf(interp->result,"PLAY");
			                      break;
                             case LOOP_AB:    sprintf(interp->result,"LOOP");
			                      break;
			     case LOOP_GAP:   sprintf(interp->result,"GAP");
			                      break;
                          }
                          break;
      case STATUS_PAUSE:  sprintf(interp->result,"PAUSE");
                          break;
   }
   return TCL_OK;
}


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - int CD_get_n_trackProc(ClientData clientdata,Tcl_Interp *interp,
  -                        int argc,char *argv[])
  -  gets the number of tracks from the cdplayer.
  -  For documentation on arguments and return code see Tcl-documentation.
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int CD_get_n_trackProc(ClientData clientdata,Tcl_Interp *interp,
                       int argc,char *argv[])
{  sprintf(interp->result,"%d",cdplayer_get_n_track());
   return TCL_OK;
}


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - int CD_get_posProc(ClientData clientdata,Tcl_Interp *interp,
  -                    int argc,char *argv[])
  -  gets the position from the cdplayer.
  -  For documentation on arguments and return code see Tcl-documentation.
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int CD_get_posProc(ClientData clientdata,Tcl_Interp *interp,int argc,
                   char *argv[])
{  int track,m,s,f;  

   cdplayer_get_pos(&track,&m,&s,&f);
   if(track>0)
   {  sprintf(interp->result,POS_FORMAT,track,m,s,f);
   }
   else
   {  sprintf(interp->result,NO_POSITION);
   }
   return TCL_OK;
}


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - int CD_playProc(ClientData clientdata,Tcl_Interp *interp,
  -                 int argc,char *argv[])
  -  starts playing.
  -  For documentation on arguments and return code see Tcl-documentation.
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int CD_playProc(ClientData clientdata,Tcl_Interp *interp,int argc,char *argv[])
{  cdplayer_play();
   return TCL_OK;
}

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - int CD_play_atProc(ClientData clientdata,Tcl_Interp *interp,
  -                    int argc,char *argv[])
  -  starts playing at the track in argv[1].
  -  For documentation on arguments and return code see Tcl-documentation.
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int CD_play_atProc(ClientData clientdata,Tcl_Interp *interp,
                   int argc,char *argv[])
{  int error,nr;

   if(argc!=2)
   {  return TCL_ERROR;
   }
   error=Tcl_GetInt(interp,argv[1],&nr);
   if(error!=TCL_OK)
   {  return error;
   }
   cdplayer_play_at(nr);
   return TCL_OK;
}


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - int CD_play_prevProc(ClientData clientdata,Tcl_Interp *interp,
  -                    int argc,char *argv[])
  -  starts playing the previous track.
  -  For documentation on arguments and return code see Tcl-documentation.
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int CD_play_prevProc(ClientData clientdata,Tcl_Interp *interp,int argc,
                     char *argv[])
{  cdplayer_play_prev();
   return TCL_OK;
}


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - int CD_play_nextProc(ClientData clientdata,Tcl_Interp *interp,
  -                    int argc,char *argv[])
  -  starts playing the next track.
  -  For documentation on arguments and return code see Tcl-documentation.
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int CD_play_nextProc(ClientData clientdata,Tcl_Interp *interp,int argc,
                     char *argv[])
{  cdplayer_play_next();
   return TCL_OK;
}


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - int CD_restartProc(ClientData clientdata,Tcl_Interp *interp,
  -                    int argc,char *argv[])
  -  restarts playing (when in pause-mode).
  -  For documentation on arguments and return code see Tcl-documentation.
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int CD_restartProc(ClientData clientdata,Tcl_Interp *interp,int argc,
                   char *argv[])
{  cdplayer_restart();
   return TCL_OK;
}


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - int CD_pauseProc(ClientData clientdata,Tcl_Interp *interp,
  -                  int argc,char *argv[])
  -  pauses playing.
  -  For documentation on arguments and return code see Tcl-documentation.
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int CD_pauseProc(ClientData clientdata,Tcl_Interp *interp,int argc,
                 char *argv[])
{  cdplayer_pause();
   return TCL_OK;
}


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - int CD_stopProc(ClientData clientdata,Tcl_Interp *interp,
  -                 int argc,char *argv[])
  -  stops playing.
  -  For documentation on arguments and return code see Tcl-documentation.
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int CD_stopProc(ClientData clientdata,Tcl_Interp *interp,int argc,char *argv[])
{  cdplayer_stop();
   return TCL_OK;
}


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - int CD_searchProc(ClientData clientdata,Tcl_Interp *interp,
  -                   int argc,char *argv[])
  -  adds the (positive or negative) offset in argv[1] to the play
  -  position.
  -  For documentation on arguments and return code see Tcl-documentation.
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int CD_searchProc(ClientData clientdata,Tcl_Interp *interp,
                  int argc,char *argv[])
{  int error,offset;

   if(argc!=2)
   {  return TCL_ERROR;
   }
   error=Tcl_GetInt(interp,argv[1],&offset);
   if(error!=TCL_OK)
   {  return error;
   }
   cdplayer_search(offset);
   return TCL_OK;
}


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - int CD_ejectProc(ClientData clientdata,Tcl_Interp *interp,
  -                  int argc,char *argv[])
  -  ejects the cd.
  -  For documentation on arguments and return code see Tcl-documentation.
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int CD_ejectProc(ClientData clientdata,Tcl_Interp *interp,int argc,
                 char *argv[])
{  cdplayer_eject();
   return TCL_OK;
}


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - int CD_close_trayProc(ClientData clientdata,Tcl_Interp *interp,
  -                       int argc,char *argv[])
  -  closes the tray of the cdrom-drive.
  -  For documentation on arguments and return code see Tcl-documentation.
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int CD_close_trayProc(ClientData clientdata,Tcl_Interp *interp,int argc,
                      char *argv[])
{  cdplayer_close_tray();
   return TCL_OK;
}


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - int CD_set_locator_aProc(ClientData clientdata,Tcl_Interp *interp,
  -                          int argc,char *argv[])
  -  sets locator a at the current position.
  -  For documentation on arguments and return code see Tcl-documentation.
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int CD_set_locator_aProc(ClientData clientdata,Tcl_Interp *interp,int argc,
                         char *argv[])
{  cdplayer_set_locator_a();
   return TCL_OK;
}


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - int CD_set_locator_bProc(ClientData clientdata,Tcl_Interp *interp,
  -                          int argc,char *argv[])
  -  sets locator b at the current position.
  -  For documentation on arguments and return code see Tcl-documentation.
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int CD_set_locator_bProc(ClientData clientdata,Tcl_Interp *interp,int argc,
                         char *argv[])
{  cdplayer_set_locator_b();
   return TCL_OK;
}


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - int CD_get_locator_aProc(ClientData clientdata,Tcl_Interp *interp,
  -                          int argc,char *argv[])
  -  gets position of locator a.
  -  For documentation on arguments and return code see Tcl-documentation.
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int CD_get_locator_aProc(ClientData clientdata,Tcl_Interp *interp,
                         int argc,char *argv[])
{  int track,m,s,f;

   cdplayer_get_locator_a(&track,&m,&s,&f);
   if(track>0)
   {  sprintf(interp->result,POS_FORMAT,track,m,s,f);
   }
   else
   {  sprintf(interp->result,NO_POSITION);
   }
   return TCL_OK;
}


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - int CD_get_locator_bProc(ClientData clientdata,Tcl_Interp *interp,
  -                          int argc,char *argv[])
  -  gets position of locator b.
  -  For documentation on arguments and return code see Tcl-documentation.
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int CD_get_locator_bProc(ClientData clientdata,Tcl_Interp *interp,
                         int argc,char *argv[])
{  int track,m,s,f;

   cdplayer_get_locator_b(&track,&m,&s,&f);
   if(track>0)
   {  sprintf(interp->result,POS_FORMAT,track,m,s,f);
   }
   else
   {  sprintf(interp->result,NO_POSITION);
   }
   return TCL_OK;
}


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - int CD_clear_locator_bProc(ClientData clientdata,Tcl_Interp *interp,
  -                            int argc,char *argv[])
  -  clears locator b.
  -  For documentation on arguments and return code see Tcl-documentation.
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int CD_clear_locator_bProc(ClientData clientdata,Tcl_Interp *interp,
                           int argc,char *argv[])
{  cdplayer_clear_locator_b();
   return TCL_OK;
}


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - int CD_clear_locatorsProc(ClientData clientdata,Tcl_Interp *interp,
  -                            int argc,char *argv[])
  -  clears both locators.
  -  For documentation on arguments and return code see Tcl-documentation.
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int CD_clear_locatorsProc(ClientData clientdata,Tcl_Interp *interp,
                          int argc,char *argv[])
{  cdplayer_clear_locators();
   return TCL_OK;
}


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - int CD_move_locator_aProc(ClientData clientdata,Tcl_Interp *interp,
  -                           int argc,char *argv[])
  -  moves locator a according to the (positive or negative) offset in
  -  argv[1].
  -  For documentation on arguments and return code see Tcl-documentation.
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int CD_move_locator_aProc(ClientData clientdata,Tcl_Interp *interp,
                          int argc,char *argv[])
{  int error,offset;

   if(argc!=2)
   {  return TCL_ERROR;
   }
   error=Tcl_GetInt(interp,argv[1],&offset);
   if(error!=TCL_OK)
   {  return error;
   }
   cdplayer_move_locator_a(offset);
   return TCL_OK;
}


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - int CD_move_locator_bProc(ClientData clientdata,Tcl_Interp *interp,
  -                           int argc,char *argv[])
  -  moves locator b according to the (positive or negative) offset in
  -  argv[1].
  -  For documentation on arguments and return code see Tcl-documentation.
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int CD_move_locator_bProc(ClientData clientdata,Tcl_Interp *interp,
                          int argc,char *argv[])
{  int error,offset;

   if(argc!=2)
   {  return TCL_ERROR;
   }
   error=Tcl_GetInt(interp,argv[1],&offset);
   if(error!=TCL_OK)
   {  return error;
   }
   cdplayer_move_locator_b(offset);
   return TCL_OK;
}


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - int CD_set_gapProc(ClientData clientdata,Tcl_Interp *interp,
  -                    int argc,char *argv[])
  -  sets the gap to the value in argv[1].
  -  For documentation on arguments and return code see Tcl-documentation.
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int CD_set_gapProc(ClientData clientdata,Tcl_Interp *interp,
                          int argc,char *argv[])
{  int error,gap;

   if(argc!=2)
   {  return TCL_ERROR;
   }
   error=Tcl_GetInt(interp,argv[1],&gap);
   if(error!=TCL_OK)
   {  return error;
   }
   cdplayer_set_gap(gap);
   return TCL_OK;
}


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - int CD_get_gapProc(ClientData clientdata,Tcl_Interp *interp,
  -                    int argc,char *argv[])
  -  gets the value of the gap.
  -  For documentation on arguments and return code see Tcl-documentation.
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int CD_get_gapProc(ClientData clientdata,Tcl_Interp *interp,
                       int argc,char *argv[])
{  int gap=cdplayer_get_gap();

   if(gap>0)
   {  sprintf(interp->result,"%d",gap);
   }
   else
   {  sprintf(interp->result,"----");
   }
   return TCL_OK;
}


/*========================================================================
  - application init
  -  see tkAppInit.c from the Tcl-package, too
  ========================================================================*/
int Tcl_AppInit(Tcl_Interp *interp)
{   if(Tcl_Init(interp)==TCL_ERROR)
    {  return TCL_ERROR;
    }
    if(Tk_Init(interp)==TCL_ERROR)
    {  return TCL_ERROR;
    }
    Tcl_StaticPackage(interp,"Tk",Tk_Init,(Tcl_PackageInitProc *) NULL);

    /* register commands */
    TclRegister(XWinInteriorWidth,XWinInteriorWidthProc);
    TclRegister(XWinInteriorHeight,XWinInteriorHeightProc);

    TclRegister(CD_init,CD_initProc); 
    TclRegister(CD_exit,CD_exitProc);
    TclRegister(CD_ask_changes,CD_ask_changesProc);
    TclRegister(CD_get_status,CD_get_statusProc);
    TclRegister(CD_get_n_track,CD_get_n_trackProc);
    TclRegister(CD_get_pos,CD_get_posProc);
    TclRegister(CD_play,CD_playProc); 
    TclRegister(CD_play_at,CD_play_atProc);
    TclRegister(CD_play_prev,CD_play_prevProc);
    TclRegister(CD_play_next,CD_play_nextProc);
    TclRegister(CD_restart,CD_restartProc);
    TclRegister(CD_pause,CD_pauseProc);
    TclRegister(CD_stop,CD_stopProc); 
    TclRegister(CD_search,CD_searchProc); 
    TclRegister(CD_eject,CD_ejectProc);
    TclRegister(CD_close_tray,CD_close_trayProc);
    TclRegister(CD_set_locator_a,CD_set_locator_aProc);
    TclRegister(CD_set_locator_b,CD_set_locator_bProc);
    TclRegister(CD_get_locator_a,CD_get_locator_aProc);
    TclRegister(CD_get_locator_b,CD_get_locator_bProc);
    TclRegister(CD_clear_locator_b,CD_clear_locator_bProc);
    TclRegister(CD_clear_locators,CD_clear_locatorsProc);
    TclRegister(CD_move_locator_a,CD_move_locator_aProc);
    TclRegister(CD_move_locator_b,CD_move_locator_bProc);
    TclRegister(CD_set_gap,CD_set_gapProc);
    TclRegister(CD_get_gap,CD_get_gapProc);

    /* memorize our interpreter */
    theTCL=interp;
    
    /* start the Tcl-script */
    return Tcl_VarEval(interp,tclscript,NULL);
}


/*========================================================================
  - main program
  ========================================================================*/
int main(int argc,char *argv[])
{
    Tk_Main(argc, argv, Tcl_AppInit);
    return 0;
}
