/*
    This file is part of CDLoop
    Copyright (C) 1997, 1998, 1999  Claus Brunzema (chb@ossi.fho-emden.de)

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  -
  -  cdhw.h
  -
  -  low level routines dealing with /dev/cdrom etc.
  -  Headerfile for cdhw.c
  -
  - $Id: cdhw.h,v 1.1.1.1 1999/06/04 13:05:41 chb Exp $
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/


#ifndef CDHW_H_INCLUDED
#define CDHW_H_INCLUDED


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  codes for cd_status.status
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#define CD_PLAY     1  /* cd is playing */
#define CD_PAUSE    2  /* cd is in pause mode */
#define CD_COMPLETE 3  /* the play command has been completed */
#define CD_NOSTATUS 4  /* no status available (no disc etc.) */

typedef unsigned long int ULONG; 

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - structure representing a position on the cd
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
typedef struct 
{  int m,s,f;             /* minutes,seconds,frames */
}cd_time;


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - structure representing a track
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
typedef struct 
{  cd_time length;
   cd_time start;
   cd_time end;
}track_info;


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - structure representing the toc (table of contents)
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
extern struct _cd_toc_ 
{  int        n_track;
   cd_time    length;
   track_info *track;
}cd_toc;


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - structure containing statusinfomation from the cd-drive
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
extern struct _cd_status_
{  int status;
   int track;
   cd_time pos_abs;
   cd_time pos_rel;
}cd_status;


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - void add_to_cd_time(cd_time *t,long f)
  -  adds f frames to a cd_time structure.
  -  in:  cd_time *t            - pointer to the cd_time struct. The result
  -                               is put into this struct.
  -       long f                - the frames to be added to t. f may be
  -                               negative.
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
void add_to_cd_time(cd_time *t,long f);


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - ULONG cd_time_to_frame(cd_time *t)
  -  converts a cd_time structure into an ULONG (the frame number of that
  -  position).
  -  in:  cd_time *t            - pointer to the cd_time struct. The result
  -                               is put into this struct.
  -  out: ULONG                 - the frame number.
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
ULONG cd_time_to_frame(cd_time *t);


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - void frame_to_cd_time(ULONG f,cd_time *t)
  -  converts a frame number into a cd_time structure.
  -  in:  long f                - the framenumber
  -       cd_time *t            - pointer to the cd_time struct. The result
  -                               is put into this struct.
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
void frame_to_cd_time(ULONG f,cd_time *t);


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - int cd_abs_to_rel(cd_time *abs,int *track,cd_time *rel)
  -  converts an absolute position on the cd to a position relative to the
  -  beginning of a certain track.
  -  in:  cd_time *abs          - pointer to the cd_time struct containing
  -                               the absolute position.
  -       int *track            - pointer to the int, where the track number
  -                               of the relative position will be stored.
  -       cd_time *rel          - pointer to the cd_time struct where the
  -                               relative position will be stored.
  -  out: int                   - 0 if ok, !=0 if error.
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int cd_abs_to_rel(cd_time *abs,int *track,cd_time *rel);


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - int cd_init(char *device_name)
  -  initializes the cdrom-drive. Has to be called before any call to the
  -  following routines.
  -  in:  char *device_name     - name of the cd-device (/dev/cdrom)
  -  out: int                   - 0 if ok, !=0 if error
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int cd_init(char *device_name);


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - void cd_exit(void)
  -  cleans up. Should be called before the application ends.
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
void cd_exit(void);


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - int cd_read_status(void)
  -  reads the status of the cdrom-drive. The status is stored in cd_status.
  -  out: int                   - 0 if ok, !=0 if error.
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int cd_read_status(void);


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - int cd_read_toc(void)
  -  reads the toc of the cd. The toc is stored in cd_toc.
  -  out: int                   - 0 if ok, !=0 if error.
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int cd_read_toc(void);


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - int cd_play(cd_time *first,cd_time *last)
  -  plays the cd from first to last.
  -  in:  cd_time *first        - pointer to a cd_time struct which
  -                               contains the start position for playing.
  -       cd_time *last         - pointer to a cd_time struct which
  -                               contains the stop position for playing.
  -  out: int                   - 0 if ok, !=0 if error.
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int cd_play(cd_time *first,cd_time *last);


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - int cd_pause(void)
  -  pauses playing.
  -  out: int                   - 0 if ok, !=0 if error.
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int cd_pause(void);


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - int cd_resume(void)
  -  resumes playing when in pause-mode.
  -  out: int                   - 0 if ok, !=0 if error.
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int cd_resume(void);


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - int cd_stop(void)
  -  stops playing.
  -  out: int                   - 0 if ok, !=0 if error.
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int cd_stop(void);


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - int cd_eject(void)
  -  ejects the cd.
  -  out: int                   - 0 if ok, !=0 if error.
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int cd_eject(void);


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - int cd_close_tray(void)
  -  closes the tray of the cdrom-drive.
  -  out: int                   - 0 if ok, !=0 if error.
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int cd_close_tray(void);

#endif
