/*
    This file is part of CDLoop
    Copyright (C) 1997, 1998, 1999  Claus Brunzema (chb@ossi.fho-emden.de)

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  -
  -  cdhw.c
  -
  -  low level routines dealing with /dev/cdrom etc.
  -
  - $Id: cdhw.c,v 1.1.1.1 1999/06/04 13:05:41 chb Exp $
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/

#include "cdhw.h"

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <sys/types.h>  
#include <fcntl.h>  
#include <mntent.h>
#include <linux/cdrom.h>
#include <string.h>
#include <errno.h>
     

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - variables global to this module 
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
struct _cd_toc_ cd_toc={ -1,{0,0,0},NULL};
struct _cd_status_ cd_status={CD_NOSTATUS,-1,{0,0,0},{0,0,0}};
int cd_fd;


/*========================================================================
  - static functions
  ========================================================================*/

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - void free_toc(void)
  -  frees memory occupied by the current toc.
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
static void free_toc(void)
{  if(cd_toc.n_track>0)
   {  free(cd_toc.track);
      cd_toc.track=NULL;
   }
   cd_toc.n_track=-1;
}


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  - void msf0_to_cd_time(struct cdrom_msf0 *m,cd_time *t)
  -  converts a cdrom_msf0 structure into a cd_time structure.
  -  in:  struct cdrom_msf0 *m  - pointer to the cdrom_msf0 struct
  -       cd_time *t            - pointer to the cd_time struct. The result
  -                               is put into this struct.
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
static void msf0_to_cd_time(struct cdrom_msf0 *m,cd_time *t)
{  t->m=m->minute;
   t->s=m->second;
   t->f=m->frame;
}


/*========================================================================
  - exported functions
  ========================================================================*/

void add_to_cd_time(cd_time *t,long f)
{  ULONG frame;

   frame=cd_time_to_frame(t);
   frame+=f;
   frame_to_cd_time(frame,t);
}


ULONG cd_time_to_frame(cd_time *t)
{  return t->f+t->s*75+t->m*60*75;
}


void frame_to_cd_time(ULONG f,cd_time *t)
{  ULONG r;

   t->m=f/(60*75);
   r=f%(60*75);
   t->s=r/75;
   t->f=r%75;
}


int cd_abs_to_rel(cd_time *abs,int *track,cd_time *rel)
{  int i;
   ULONG absframe=cd_time_to_frame(abs);
   ULONG startframe;
   
   if(cd_toc.n_track<1)
   {  return 1;
   }
   for(i=0;i<cd_toc.n_track;i++)
   {  startframe=cd_time_to_frame(&cd_toc.track[i].start);
      if(startframe>=absframe)
      {  break;
      }
   }
   *track=i-1;
   frame_to_cd_time(absframe-cd_time_to_frame(&cd_toc.track[i-1].start),rel);
   return 0;
}   


int cd_init(char *device_name)
{  struct mntent *mnt;
   FILE *fp;
   
   cd_toc.n_track=-1;
   cd_toc.track=NULL;
   if((fp = setmntent (MOUNTED, "r")) == NULL)
   {  return 1;
   }
   while((mnt = getmntent (fp)) != NULL)
   {  if(strcmp(mnt->mnt_type, "iso9660") == 0)
      {  endmntent(fp);
         return 1;
      }
   }
   endmntent(fp);
   if((cd_fd=open(device_name,O_RDONLY | O_NONBLOCK))<0)
   {  return 1;
   }
   return 0;
}


void cd_exit(void)
{  if(cd_fd>=0)
   {  close(cd_fd);
   } 
   free_toc();
}


int cd_read_status(void)
{  struct cdrom_subchnl sc;

   sc.cdsc_format=CDROM_MSF;
   if(ioctl(cd_fd,CDROMSUBCHNL,&sc))
   {  return 1;
   }

   switch(sc.cdsc_audiostatus)
   {  case CDROM_AUDIO_PLAY:      cd_status.status=CD_PLAY;
                                  break;
      case CDROM_AUDIO_PAUSED:    cd_status.status=CD_PAUSE;
                                  break;
      case CDROM_AUDIO_COMPLETED: cd_status.status=CD_COMPLETE;
                                  break;
      default:                    cd_status.status=CD_NOSTATUS;
   }
   cd_status.track=sc.cdsc_trk-1;
   msf0_to_cd_time(&sc.cdsc_absaddr.msf,&cd_status.pos_abs);
   msf0_to_cd_time(&sc.cdsc_reladdr.msf,&cd_status.pos_rel);
   return 0;
}


int cd_read_toc(void)
{  struct cdrom_tochdr h;
   struct cdrom_tocentry e;
   cd_time t={0,0,0};
   int i,n_track,track_nr;

   free_toc();
   if(ioctl(cd_fd,CDROMREADTOCHDR,&h))
   {  return 1;
   }
   n_track=h.cdth_trk1;
   if(n_track<1)
   {  return 1;
   }
   cd_toc.track=(track_info *)malloc(n_track*sizeof(track_info));
   if(!cd_toc.track)
   {  return 1;
   }
   
   track_nr=0;
   for(i=0;i<n_track;i++)
   {  e.cdte_track=i+1;
      e.cdte_format=CDROM_MSF;
      if(ioctl(cd_fd,CDROMREADTOCENTRY,&e))
      {  return 1;
      }
      msf0_to_cd_time(&e.cdte_addr.msf,&t);
      if(i>0)
      {  if(cd_toc.track[track_nr-1].end.m==-1)
         {  /* compute length of previous track */
	    cd_toc.track[track_nr-1].end=t;
            frame_to_cd_time(cd_time_to_frame(&cd_toc.track[track_nr-1].end)-
                               cd_time_to_frame(&cd_toc.track[track_nr-1].start),
                             &cd_toc.track[track_nr-1].length);
            add_to_cd_time(&cd_toc.track[track_nr-1].end,-1L);
         }   
      }
      if(e.cdte_ctrl & CDROM_DATA_TRACK)
      {  continue;
      }
      cd_toc.track[track_nr].start=t;
      cd_toc.track[track_nr].end.m=-1; /* mark track.end invalid */
      track_nr++;
   }
   e.cdte_track=CDROM_LEADOUT;
   e.cdte_format=CDROM_MSF;
   if(ioctl(cd_fd,CDROMREADTOCENTRY,&e))
   {  return 1;
   }
   msf0_to_cd_time(&e.cdte_addr.msf,&t);
   if(cd_toc.track[track_nr-1].end.m==-1)
   {  /* compute length of the last track */
      cd_toc.track[track_nr-1].end=t;
      frame_to_cd_time(cd_time_to_frame(&cd_toc.track[track_nr-1].end)-
                        cd_time_to_frame(&cd_toc.track[track_nr-1].start),
                       &cd_toc.track[track_nr-1].length);   
      add_to_cd_time(&cd_toc.track[track_nr-1].end,-1L);
   }
   cd_toc.n_track=track_nr;
   
   /* compute overall length */
   frame_to_cd_time(cd_time_to_frame(&cd_toc.track[cd_toc.n_track-1].end)-
                      cd_time_to_frame(&cd_toc.track[0].start),
                    &cd_toc.length);
   return 0;
}


int cd_play(cd_time *first,cd_time *last)
{  struct cdrom_msf msf;

   msf.cdmsf_min0=first->m;
   msf.cdmsf_sec0=first->s;
   msf.cdmsf_frame0=first->f;
   msf.cdmsf_min1=last->m;
   msf.cdmsf_sec1=last->s;
   msf.cdmsf_frame1=last->f;
   if(ioctl(cd_fd,CDROMSTART))
   {  return 1;
   }
   return ioctl(cd_fd,CDROMPLAYMSF,&msf);
}


int cd_pause(void)
{  return ioctl(cd_fd,CDROMPAUSE);
}


int cd_resume(void)
{  return ioctl(cd_fd,CDROMRESUME);
}

 
int cd_stop(void)
{  return ioctl(cd_fd,CDROMSTOP);
}


int cd_eject(void)
{  cd_status.status=CD_NOSTATUS;
   cd_toc.n_track=-1;
   return ioctl(cd_fd,CDROMEJECT);
}


int cd_close_tray(void)
{  return ioctl(cd_fd,CDROMCLOSETRAY);
}
