/* --------------------------------------------------------------------
   Project: Communication package Linux-HPx00LX Filer
   Module:  lxdir.c
   Author:  A. Garzotto
   Started: 28. Nov. 95
   Subject: Copy between Linux and the palmtop
   -------------------------------------------------------------------- */

/* --------------------------------------------------------------------
                             includes
   -------------------------------------------------------------------- */
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/stat.h>

#define NIL 0

/* --------------------------------------------------------------------
                         local includes
   -------------------------------------------------------------------- */

#include "pal.h"
#include "palpriv.h"
#include "config.h"

size_t  MySendBlock(void *Buf, size_t Size, void *Handle);
size_t  MyRecvBlock(void *Buf, size_t Size, void *Handle);

/* --------------------------------------------------------------------
                        Global types and vars
   -------------------------------------------------------------------- */

struct list_desc
{
   char *name;
   char *fname;
   int processed;
   struct list_desc *next;
};

typedef struct list_desc *LIST;

LIST inlist = NIL;
LIST outlist = NIL;

/* --------------------------------------------------------------------
                        display help
   -------------------------------------------------------------------- */

static void help(void)
{
   printf("USAGE: lxcopy [options] <source file > {<source file>} <dest file>\n");
   printf("  options: -<n> sets comm port <n>\n");
   printf("           -b <baud> sets baud rate to <baud>\n");
   printf("  note that the palmtop file name must have a drive specified\n");
   exit(1);
}

/* --------------------------------------------------------------------
                        make DOS directory string
   -------------------------------------------------------------------- */

static void makedir(char *dir, int stars)
{
   char *p = dir;
   int hasdot = 0;
   int hasstar = 0;
   
   while (*p)
   {
      if (*p == '/') *p = '\\';
      if (*p == '.') hasdot = 1;
      if (*p == '*') hasstar = 1;
      p++;
   }
   if (stars && !hasdot && !hasstar)
   {
      if (p[-1] == '\\')
	 strcat(dir, "*.*");
      else
	 strcat(dir, "\\*.*");
   }
}

/* --------------------------------------------------------------------
                        User defined GET/SEND blocks
   -------------------------------------------------------------------- */

FLCB MyFlCb;

/* User defined Send-block routine, replaces default FlCb */

size_t  MySendBlock(void *Buf, size_t Size, void *Handle)
{
   printf("."); fflush(stdout);
   return fread(Buf, 1, Size, Handle);
}

/* User defined Get-block routine, replaces default FlCb */

size_t  MyRecvBlock(void *Buf, size_t Size, void *Handle)
{
   printf("."); fflush(stdout);
   return fwrite(Buf, 1, Size, Handle);
}


/* --------------------------------------------------------------------
                               Add name to file list
   -------------------------------------------------------------------- */

LIST add_list(char *fname, int input, int processed)
{
   LIST li = inlist;
   LIST l = (LIST)malloc(sizeof(struct list_desc));
   
   while (li && (li->next)) li = li->next;
   l->name = (char *)malloc(strlen(fname) + 15);
   strcpy(l->name, fname);
   l->processed = processed;
   l->fname = NIL;
   l->next = NIL;
   if (input)
   {
      if (li)
	 li->next = l;
      else
	 inlist = l;
   }
   else
      outlist = l;
   return l;
}

/* --------------------------------------------------------------------
                               Copy from LX to Linux
   -------------------------------------------------------------------- */

void from_lx(FILERCOM *pFiler)
{
   int sta, num, is_dir;
   LIST l = inlist;
   LIST l1;
   char *p, source[256], target[256];
   struct stat statbuf;
   
   while (l)
   {
      if (!l->processed)
      {
	 makedir(l->name, 1);
	 sta = FilerAskDir(pFiler, l->name);
         if (sta == NO_RESPONSE) printf("\nServer Not responding.\n");
	 num = 0;
         while (1)
	 {
            if(FilerGetDir(pFiler) == CANNOT_GET_ENTRY) break;
            if (pFiler->Attribute & 0x10) continue;
	    num++;
	    l1 = add_list(l->name, 1, 1);
	    l1->fname = (char *)malloc(strlen(pFiler->Name) + 1);
	    strcpy(l1->fname, pFiler->Name);
            p = &(l1->name[strlen(l1->name) - 1]);
            while (*p != '\\') p--;
            *p = '\0';
	 }
	 if (!num)
	    printf("Cannot find '%s'!\n", l->name);
      }
      l = l->next;
   }

   stat(outlist->name, &statbuf);
   is_dir = statbuf.st_mode & S_IFDIR;
   l = inlist;
   while (l)
   {
      if (l->processed)
      {
	 sprintf(source, "%s\\%s", l->name, l->fname);
	 if (is_dir)
            sprintf(target, "%s/%s", outlist->name, l->fname);
	 else
            sprintf(target, "%s", outlist->name);
         printf("Copying '%s' to '%s'.", source, target);
	 fflush(stdout);
         sta = FilerGetFile(pFiler, source, target);
         if (sta !=  GOT_FILE_OK) printf("\nCannot get file.\n");
	 printf("\n");
      }
      l = l->next;
   }
}


/* --------------------------------------------------------------------
                               Copy from Linux to LX
   -------------------------------------------------------------------- */

void from_linux(FILERCOM *pFiler)
{
   char target[356];
   LIST l = inlist;
   char *p;
   int sta, is_dir = 0;
   
   makedir(outlist->name, 0);
   sta = FilerAskDir(pFiler, outlist->name);
   if (sta == NO_RESPONSE) printf("\nServer Not responding.\n");
   while (1)
   {
      if(FilerGetDir(pFiler) == CANNOT_GET_ENTRY) break;
      if (pFiler->Attribute & 0x10) is_dir = 1;
   }
   if (outlist->name[strlen(outlist->name) - 1] == '\\')
   {
      is_dir = 1;
      outlist->name[strlen(outlist->name) - 1] = '\0';
   }
   
   while (l)
   {
      strcpy(target, outlist->name);
      if (is_dir)
      {
	 p = &(l->name[strlen(l->name) - 1]);
	 while ((p > l->name) && (*p != '/')) p--;
	 if (*p == '/') p++;
	 strcat(target, "\\");
	 strcat(target, p);
      }
      printf("Copying '%s' to '%s'.", l->name, target);
      fflush(stdout);
      sta = FilerSendFile(pFiler, l->name, target);
      if (sta != FILE_SEND_OK) printf("\nCannot send file.\n");
      printf("\n");
      l = l->next;
   }
}

/* --------------------------------------------------------------------
                               M A I N
   -------------------------------------------------------------------- */

void main (int argc, char **argv)
{
   int   stat;
   int port = 1;
   int speed = DEF_BAUD;
   FILERCOM *pFiler;
   FLCB MyFlCb;
   int i = 1;
   
   printf("LXCOPY 1.0 by A. Garzotto\n\n");
   
   while (i < argc)
   {
      if (argv[i][0] == '-')
      {
	 switch (argv[i][1])
	 {
	  case '1': port = 1; break;
	  case '2': port = 2; break;
	  case '3': port = 3; break;
	  case '4': port = 4; break;
	  case 'B':
	  case 'b': speed = atoi(argv[++i]); break;
	  default: help(); break;
	 }
      }
      else if (i < argc - 1)
	 add_list(argv[i], 1, 0);
      else
	 add_list(argv[i], 0, 0);
      i++;      
   }
   if (!inlist || !outlist) help();
   
   /* replace default (PAL) FlCb handler by one of our own */
   MyFlCb = FlCb;
   MyFlCb.FlcbSendBlock = MySendBlock;
   MyFlCb.FlcbRecvBlock = MyRecvBlock;

   if(!(pFiler = FilerConnect(port, speed, &MyFlCb))) {
      printf("\nUnable to connect to palmtop!\n");
      exit(1);
   }
   
   if (strchr(outlist->name, ':'))
      from_linux(pFiler);
   else
      from_lx(pFiler);
   
   FilerDisconnect(pFiler);
   exit(0);
}

