/*
 *
 *	SixPack is Copyright (C) 1996 Kaz Kylheku
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *	The author may be contacted at:
 *
 *	Kaz Kylheku
 *	2869 East 14th Avenue
 *	Vancouver, B.C.
 *	CANADA
 *	V5M 2H8
 *	email: kaz@cafe.net
 *
 */


#include <time.h>
#include <stdlib.h>
#include "transmit.h"
#include "packet.h"
#include "stream.h"
#include "trace.h"
#include "local.h"

#define MIN(a, b)	((a) < (b) ? (a) : (b))

static stream_t newstream(bufio_t iodev, size_t bufsize, size_t segsize)

{
	stream_t str = malloc(sizeof *str);

	if (!str)
		goto fail;

	str->buf = malloc(bufsize);

	if (!str->buf)
		goto fail_free_str;

	str->dgram = malloc(STR_MAX_DGRAM);

	if (!str->dgram)
		goto fail_free_buf;

	str->iodev = iodev;
	str->dev = buf_getdev(iodev);
	str->bufsize = bufsize;
	str->dgrsize = STR_MAX_DGRAM;
	str->bufhead = 0;
	str->buftail = 0;
	str->position = 0;
	str->bufcount = 0;
	str->ackcount = 0;
	str->lowater = bufsize - bufsize / 4;
	str->hiwater = bufsize;
	str->segsize = segsize;
	str->state = SS_closed;

	reg_init(&str->reg);

	return str;

fail_free_buf:
	free(str->buf);
fail_free_str:
	free(str);
fail:
	return 0;
}

static void deletestream(stream_t str)

{
	reg_deinit(&str->reg);
	free(str->dgram);
	free(str->buf);
	free(str);
}

static int closestream(stream_t str)

{
	estream_t es, *s = &es;
	unsigned char send[64];

	ex_bindbuf(s, send, sizeof send);

	ex_seek(s, packet_headersize());
	ex_put16u(s, SC_close);
	packet_encode(s, 2);

	if (!dgram_transmit(str->iodev, ex_getbuf(s), ex_getpos(s)))
		return 0;
	if (!dgram_transmit(str->iodev, ex_getbuf(s), ex_getpos(s)))
		return 0;
	if (!dgram_transmit(str->iodev, ex_getbuf(s), ex_getpos(s)))
		return 0;

	return 1;
}


static int connectstream(stream_t str)

{
	estream_t es, er, *s = &es, *r = &er;
	unsigned char send[64], recv[64];
	int tries = STR_CON_RETRANS;

	ex_bindbuf(s, send, sizeof send);
	ex_bindbuf(r, recv, sizeof recv);

	ex_seek(s, packet_headersize());
	ex_put16u(s, SC_connect);
	packet_encode(s, 2);

	device_timeout(str->dev, STR_CON_TIMEOUT);

	while (tries--) {
		size_t nrecv;
		unsigned datasz;

		if (!dgram_transmit(str->iodev, ex_getbuf(s), ex_getpos(s)))
			return 0;
	again:
		if (!dgram_receive(str->iodev, recv, sizeof recv, &nrecv)) {
			if (buf_error(str->iodev) == bufe_timeout) {
				buf_clrerror(str->iodev);
				continue;
			}
			goto again;
		}


		if (nrecv == 0)
			continue;
		ex_seek(r, 0);
		if (!packet_decode(r, &datasz) || datasz != 2)
			goto again;
		ex_seek(r, packet_headersize());
		if (ex_get16u(r) != SC_accept)
			goto again;
		break;
	}

	if (tries >= 0) {
		str->state = SS_connected;
		return 1;
	}

	return 0;
}

static int acceptstream(stream_t str)

{
	estream_t es, er, *s = &es, *r = &er;
	unsigned char send[64], recv[64];
	int tries = STR_CON_RETRANS;

	ex_bindbuf(s, send, sizeof send);
	ex_bindbuf(r, recv, sizeof recv);

	ex_seek(s, packet_headersize());
	ex_put16u(s, SC_accept);
	packet_encode(s, 2);

	device_timeout(str->dev, STR_CON_TIMEOUT);

	while (tries--) {
		size_t nrecv;
		unsigned datasz;
		if (!dgram_receive(str->iodev, recv, sizeof recv, &nrecv)) {
			buf_clrerror(str->iodev);
			continue;
		}
		if (nrecv == 0)
			continue;
		ex_seek(r, 0);
		if (!packet_decode(r, &datasz) || datasz != 2)
			continue;
		ex_seek(r, packet_headersize());
		if (ex_get16u(r) != SC_connect)
			continue;
		if (!dgram_transmit(str->iodev, ex_getbuf(s), ex_getpos(s)))
			return 0;
		break;
	}

	if (tries >= 0) {
		str->state = SS_connected;
		return 1;
	}

	return 0;
}

static void advance(stream_t str, unsigned long pos)

{
	unsigned long diff = pos - str->position;

	if (diff <= str->bufcount) {
		str->bufcount -= diff;
		str->buftail = (str->buftail + diff) % str->bufsize;
		str->position = pos;
	}
}

static void retransmit(stream_t str, size_t offset, size_t howmuch)

{
	estream_t es, *s = &es;
	size_t ch = (str->buftail + offset) % str->bufsize;
	size_t remain = MIN(str->bufcount - offset, howmuch);
	unsigned long pos = str->position + offset;
	unsigned segsz;
	size_t i;
	static unsigned long count;

	ex_bindbuf(s, str->dgram, str->dgrsize);

	while (remain) {
		segsz = MIN(str->segsize, remain);

		ex_seek(s, packet_headersize());
		ex_put16u(s, SC_data);
		ex_put32u(s, pos);
		ex_put16u(s, segsz);
		
		for (i = segsz; i > 0; i--) {
			ex_put8(s, str->buf[ch++]);
			if (ch >= str->bufsize)
				ch = 0;
		}

		tprintf(3, "transmit #%lu (%d bytes/%lu pos)" NL,
		    count++, segsz, pos);

		remain -= segsz;
		pos += segsz;

		packet_encode(s, 8 + segsz);

		if (!dgram_transmit(str->iodev, ex_getbuf(s), ex_getpos(s))) {
			tprintf(2, "transmit failed!" NL);
		}
	}
}


static int pollwrite(stream_t str, timeout_t timeout)

{
	estream_t er, *r = &er;
	size_t nrecv;
	unsigned datasz;
	unsigned long ackpos;
	unsigned char recv[64];
	streamcmd_t command;

	device_timeout(str->dev, timeout);
	ex_bindbuf(r, recv, sizeof recv);

tryagain:
	if (!dgram_receive(str->iodev, recv, sizeof recv, &nrecv)) {
		if (buf_error(str->iodev)) {
			buf_clrerror(str->iodev);
			if (str->segsize > 255)
				str->segsize /= 2;
			tprintf(2, "timeout---retransmitting" NL);
			retransmit(str, 0, -1);
			return 0;
		}
		tprintf(2, "receive error" NL);
		goto tryagain;
	}

	if (nrecv == 0)
		return 0;

	if (!packet_decode(r, &datasz)) {
		tprintf(2, "packet decode failed" NL);
		goto tryagain;
	}

	if (datasz < 2) {
		tprintf(2, "unexpectedly small packet received" NL);
		goto tryagain;
	}

	ex_seek(r, packet_headersize());

	command = ex_get16u(r);

	switch (command) {
	case SC_data_ack:
		ackpos = ex_get32u(r);

		if (ackpos > str->position) {
#if 0
			if (str->ackcount == 0 && str->segsize < 1024) {
				str->segsize *= 2;
				str->hiwater += str->bufsize / 8;
			} 
#endif
			advance(str, ackpos);
		}

		if (ackpos == str->ackposition)
			str->ackcount++;
		else {
			str->ackposition = ackpos;
			str->ackcount = 1;
		}

		tprintf(3, "ack received (%lu/%d)" NL, ackpos, str->ackcount);

		if (str->ackcount > 2) {
			tprintf(2, "duplicate acks (%lu)---retransmitting" NL,
			    ackpos);
			str->ackcount = 0;
			retransmit(str, 0, 2 * str->segsize);
		}

		break;
	case SC_close:
		str->state = SS_closed;
		break;
	default:
		tprintf(2, "unknown packet received" NL);
		break;
	}

	return 1;
}

static int drain(stream_t str, size_t lowater)

{
	int tries = 12;

	while (str->bufcount > lowater && tries) {
		if (!pollwrite(str, 5000))
			tries--;
		else
			tries = 12;
	}

	return tries > 0;
}

static int add_write(stream_t str, unsigned char *data, size_t size)

{
	while (size) {
		size_t room;
		size_t offset;

		if (str->bufcount >= str->hiwater && !drain(str, str->lowater))
			return 0;

		room = str->bufsize - str->bufcount;
		offset = str->bufcount;

		while (size && room) {
			str->buf[str->bufhead++] = *data++;
			if (str->bufhead >= str->bufsize)
				str->bufhead = 0;
			size--;
			room--;
			str->bufcount++;
		}

		retransmit(str, offset, -1);
	}

	return 1;
}

static int add_read(stream_t str, estream_t *e)

{
	unsigned long pos = ex_get32u(e);
	unsigned int segsz = ex_get16u(e);
	size_t room = str->bufsize - str->bufcount;
	unsigned long maxpos = str->position + room - segsz;
	unsigned long newpos;
	size_t writeptr;
	static unsigned long count;

	if (pos > maxpos)
		return 0;

	if (pos < str->position)
		return 1;

	if (pos > str->position) {
		tprintf(3, "out of order receive #%lu (%d bytes/%lu pos)" NL,
		    count++, segsz, pos);
	} else {
		tprintf(3, "in order receive #%lu (%d bytes/%lu pos)" NL,
		    count++, segsz, pos);
	}

	reg_add(&str->reg, pos, pos + segsz - 1);
	newpos = reg_firsthole(&str->reg);

	writeptr = (pos - str->position + str->bufhead) % str->bufsize;

	if (newpos != str->position) {
		str->bufcount += (newpos - str->position);
		str->bufhead += (newpos - str->position);
		str->bufhead %= str->bufsize;
		str->position = newpos;
	}

	while (segsz--) {
		str->buf[writeptr++] = ex_get8(e);
		if (writeptr >= str->bufsize)
			writeptr = 0;
	}

	return 1;
}

static void acknowledge(stream_t str)

{
	estream_t es, *s = &es;
	unsigned char send[64];

	ex_bindbuf(s, send, sizeof send);
	ex_seek(s, packet_headersize());

	ex_put16u(s, SC_data_ack);
	ex_put32u(s, str->position);

	packet_encode(s, 6);

	dgram_transmit(str->iodev, ex_getbuf(s), ex_getpos(s));
}



static int pollread(stream_t str, timeout_t timeout)

{
	estream_t er, *r = &er;
	size_t nrecv;
	unsigned datasz;
	streamcmd_t command;

	device_timeout(str->dev, timeout);
	ex_bindbuf(r, str->dgram, str->dgrsize);

tryagain:
	if (!dgram_receive(str->iodev, str->dgram, str->dgrsize, &nrecv)) {
		buferr_t err = buf_error(str->iodev);

		tprintf(2, "receive error or timeout" NL); 
		buf_clrerror(str->iodev);
		/* acknowledge(str); */
		if (err)
			return 0;
		goto tryagain;
	}

	if (nrecv == 0)
		return 0;

	if (!packet_decode(r, &datasz)) {
		tprintf(2, "packet decode failed" NL);
		goto tryagain;
	}

	if (datasz < 2) {
		tprintf(2, "unexpectedly small packet received" NL);
		goto tryagain;
	}

	ex_seek(r, packet_headersize());

	command = ex_get16u(r);

	switch (command) {
	case SC_data:
		add_read(str, r);
		acknowledge(str);
		break;
	case SC_close:
		str->state = SS_closed;
		break;
	default:
		tprintf(2, "unknown packet received" NL);
		break;
	}

	return 1;
}

static int get_read(stream_t str, unsigned char *buf, size_t size, size_t *nr)

{
	size_t count = 0;
	int tries = 60;

	while (tries-- && !str->bufcount)
		if (pollread(str, 1000))
			tries = 60;
	
	if (tries < 0)
		return 0;

	while (size && str->bufcount) {
		*buf++ = str->buf[str->buftail++];
		if (str->buftail >= str->bufsize)
			str->buftail = 0;
		str->bufcount--;
		size--;
		count++;
	}

	*nr = count;

	return 1;
}

stream_t stream_connect(bufio_t iodev, size_t bufsize, size_t segsize)

{
	stream_t str = newstream(iodev, bufsize, segsize);

	if (!str)
		return 0;

	if (connectstream(str))
		return str;

	deletestream(str);
	return 0;
}

stream_t stream_accept(bufio_t iodev, size_t bufsize)

{
	stream_t str = newstream(iodev, bufsize, 0);

	if (!str)
		return 0;

	if (acceptstream(str))
		return str;

	deletestream(str);
	return 0;
}

void stream_close(stream_t str)

{
	if (str->state == SS_connected)
		closestream(str);
	deletestream(str);
}


int stream_write(stream_t str, void *buf, size_t size)

{
	if (str->state == SS_connected)
		return add_write(str, buf, size);
	return 0;
}

int stream_drain(stream_t str)

{
	if (str->state == SS_connected)
		return drain(str, 0);
	return 0;
}

int stream_read(stream_t str, void *buf, size_t size, size_t *nread)

{
	if (str->state == SS_connected)
		return get_read(str, buf, size, nread);
	return 0;
}

int stream_pause(stream_t str);
int stream_resume(stream_t str);
