/*
 *
 *	SixPack is Copyright (C) 1996 Kaz Kylheku
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *	The author may be contacted at:
 *
 *	Kaz Kylheku
 *	2869 East 14th Avenue
 *	Vancouver, B.C.
 *	CANADA
 *	V5M 2H8
 *	email: kaz@cafe.net
 *
 */


#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include "bufio.h"
#include "stream.h"
#include "sixbit.h"
#include "basename.h"
#include "exrep.h"
#include "trace.h"
#include "local.h"
#include "help.h"

static char *filename;
static char *devicename = DEFAULT_SEND_DEVICE;
static char *drivername = DEFAULT_DRIVER;

static driver_t *driver;
static device_t device;
static bufio_t bio;
static size_t bufsize = 4096;
static size_t segsize = 1024;

static void deinit(void)

{
	buf_dispose(bio);
	device_close(device);
	driver_deinit();
}

static void init(void)

{
	if (!filename) {
		tprintf(0, "Filename argument missing, try the -h option!" NL);
		exit(EXIT_FAILURE);
	}

	driver_init();

	driver = driver_lookup(drivername);

	if (!driver) {
		tprintf(0, "Driver ``%s'' not compiled in!" NL, drivername);
		exit(EXIT_FAILURE);
	}

	device = device_open(driver, devicename);

	if (!device) {
		driver_deinit();
		tprintf(0, "Failed to open file ``%s'' using driver ``%s''!" NL,
		    devicename, drivername);
		exit(EXIT_FAILURE);
	}

	bio = buf_create(device, 1024,
	    sixbit_bloat(STR_MAX_DGRAM + STR_MAX_DGRAM / 4));

	if (!bio) {
		device_close(device);
		driver_deinit();
		tprintf(0, "Memory allocation failure" NL);
		exit(EXIT_FAILURE);
	}

	atexit(deinit);
}

static int options(int argc, char **argv)

{
	int i, j;

	for (i = 1; i < argc; i++) {
		if (argv[i][0] == '-') {
			for (j = 1; argv[i][j]; j++) {
				int option;
				switch(argv[i][j]) {
				case 't':
					option = 't';
					if (i == argc - 1 || argv[i][j+1])
						goto arg_error;
					devicename = argv[++i];
					goto next_arg;
				case 'd':
					option = 'd';
					if (i == argc - 1 || argv[i][j+1])
						goto arg_error;
					drivername = argv[++i];
					goto next_arg;
				case 'b':
					option = 'b';
					if (i == argc - 1 || argv[i][j+1])
						goto arg_error;
					bufsize = atoi(argv[++i]);
					goto next_arg;
				case 's':
					option = 's';
					if (i == argc - 1 || argv[i][j+1])
						goto arg_error;
					segsize = atoi(argv[++i]);
					if (segsize > STR_MAX_DGRAM) {
						tprintf(0, "max segment size "
						    "is %u" NL,
						    (unsigned) STR_MAX_DGRAM);
						return 0;
					}
					goto next_arg;
				case 'v':
					tracelevel++;
					break;
				case 'w':
					tprint_warranty();
					return 0;
				case 'c':
					tprint_distrib();
					return 0;
				case 'h':
					tprint_sps_help();
					return 0;
				case 'l':
					tprint_drivers();
					return 0;
				default:
					tprintf(0, "unknown option ``-%c'' "
					   "try -h for help" NL,
					    argv[i][j]);
					return 0;
				arg_error:
					tprintf(0, "option ``-%c''"
					    " requires argument" NL, option);
					return 0;
				}
			}
		next_arg:
			continue;
		} else {
			for (j = i; j < argc; j++) {
				switch (j-i) {
				case 0:
					filename = argv[j];
					break;
				default:
					tprintf(0, "too many arguments" NL);
					return 0;
				}
			}
		}
	}

	return 1;
}

int senddata(void)

{
	stream_t str;
	size_t nread;
	static unsigned char buf[STR_MAX_DGRAM];
	char base[64] = "";
	unsigned char packet[68];
	FILE *sendfile;
	unsigned long filesize, total = 0;
	estream_t es, *e = &es;

	if (!basename(base, sizeof base, PATH_SEP, filename)) {
		tprintf(0, "The base filename of ``%s'' exceeds"
		    " %u characters in length" NL, filename,
		    (unsigned) (sizeof base - 1));
		exit(EXIT_FAILURE);
	}

	sendfile = fopen(filename, "rb");

	if (!sendfile) {
		tprintf(0, "Unable to open ``%s''." NL, filename);
		exit(EXIT_FAILURE);
	}

	if (fseek(sendfile, 0, SEEK_END)) {
		tprintf(0, "Unable to seek the end of ``%s''." NL, filename);
		exit(EXIT_FAILURE);
	}

	filesize = ftell(sendfile);
	rewind(sendfile);

	str = stream_connect(bio, bufsize, segsize);

	if (!str) {
		tprintf(1,"Got bored waiting for peer, exiting!" NL);
		exit(EXIT_FAILURE);
	}

	ex_bindbuf(e, packet, sizeof packet);

	ex_put8string(e, (unsigned char *) base, sizeof base);
	ex_put32u(e, filesize);

	if (!stream_write(str, ex_getbuf(e), ex_getpos(e))) {
		tprintf(1, "A serious communication error occured");
		tprintf(1, "during transmission of the file name" NL);
		exit(EXIT_FAILURE);
	}

	while ((nread = fread(buf, 1, segsize, sendfile))) {
		if (!stream_write(str, buf, nread)) {
			tprintf(1, "The communication link failed" NL);
			exit(EXIT_FAILURE);
		}
		total += nread;
		tprintf(1, "transmitted %lu/%lu bytes" CR, total, filesize);
		tputchar(2, '\n');
		fflush(stdout);
		if (nread < segsize)
			break;
	}

	tprintf(2, "waiting for buffers to drain" NL);
	if (!stream_drain(str)) {
		tprintf(1, "The communication link failed" NL);
		exit(EXIT_FAILURE);
	}


	fclose(sendfile);
	stream_close(str);

	return 0;
}

static void sigexit(int sig)

{
	tprintf(0, "Exiting on signal %d." NL, sig);
	exit(EXIT_FAILURE);
}

#include "region.h"

int main(int argc, char **argv)

{
#if 1
	signal(SIGTERM, sigexit);
	signal(SIGINT, sigexit);

	if (!options(argc, argv))
		exit(EXIT_FAILURE);

	init();
	tprint_title();
	senddata();
#else

	region_t reg;
	unsigned long low, high;
	char cmd[2];

	reg_init(&reg);
	while (scanf("%1s%lu%lu", cmd, &low, &high) == 3) {
		switch(cmd[0]) {
		case 'a':
			reg_add(&reg, low, high);
			break;
		case 'd':
			reg_del(&reg, low, high);
			break;
		case 't':
			printf("%d\n", reg_testhole(&reg, low, high));
			break;
		}
	}

	reg_print(&reg, stdout);
	reg_deinit(&reg);

#endif
	return 0;
}
