/*
 *
 *	SixPack is Copyright (C) 1996 Kaz Kylheku
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *	The author may be contacted at:
 *
 *	Kaz Kylheku
 *	2869 East 14th Avenue
 *	Vancouver, B.C.
 *	CANADA
 *	V5M 2H8
 *	email: kaz@cafe.net
 *
 */


#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include "bufio.h"
#include "stream.h"
#include "sixbit.h"
#include "exrep.h"
#include "trace.h"
#include "local.h"
#include "help.h"

static char *devicename = DEFAULT_RECEIVE_DEVICE;
static char *drivername = DEFAULT_DRIVER;

static driver_t *driver;
static device_t device;
static bufio_t bio;
static size_t bufsize = 8192;

static void deinit()

{
	buf_dispose(bio);
	device_close(device);
	driver_deinit();
}

static void init(void)

{
	driver_init();

	driver = driver_lookup(drivername);

	if (!driver) {
		tprintf(0, "Driver ``%s'' not compiled in!\n", drivername);
		exit(EXIT_FAILURE);
	}

	device = device_open(driver, devicename);

	if (!device) {
		driver_deinit();
		tprintf(0, "Failed to open file ``%s'' using driver ``%s''!" NL,
		    devicename, drivername);
		exit(EXIT_FAILURE);
	}

	bio = buf_create(device,
	    sixbit_bloat(STR_MAX_DGRAM + STR_MAX_DGRAM / 4), 1024);

	if (!bio) {
		driver_deinit();
		device_close(device);
		tprintf(0, "Memory allocation failure" NL);
		exit(EXIT_FAILURE);
	}

	atexit(deinit);
}

static int options(int argc, char **argv)

{
	int i, j;

	for (i = 1; i < argc; i++) {
		if (argv[i][0] == '-') {
			for (j = 1; argv[i][j]; j++) {
				int option;
				switch(argv[i][j]) {
				case 't':
					option = 't';
					if (i == argc - 1 || argv[i][j+1])
						goto arg_error;
					devicename = argv[++i];
					goto next_arg;
				case 'd':
					option = 'd';
					if (i == argc - 1 || argv[i][j+1])
						goto arg_error;
					drivername = argv[++i];
					goto next_arg;
				case 'b':
					option = 'b';
					if (i == argc - 1 || argv[i][j+1])
						goto arg_error;
					bufsize = atoi(argv[++i]);
					goto next_arg;
				case 'v':
					tracelevel++;
					break;
				case 'w':
					tprint_warranty();
					return 0;
				case 'c':
					tprint_distrib();
					return 0;
				case 'h':
					tprint_spr_help();
					return 0;
				case 'l':
					tprint_drivers();
					return 0;
				default:
					tprintf(0, "unknown option ``-%c'' "
					   "try -h for help" NL,
					    argv[i][j]);
					return 0;
				arg_error:
					tprintf(0, "option ``-%c'' requires"
					    " argument" NL, option);
					return 0;
				}
			}
		next_arg:
			continue;
		} else {
			for (j = i; j < argc; j++) {
				switch (j-i) {
				default:
					tprintf(0, "too many arguments" NL);
					return 0;
				}
			}
		}
	}

	return 1;
}


int receivedata()

{
	stream_t str;
	size_t nread;
	unsigned long total = 0;
	char buf[4096];
	char base[64];
	unsigned char packet[68];
	FILE *recvfile;
	unsigned long filesize;
	estream_t es, *e = &es;

	str = stream_accept(bio, bufsize);

	if (!str) {
		tprintf(1, "Got bored waiting for peer, exiting!" NL, stderr);
		exit(EXIT_FAILURE);
	}

	if (!stream_read(str, packet, sizeof packet, &nread)
		|| nread != sizeof packet) {
		tprintf(1, "Unable to receive file name/size!" NL);
		exit(EXIT_FAILURE);
	}

	ex_bindbuf(e, packet, sizeof packet);
	ex_get8string(e, (unsigned char *) base, sizeof base);
	filesize = ex_get32u(e);

	recvfile = fopen(base, "w");

	if (!recvfile) {
		tprintf(0, "Could not open ``%s'' for writing!" NL, base);
		exit(EXIT_FAILURE);
	}

	while (stream_read(str, buf, sizeof buf, &nread)) {
		if (fwrite(buf, 1, nread, recvfile) < nread) {
			tprintf(0, "Error writing to ``%s''!" NL, base);
			exit(EXIT_FAILURE);
		}
		fflush(recvfile);
		total += nread;
		tprintf(1, "received %lu/%u bytes" CR, total, filesize);
		tputchar(2, '\n');
		if (total == filesize)
			break;
		tflush(1);
	}

	tputchar(1, '\n');

	if (total != filesize) {
		tprintf(1, "Received %lu out of %lu bytes!" NL,
			total, filesize);
	} else {
		tprintf(1, "Successful receipt of %lu bytes!" NL, total);
	}


	fclose(recvfile);
	stream_close(str);

	return 0;
}

static void sigexit(int sig)

{
	tprintf(0, "Exiting on signal %d." NL, sig);
	exit(EXIT_FAILURE);
}

int main(int argc, char **argv)

{
	signal(SIGTERM, sigexit);
	signal(SIGINT, sigexit);

	if (!options(argc, argv))
		exit(EXIT_FAILURE);

	init();
	tprint_title();
	receivedata();

	return 0;
}
