/*
 *
 *	SixPack is Copyright (C) 1996 Kaz Kylheku
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *	The author may be contacted at:
 *
 *	Kaz Kylheku
 *	2869 East 14th Avenue
 *	Vancouver, B.C.
 *	CANADA
 *	V5M 2H8
 *	email: kaz@cafe.net
 *
 */

#include "region.h"

extern void reg_init(region_t *reg)

{
	int i;

	for (i = 1; i < REGIONS_MAX; i++)
		reg->reg[i].next = &reg->reg[i-1];
	reg->free = &reg->reg[i-1];
	reg->reg[0].next = 0;
	reg->head = 0;
}

extern void reg_deinit(region_t *reg)

{
}

static regent_t *alloc(region_t *reg, unsigned long low, unsigned long high)

{
	regent_t *rn;

	if (!reg->free)
		return 0;

	rn = reg->free;
	reg->free = reg->free->next;
	rn->low = low;
	rn->high = high;

	return rn;
}


static void delete(region_t *reg, regent_t **rn)

{
	regent_t *nx = (*rn)->next;

	(*rn)->next = reg->free;
	reg->free = *rn;

	*rn = nx;
}


static void insert(region_t *reg, regent_t *rn)

{
	regent_t **re;

	for (re = &reg->head; *re; re = &(*re)->next) {
		if (rn->high < (*re)->low) {
			rn->next = *re;
			*re = rn;
			return;
		}
	}

	rn->next = 0;
	*re = rn;
}

static int clip(region_t *reg, unsigned long *low, unsigned long *high)

{
	regent_t **re;

	for (re = &reg->head; *re; ) {
		if (*low < (*re)->low && *high > (*re)->high) {
			delete(reg, re);
			continue;
		}
		if (*low >= (*re)->low && *low <= (*re)->high)
			*low = (*re)->high + 1;
		if (*high >= (*re)->low && *high <= (*re)->high)
			*high = (*re)->low - 1;
		if (*high == -1 || *low == -1 || *low > *high)
			return 0;

		re = &(*re)->next;
	}

	return 1;
}

static void coalesce(region_t *reg)

{
	regent_t **pre;

	for (pre = &reg->head; *pre; ) {
		regent_t *nx = (*pre)->next;

		if (nx) {
			if ((*pre)->high + 1 == nx->low) {
				nx->low = (*pre)->low;
				delete(reg, pre);
				continue;
			}
		} else {
			break;
		}
		pre = &(*pre)->next;
	}
}

int makehole(region_t *reg, unsigned long low, unsigned long high)

{
	regent_t **pre;

	for (pre = &reg->head; *pre; ) {
		if (low <= (*pre)->low && high >= (*pre)->high) {
			delete(reg, pre);
			continue;
		}
		if (low > (*pre)->low && high < (*pre)->high) {
			regent_t *rn = alloc(reg, (*pre)->low, low - 1);

			if (!rn)
				return 0;

			(*pre)->low = high + 1;
			rn->next = *pre;
			*pre = rn;
			return 1;
		} else if (low > (*pre)->low && low <= (*pre)->high) {
			(*pre)->high = low - 1;
		} else if (high < (*pre)->high && high >= (*pre)->low) {
			(*pre)->low = high + 1;
			return 1;
		}

		pre = &(*pre)->next;
	}

	return 1;
}

int reg_add(region_t *reg, unsigned long low, unsigned long high)

{
	regent_t *rn;

	if (!clip(reg, &low, &high))
		return 0;

	if (!(rn = alloc(reg, low, high)))
		return 0;

	insert(reg, rn);
	coalesce(reg);

	return 1;
}

int reg_del(region_t *reg, unsigned long low, unsigned long high)

{
	return makehole(reg, low, high);
}

int reg_testhole(region_t *reg, unsigned long low, unsigned long high)

{
	regent_t *re;

	for (re = reg->head; re; re = re->next) {
		if  (low <= re->high && high >= re->low)
			return 0;
	}

	return 1;

}


unsigned long reg_firsthole(region_t *reg)

{
	regent_t *re = reg->head;

	if (!re)
		return -1;

	if (re->low != 0)
		return 0;

	return (re->high + 1);
}

unsigned long reg_lasthole(region_t *reg)

{
	regent_t *re;

	for (re = reg->head; re; re = re->next) {
		if (!re->next)
			return re->high + 1;
	}

	return -1;
}

extern void reg_print(region_t *reg, FILE *f)

{
	regent_t *re;

	fprintf(f, "region %p:\n", (void *) reg);

	for (re = reg->head; re; re = re->next)
		fprintf(f, "<%lu, %lu>\n", re->low, re->high);

	fprintf(f, "first hole: %lu\n", reg_firsthole(reg));
	fprintf(f, "last hole:  %lu\n", reg_lasthole(reg));
}
