/*
 *
 *	SixPack is Copyright (C) 1996 Kaz Kylheku
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *	The author may be contacted at:
 *
 *	Kaz Kylheku
 *	2869 East 14th Avenue
 *	Vancouver, B.C.
 *	CANADA
 *	V5M 2H8
 *	email: kaz@cafe.net
 *
 */


#include <sys/time.h>
#include <stdlib.h>
#include <termios.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>

#include "device.h"
#include "posixtty.h"

typedef struct posix_s {
	struct termios save;
	struct termios curr;
} posix_t;

#define IFLAGS_ON	(IGNBRK)
#define IFLAGS_OFF	(INLCR | IGNCR | ICRNL | IUCLC | IXON | IXOFF | IXANY)

#define OFLAGS_ON	(0)
#define OFLAGS_OFF	(OPOST | OLCUC | ONLCR | OCRNL | ONOCR | ONLRET)

#define CFLAGS_ON	(CS8 | CREAD)
#define CFLAGS_OFF	(CLOCAL)

#define LFLAGS_ON	(0)
#define LFLAGS_OFF	(ISIG | ICANON | ECHO | IEXTEN)

int posix_init(driver_t *driver)

{
	return 1;
}

void posix_deinit(driver_t *driver)

{
}

int posix_ttyopen(device_t dev, void *specific, va_list args)

{
	char *name = specific;
	posix_t *driverdata;
	int fd;

	if ((fd = open(name, O_RDWR | O_NOCTTY)) < 0)
		goto fail;

	dev->handle.i = fd;

	if ((driverdata = malloc(sizeof *driverdata)) == 0)
		goto fail_free_data;

	dev->driverdata = driverdata;

	if (tcgetattr(fd, &driverdata->curr) < 0)
		goto fail_free_data;

	driverdata->save = driverdata->curr;

	driverdata->curr.c_iflag |=  IFLAGS_ON;
	driverdata->curr.c_iflag &= ~IFLAGS_OFF;

	driverdata->curr.c_oflag |=  OFLAGS_ON;
	driverdata->curr.c_oflag &= ~OFLAGS_OFF;

	driverdata->curr.c_cflag |=  CFLAGS_ON;
	driverdata->curr.c_cflag &= ~CFLAGS_OFF;

	driverdata->curr.c_lflag |=  LFLAGS_ON;
	driverdata->curr.c_lflag &= ~LFLAGS_OFF;

	driverdata->curr.c_cc[VMIN] =  1;
	driverdata->curr.c_cc[VTIME] = 0;

	if (tcsetattr(fd, TCSANOW, &driverdata->curr) < 0)
		goto fail_free_data;

	return 1;

fail_free_data:
	free(driverdata);
fail:
	return 0;
}

int posix_ttyclose(device_t dev)

{
	posix_t *driverdata = dev->driverdata;

	tcsetattr(dev->handle.i, TCSANOW, &driverdata->save);
	close(dev->handle.i);
	free(dev->driverdata);

	return 1;
}

int posix_ttyread(device_t dev, void *buf, size_t nbytes, size_t *nread)

{
	struct timeval tv;
	fd_set rfd;

	if (dev->timeout) {
		int sel;

		tv.tv_sec = dev->timeout / 1000;
		tv.tv_usec = dev->timeout % 1000;
		FD_SET(dev->handle.i, &rfd);
		do {
			sel = select(dev->handle.i + 1, &rfd, NULL, NULL, &tv);
		} while (sel < 0 && errno == EINTR);
		if (sel == 0) {
			*nread = 0;
			return 1;
		}
	}

	if ((*nread = read(dev->handle.i, buf, nbytes)) < 0) {
		*nread = 0;
		return 0;
	}

	return 1;
}

int posix_ttywrite(device_t dev, void *buf, size_t nbytes, size_t *nwrit)

{
	if ((*nwrit = write(dev->handle.i, buf, nbytes)) < 0) {
		*nwrit = 0;
		return 0;
	}

	return 1;
}
