/*
 *
 *	SixPack is Copyright (C) 1996 Kaz Kylheku
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *	The author may be contacted at:
 *
 *	Kaz Kylheku
 *	2869 East 14th Avenue
 *	Vancouver, B.C.
 *	CANADA
 *	V5M 2H8
 *	email: kaz@cafe.net
 *
 */


#include <stdlib.h>
#include "bufio.h"

#undef buf_error
#undef buf_getchar
#undef buf_putchar
#undef buf_ungetchar

bufio_t buf_create(device_t dev, size_t max_in, size_t max_out)

{
	bufio_t buf = malloc(sizeof *buf);
	
	if (!buf)
		goto fail;

	if ((buf->inbuf = malloc(max_in)) == 0)
		goto fail_free_struct;

	if ((buf->outbuf = malloc(max_out)) == 0)
		goto fail_free_inbuf;

	buf->max_in = max_in;
	buf->max_out = max_out;
	buf->cur_in = buf->cur_out = 0;
	buf->avail_in = 0;
	buf->dev = dev;
	buf->error = bufe_ok;

	return buf;

fail_free_inbuf:
	free(buf->inbuf);
fail_free_struct:
	free(buf);
fail:
	return 0;
}

void buf_dispose(bufio_t buf)

{
	free(buf->outbuf);
	free(buf->inbuf);
	free(buf);
}

device_t buf_getdev(bufio_t buf)

{
	return buf->dev;
}



int buf_replenish(bufio_t buf)

{
	int res;
	buf->cur_in = 0;
	res = device_read(buf->dev, buf->inbuf, buf->max_in, &buf->avail_in);

	if (res == 1 && buf->avail_in == 0) {
		buf->error = bufe_timeout;
		return 0;
	}

	if (res == 0)
		buf->error = bufe_error;

	return res;
}

int buf_flush(bufio_t buf)

{
	size_t nwrit;

	int ok = device_write(buf->dev, buf->outbuf, buf->cur_out, &nwrit);

	if (!ok)
		buf->error = bufe_error;
	else
		buf->cur_out = 0;

	return ok;
}


int buf_getchar(bufio_t buf)

{
	if (buf->avail_in <= buf->cur_in)
		if (!buf_replenish(buf))
			return B_EOF;

	return buf->inbuf[buf->cur_in++];
}

int buf_putchar(bufio_t buf, int ch)

{
	if (buf->cur_out == buf->max_out)
		if (!buf_flush(buf))
			return 0;

	buf->outbuf[buf->cur_out++] = ch;
	return 1;
}

int buf_ungetchar(bufio_t buf, int ch)

{
	if (buf->cur_in > 0) {
		buf->inbuf[--buf->cur_in] = ch;
		return 1;
	}
	return 0;
}

int buf_write(bufio_t buf, void *data, size_t nbyte)

{
	char *out = data;

	while (nbyte--)
		if (!buf_putchar(buf, *out++))
			return 0;

	return 1;
}

int buf_read(bufio_t buf, void *data, size_t nbyte)

{
	char *in = data;

	while (nbyte--)
		if ((*in++ = buf_getchar(buf)) == B_EOF)
			return 0;

	return 1;
}

buferr_t buf_error(bufio_t buf)

{
	return buf->error;
}

void buf_clrerror(bufio_t buf)

{
	buf->error = bufe_ok;
}

