/*
    terminal.c -- Terminal code
    Copyright (C) 1996  Nadav Cohen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
#include "terminal.h"
#include "screen.h"
#include "ansi.h"
#include "avatar.h"
#include "vt100.h"
#include "config.h"
#include "modem.h"
#include "phone.h"
#include "window.h"
#include "menu.h"
#include "log.h"
#include "utils.h"
#include "filemgr.h"
#include "setup.h"
#include "error.h"
#include <fcntl.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <ctype.h>
#include <signal.h>

int Online, Offline;
tTermPhoneRec CurrentTerm;
time_t Starttime;
char EMSI_CHT[16], EMSI_ACK[16], EMSI_TCH[16], RING[2], RingStr[20];
WINDOW *Remote, *Local, *Chat;
chtype *Scrl;
char CommandKey, ClosePort, Warned, Device;
tList TagList;
int Tagged;
char *NameOfDialEntry;
int DialIndex = 0;
pid_t pid;

tTag::tTag()
{
 Tag = 0;
}

void SigInt(int sig)
{
 kill(pid, SIGKILL);
}

void PrintStatus()
{
 char Temp[50];
 int i;
 time_t t;
 tm *Time;

 textAttr(Status, Integers[cStatus]);
 werase(Status); 
 for (i=0;i<COLS;i++)
  waddch(Status, ' ');
 switch(Online?Integers[StatusStartOn]:Integers[StatusStartOff]){
  case 0:
    if (Integers[EscType] == 1)
     strcpy(Temp, "Ctrl-Alt-A for help");
    else
     strcpy(Temp, "Alt-A for help");
    break;
  case 1:
    linestatus();
    if (DSR) strcpy(Temp, "DSR");
    else strcpy(Temp, "dsr");
    if (DTR) strcat(Temp, "DTR ");
    else strcat(Temp, "dtr ");
    if (CAR) strcat(Temp, "CD");
    else strcat(Temp, "cd");
    if (ST) strcat(Temp, "ST");
    else strcat(Temp, "st");
    if (SR) strcat(Temp, "SR ");
    else strcat(Temp, "sr ");
    if (CTS) strcat(Temp, "CTS ");
    else strcat(Temp, "cts ");
    if (RNG) strcat(Temp, "R");
    else strcat(Temp, "r");
    break;
  case 2:
    strncpy(Temp, CurrentTerm.Name+1, CurrentTerm.Name[0]);
    Temp[CurrentTerm.Name[0]]=0;
    Temp[20] = 0;
    break;
  default:break;
  }
 waddstr(Status, Temp);
 wmove(Status, 0, 20);
 /* NOTE: DefaultDevice */
 if (CurrentTerm.Device == 0) i = Integers[DefaultDevice-1];
  else i = CurrentTerm.Device-1;
 if (i > 4) i = CurrentTerm.Device-1;
 if (strlen(Strings[Device1+i])>17){
   strncpy(Temp, Strings[Device1+i], 17);
   Temp[17] = 0;
 }
 else strcpy(Temp, Strings[Device1+i]);
 waddch(Status, SVLINE);
 waddstr(Status, Temp);
 sprintf(Temp, " %lu %d%c%d", CurrentTerm.Baud, CurrentTerm.Databits, CurrentTerm.Parity, CurrentTerm.Stopbits+1);
 wmove(Status, 0, 48-strlen(Temp)-1);
 waddstr(Status, Temp);
 wmove(Status, 0, 48);
 sprintf(Temp, "%c %s", SVLINE, TermEmu[CurrentTerm.Terminal]);
 waddstr(Status,Temp);
 wmove(Status, 0, 62);
 if (Offline)
  switch(Integers[StatusEndOff]){
  case 0:
    time(&t);
    Time = localtime(&t);
    sprintf(Temp, "%c Time %02d:%02d:%02d",SVLINE, Time->tm_hour, Time->tm_min, Time->tm_sec);
    break;
  default:break;
  }
 if (Online)
 switch(Integers[StatusEndOn]){
 case 0:
   time(&t);
   Time = localtime(&t);
   sprintf(Temp, "%c Time %02d:%02d:%02d",SVLINE, Time->tm_hour, Time->tm_min, Time->tm_sec);
   break;
 case 1:
   time(&t);
   t-=Starttime;
   sprintf(Temp, "%c Online %02lu:%02lu:%02lu",SVLINE, t / 3600, t / 60, t % 60);
   break;
 default:break;
 }
 waddstr(Status, Temp);
 wrefresh(Status);
 if (EMSI_CHT[0] != 15) wrefresh(Screen);
 else wrefresh(Local);
}

void GetFileName(WINDOW *Window, char *Str, char *Headline, char *Instr)
{
 tWindow *Win;
 char Temp[40];
 int Center, sx, sy;
 
 getyx(Window, sy, sx);
 Center = findCenter(0, COLS, 40);
 Win = new tWindow(Window, Center, LINES/2-1, Center+40, LINES/2+1, Integers[cMenu] >> 4, Integers[cMenu] % 16);
 Win->Frame(2, Integers[cMenu]);
 Win->Headline(Headline, Integers[cMenuTitle]);
 wrefresh(Window);
 
 if (getstring(Window, LINES/2, Center+1, Instr, FileEdit, "\n\e", 39, -1, Temp) == '\n')
  strcpy(Str, Temp);
 else
  strcpy(Str, "");
  
 delete Win;
 wmove(Window, sy, sx);
}

void WarnOnline()
{
 tWindow *Win;
 int Center, Y, X, Attr;
 char Temp[10];
 time_t Time;
 
 Warned = 1;
 time(&Time);
 Attr = Screen->_attrs;
 getyx(Screen, Y, X); 
 Center = findCenter(0, COLS, 28);
 Win = new tWindow(Screen, Center-3, LINES/2-2, Center+31, LINES/2+2, Integers[cInfo] >> 4, Integers[cInfo] % 16);
 Win->Frame(2, Integers[cInfo]);
 textAttr(Screen, Integers[cInfoBold]);
 mvwaddstr(Screen, LINES/2, Center, "You have been online for");
 sprintf(Temp, "%lu", (Time-Starttime) / 60);
 mvwaddstr(Screen, LINES/2, Center+24, Temp);
 wrefresh(Screen);
 sleep(1);

 delete Win;
 wmove(Screen, Y, X);
 Screen->_attrs = Attr;
 if (EMSI_CHT[0] == 15){
  clearok(Remote, TRUE);
  clearok(Local, TRUE);
  redrawwin(Remote);
  redrawwin(Local);
  wrefresh(Remote);
  wrefresh(Local);
 }
}

void Hangup()
{
 tWindow *Win;
 int Center, Y, X;
 
 getyx(Screen, Y, X);
 Center = findCenter(0, COLS, 10);
 Win = new tWindow(Screen, Center-3, LINES/2-2, Center+13, LINES/2+2, Integers[cInfo] >> 4, Integers[cInfo] % 16);
 Win->Frame(2, Integers[cInfo]);
 textAttr(Screen, Integers[cInfoBold]);
 mvwaddstr(Screen, LINES/2, Center, "Hanging up");
 wrefresh(Screen);
 hangup();

 Printlogln("Manual hangup"); 
 delete Win;
 wmove(Screen, Y, X);
}

void ShowEcho()
{
 tWindow *Win;
 int Center, Y, X;
 
 getyx(Screen, Y, X);
 Center = findCenter(0, COLS, 11);
 Win = new tWindow(Screen, Center-3, LINES/2-2, Center+14, LINES/2+2, Integers[cInfo] >> 4, Integers[cInfo] % 16);
 Win->Frame(2, Integers[cInfo]);
 textAttr(Screen, Integers[cInfoBold]);
 mvwaddstr(Screen, LINES/2, Center, "Local Echo");
 if (Integers[GlobalLocalEcho]) mvwaddstr(Screen, LINES/2, Center+11, "On");
 else mvwaddstr(Screen, LINES/2, Center+11, "Off");
 wrefresh(Screen);
 sleep(1);

 delete Win;
 wmove(Screen, Y, X);
}

void ShowStrip()
{
 tWindow *Win;
 int Center, Y, X;
 
 getyx(Screen, Y, X);
 Center = findCenter(0, COLS, 11);
 Win = new tWindow(Screen, Center-3, LINES/2-2, Center+14, LINES/2+2, Integers[cInfo] >> 4, Integers[cInfo] % 16);
 Win->Frame(2, Integers[cInfo]);
 textAttr(Screen, Integers[cInfoBold]);
 mvwaddstr(Screen, LINES/2, Center, "Strip high");
 if (Integers[GlobalStripHigh]) mvwaddstr(Screen, LINES/2, Center+11, "On");
 else mvwaddstr(Screen, LINES/2, Center+11, "Off");
 wrefresh(Screen);
 sleep(1);

 delete Win;
 wmove(Screen, Y, X);
}

void SendBreak()
{
 tWindow *Win;
 int Center, Y, X;
 
 getyx(Screen, Y, X);
 Center = findCenter(0, COLS, 13);
 Win = new tWindow(Screen, Center-3, LINES/2-2, Center+15, LINES/2+2, Integers[cInfo] >> 4, Integers[cInfo ]% 16);
 Win->Frame(2, Integers[cInfo]);
 textAttr(Screen, Integers[cInfoBold]);
 mvwaddstr(Screen, LINES/2, Center, "Sending break");
 wrefresh(Screen);
 sbreak();

 delete Win;
 wmove(Screen, Y, X);
}

void ImageSave(WINDOW *Window)
{
 char Temp[40], Buffer[COLS*(LINES-1)];
 int Handle, y, x, Y, X;

 getyx(Window, Y, X);
 GetFileName(Window, Temp, "Image file", "TERMITE.IMAGE");
 if (strlen(Temp)>0){
  if ((Handle = open(Temp, O_CREAT | O_RDWR)) > 0){
  for (y=0;y<LINES-1;y++)
   for (x=0;x<COLS;x++)
    Buffer[y*COLS+x] = mvwinch(Window, y, x) & A_CHARTEXT;
  write(Handle, Buffer, sizeof(Buffer));
  close(Handle);
  }
  else WindowError(Window, "Unable to open file");
 }
 Printlog("Image save: ");
 Printlogln(Temp);
 curs_set(1);
 wmove(Window, Y, X);
}

void OpenCapture(WINDOW *Window)
{
 char Temp[40], TempStr[255];
 tWindow *Win;
 int Center, Y, X;
 int mode;

 if (!CaptureOn){
  PasToChar(CurrentTerm.Capture, Temp);
  if (strlen(Temp) > 0)
   GetFileName(Window, Temp, "Capture file", Temp);
  else
   GetFileName(Window, Temp, "Capture file", "TERMITE.CAPTURE");
  OpenCaptureFile(Window, &Temp[0]);
  curs_set(1);
 }
 else
 {
  getyx(Window, Y, X);
  Center = findCenter(0, COLS, 17);
  Win = new tWindow(Window, Center-3, LINES/2-2, Center+19, LINES/2+2, Integers[cInfo] >> 4, Integers[cInfo] % 16);
  Win->Frame(2, Integers[cInfo]);
  textAttr(Window, Integers[cInfoBold]);
  mvwaddstr(Window, LINES/2, Center, "Capture is closed");
  wrefresh(Window);

  delete Win;
  wmove(Window, Y, X);
  Printlogln("Capture closed");
  CaptureOn = 0;
  write(CaptureHandle, CaptureStart, CapturePtr-CaptureStart);
  close(CaptureHandle);
 }
}

void ChangeDevice()
{
 Device = CurrentTerm.Device = ChooseDevice(Screen, CurrentTerm.Device, 1);
 CurrentTerm.Parity = Integers[Parities1+CurrentTerm.Device-1];
 CurrentTerm.Baud = Integers[Bauds1+CurrentTerm.Device-1];
 CurrentTerm.Stopbits = Integers[Stopbits1+CurrentTerm.Device-1];
 CurrentTerm.Databits = Integers[Databits1+CurrentTerm.Device-1];
 resetModem();
 closeModem();
 unlock(Strings[Device1+Device-1]);
 initModem(Strings[Device1+Device-1], CurrentTerm.Baud);
 setparms(CurrentTerm.Baud, CurrentTerm.Parity, CurrentTerm.Databits, CurrentTerm.Stopbits,
           Integers[FlowCtrl1+Device-1]);
}

void Upload()
{
 tWindow *Win;
 tMenu *Menu;
 int i, y, x, Key, Attr, hangup;
 char Temp[256], TempStr[256], PWD[256];
 char *FilesToUpload;
 
 nl();
 getyx(Screen, y, x);
 Attr = Screen->_attrs;
 Win = new tWindow(Screen, COLS/2-15, LINES/3-3, COLS/2+35, LINES/3+3, Integers[cMenu] >> 4, Integers[cMenu] % 16);
 Win->Frame(2, Integers[cMenu]);
 Win->Headline("Upload", Integers[cMenuTitle]);
 Menu = new tMenu(Screen, Integers[cMenuText], Integers[cMenuHot], Integers[cMenuBar], Integers[cMenuText],                         ALIGNLEFT);
 for (i=0;i<5;i++){
  Menu->Add(Strings[ProtocolName1+i], COLS/2-14, COLS/2, i+LINES/3-2);
  textAttr(Screen, Integers[cMenuText]);
  mvwaddstr(Screen, i+LINES/3-2, COLS/2+1, Strings[ProtocolInfo1+i]);
 }
 Menu->Show();
 curs_set(0);
 wrefresh(Screen);
 Key = 0;
 while (Key != '\e' && Key != '\n'){
  Key = getEscChar();
  Menu->GetInput(Key);
 }
 
 if (Key == '\n'){
  getcwd(PWD, 255);
  if (Integers[ChangeUl1+Menu->Focus]) 
   if (chdir(Strings[UploadPath]) < 0) WindowError(Screen, "Unable to chdir");
  FileMgr(&FilesToUpload, Strings[UploadPath], 1);
  clearok(Screen, TRUE);
  redrawwin(Screen);
  wrefresh(Screen);
  if (strlen(FilesToUpload) > 0){
   nl();
   hangup = ChooseYesno(Screen, "Hangup after transfer");
   CurrentTerm.UploadKb += UploadSize / 1024;
   SavePhonebook();
   strcpy(TempStr, "Uploading in ");
   strcat(TempStr, Strings[ProtocolName1+Menu->Focus]);
   Printlogln(TempStr);
   Printlogln(FilesToUpload);
   strcpy(TempStr, Strings[ProtocolUlExec1+Menu->Focus]);
   ProtocolParser(TempStr, Strings[UploadPath], Strings[Device1+CurrentTerm.Device-1], CurrentTerm.Baud,                   FilesToUpload);
   clear();
   refresh();
   signal(SIGINT, SigInt);
   system(TempStr);
   signal(SIGINT, SIG_DFL);
   chdir(PWD);
   redrawwin(Screen);
   clearok(Screen, TRUE);
   wrefresh(Status);
   wrefresh(Screen);
   if (hangup) Hangup();
  }
  free(FilesToUpload);
 }
 
 delete Menu;
 delete Win;
 curs_set(1);
 nonl();
 wmove(Screen, y, x);
 Screen->_attrs = Attr;
}

void Download()
{
 tWindow *Win;
 tMenu *Menu;
 int i, y, x, Key, Attr, hangup;
 char Temp[256], TempStr[256], PWD[256];
 
 nl();
 getyx(Screen, y, x);
 Attr = Screen->_attrs;
 Win = new tWindow(Screen, COLS/2-15, LINES/3-3, COLS/2+35, LINES/3+3, Integers[cMenu] >> 4, Integers[cMenu] % 16);
 Win->Frame(2, Integers[cMenu]);
 Win->Headline("Download", Integers[cMenuTitle]);
 Menu = new tMenu(Screen, Integers[cMenuText], Integers[cMenuHot], Integers[cMenuBar], Integers[cMenuText],             ALIGNLEFT);
 for (i=0;i<5;i++){
  Menu->Add(Strings[ProtocolName1+i], COLS/2-14, COLS/2, i+LINES/3-2);
  textAttr(Screen, Integers[cMenuText]);
  mvwaddstr(Screen, i+LINES/3-2, COLS/2+1, Strings[ProtocolInfo1+i]);
 }
 Menu->Show();
 curs_set(0);
 wrefresh(Screen);
 Key = 0;
 while (Key != '\e' && Key != '\n'){
  Key = getEscChar();
  Menu->GetInput(Key);
 }
 
 if (Key == '\n'){
  getcwd(PWD, 255);
  if (Integers[ChangeDl1+Menu->Focus]) 
   if (chdir(Strings[DownloadPath]) < 0) WindowError(Screen, "Unable to chdir");
  strcpy(TempStr, "Downloading in ");
  strcat(TempStr, Strings[ProtocolName1+Menu->Focus]);
  Printlogln(TempStr);
  if (Integers[ProtocolAsk1+Menu->Focus]) GetFileName(Screen, Temp, "Filename to download", "");
  else strcpy(Temp, "");
  Printlogln(Temp);
  nl();
  hangup = ChooseYesno(Screen, "Hangup after transfer");
  strcpy(TempStr, Strings[ProtocolDlExec1+Menu->Focus]);
  ProtocolParser(TempStr, Strings[DownloadPath], Strings[Device1+CurrentTerm.Device-1], CurrentTerm.Baud, Temp);
  clear();
  refresh();
  system(TempStr);
  chdir(PWD);
  clearok(Screen, TRUE);
  redrawwin(Screen);
  wrefresh(Status);
  wrefresh(Screen);
  if (hangup) Hangup();
 }
 
 delete Menu;
 delete Win;
 curs_set(1);
 nonl();
 wmove(Screen, y, x);
 Screen->_attrs = Attr;
}

void SetTerminal()
{
  switch(CurrentTerm.Terminal){
  case 0:
    Ansi = 1;
    Avatar = 1;
    vt100 = 0;
    break;
  case 1:
    Ansi = 0;
    Avatar = 1;
    vt100 = 0;
    break;
  case 2:
    Ansi = 1;
    Avatar = 0;
    vt100 = 0;
    break;
  case 3:
    Ansi = 0;
    Avatar = 0;
    vt100 = 1;
    break;
  case 4:
    Ansi = 0;
    Avatar = 0;
    vt100 = 0;
    break;
  default:break;
  }
}

void AutoDownload(int Protocol)
{
 char PWD[256], Temp[256], Filename[40];
 
 Printlogln("Auto-download initiated");
 clear();
 refresh();
 getcwd(PWD, 255);
 if (Integers[ChangeDl1+Protocol]) 
  if(chdir(Strings[DownloadPath]) < 0) WindowError(Screen, "Unable to chdir");
 if (!Integers[ProtocolAsk1+Protocol]) Filename[0]=0;
 else{ 
  strcpy(Temp, Strings[ProtocolName1+Protocol]);
  strcat(Temp, ": Filename to download");
  GetFileName(Screen, Filename, Temp, "");
 }
 strcpy(Temp, Strings[ProtocolDlExec1+Protocol]);
 ProtocolParser(Temp, Strings[DownloadPath], Strings[Device1+CurrentTerm.Device-1], CurrentTerm.Baud,Filename);
 system(Temp);
 chdir(PWD);
 redrawwin(Screen);
 wrefresh(Status);
 wrefresh(Screen);
}

void Phone()
{
 int Key;
 
 ShowTerm();
 ShowTermBook(1);
 curs_set(0);
 wrefresh(PhoneScreen);
 nl();
 Key = 0;
 while (Key != 27 && !DialerConnect){
  Key = getEscChar();
  TermPhoneKey(Key);
  wrefresh(PhoneScreen);
 }
 DialerConnect = 0;
 curs_set(1);
 nonl();
 redrawwin(Screen);
}

void DeleteTagList()
{
 tTag *Tmp, *Next;
 
 Tmp = (tTag *)TagList.Base;
 while (Tmp != NULL){
   Next = (tTag *)Tmp->Next;
   delete Tmp;
   Tmp = Next;
 }
 TagList.Base = NULL;
}

void WipeLine(WINDOW *Window, WINDOW *Tag, int Y, int Line0, int Line1, int Line2)
{
 chtype ch;
 int i, Color;
 
 Color = (int)(BLUE << 4) + YELLOW + 8;
 for (i=0;i<Line1;i++){
  ch = mvwinch(Window, Y, i+Line0);
  if (Line2)
   textAttr(Tag, Color);
  else
   wattrset(Tag, ch & (A_COLOR | A_ATTRIBUTES));
  ch &= A_CHARTEXT;
  mvwaddch(Tag, Y, i+Line0, ch);
 }
}

void Tagger(WINDOW *Window)
{
 int i, j, k, Color, y, x, Attr, Key;
 char Line[LINES-1][3];
 char Temp[20], TempStr[20], Found, PrevChar, C;
 chtype ch;
 WINDOW *TagScreen;
 tTag *Tmp;
 
 Attr = Window->_attrs;
 getyx(Window, y, x);
 
 Color = (int)(WHITE << 4) + 8;
 for (i=0;i<LINES-1;i++)
 {
  Line[i][0] = 0;
  Line[i][1] = 0;
  Line[i][2] = 0;
  Found = 0;
  PrevChar = ' ';
  for (j=0;j<COLS-12 && !Found;j++){
   for (k=0;k<12;k++)
    Temp[k] = mvwinch(Window, i, j+k) & A_CHARTEXT;
   if (j > 0) PrevChar = mvwinch(Window, i, j-1) & A_CHARTEXT;
   Temp[12] = 0;
   if (ValidName(Temp, TempStr) && isspace(PrevChar)){
    Found = 1;
    Line[i][0] = j;
    Line[i][1] = strlen(TempStr);
   }
  }
 }
 TagScreen = dupwin(Window);
 Key = 0;
 i = 0;
 nl();
 while (Key != '\e'){
  textAttr(TagScreen, Color);
  for (k=0;k<Line[i][1];k++){
   ch = mvwinch(TagScreen, i, k+Line[i][0]) & A_CHARTEXT;
   mvwaddch(TagScreen, i, k+Line[i][0], ch);
  }
  wmove(TagScreen, i, Line[i][1]);
  clearok(TagScreen, TRUE);
  redrawwin(TagScreen);
  wrefresh(TagScreen);
  Key = getEscChar();
  switch(Key){
  case ' ':
    Line[i][2] ^= 1;
    break;
  case KEY_DOWN: 
    if (i < LINES-2){
     WipeLine(Window, TagScreen, i, Line[i][0], Line[i][1], Line[i][2]);
     i++;
    }
    break;
  case KEY_UP:
    if (i > 0){
     WipeLine(Window, TagScreen, i, Line[i][0], Line[i][1], Line[i][2]);
     i--;
    }
    break;
  case KEY_RIGHT:
    if (Line[i][0]+Line[i][1] < COLS){ 
     WipeLine(Window, TagScreen, i, Line[i][0], Line[i][1], Line[i][2]);
     Line[i][0]++;
    }
    break;
  case KEY_LEFT:
    if (Line[i][0] > 0){ 
     WipeLine(Window, TagScreen, i, Line[i][0], Line[i][1], Line[i][2]);
     Line[i][0]--;
    }
    break;
  case KEY_HOME:
    if (Line[i][1] > 0){ 
     WipeLine(Window, TagScreen, i, Line[i][0], Line[i][1], Line[i][2]);
     Line[i][1]--;
    }
    break;
  case KEY_END:
    if (Line[i][0]+Line[i][1] < COLS && Line[i][1] < 12){ 
     WipeLine(Window, TagScreen, i, Line[i][0], Line[i][1], Line[i][2]);
     Line[i][1]++;
    }
    break;
  default:break;
  }
 }
 for (i=0;i<LINES-1;i++){
  if (Line[i][2]){
   Tmp = new tTag;
   for (j=0;j<Line[i][1];j++){
    C = mvwinch(TagScreen, i, Line[i][0]+j) & A_CHARTEXT;
    Tmp->Name[j] = C;
   }
   Tmp->Name[j+1] = 0;
   TagList.Add(Tmp);
   }
 }
 delwin(TagScreen);
 clearok(Window, TRUE);
 redrawwin(Window);
 wmove(Window, y, x);
 Window->_attrs = Attr;
 wrefresh(Window);
 nonl();
}

void MarkLine(WINDOW *Window, int y, int x, int Color)
{
 chtype ch;
 int i;
 
 for (i=x;i<x+12;i++){
   ch = mvwinch(Window, y, i) & A_CHARTEXT;
   textAttr(Window, Color);
   mvwaddch(Window, y, i, ch);
 }
}

tTag *GetTag(int y, int x)
{
 tTag *Tmp;
 int i;
 
 Tmp = (tTag *)TagList.Base;
 for (i=1;i<y*(x+1);i++)
  if (Tmp != NULL)
   Tmp = (tTag *)Tmp->Next;
 return (Tmp);
}

void DrawTag(WINDOW *Window)
{
 int i,j;
 tTag *Tmp;
 char TempStr[5];
 
 wclear(Window);
 textAttr(Window, Integers[cMenu]);
 box(Window, SVLINE, SHLINE);
 textAttr(Window, Integers[cMenuText]);
 mvwaddstr(Window, 1,1, "Clear all");
 mvwaddstr(Window, 2,1, "Delete tagged");
 mvwaddstr(Window, 3,1, "Add entry");
 mvwaddstr(Window, 4,1, "Edit entry");
 mvwaddstr(Window, 5,1, "Untag");
 mvwaddstr(Window, 6,1, "Del  remove");
 mvwaddstr(Window, 7,1, "Space tag");
 mvwaddstr(Window, 8,1, "+ Tag+13");
 mvwaddstr(Window, 9,1, "- Tag+32");
 mvwaddstr(Window, 10,1, "Tagged:");
 textAttr(Window, Integers[cMenu]);
 wmove(Window, 1, 14);
 wvline(Window, SVLINE, LINES-3);
 wmove(Window, 1, 14+13);
 wvline(Window, SVLINE, LINES-3);
 wmove(Window, 1, 14+26);
 wvline(Window, SVLINE, LINES-3);
 wmove(Window, 1, 14+39);
 wvline(Window, SVLINE, LINES-3);
 wmove(Window, 1, 14+52);
 wvline(Window, SVLINE, LINES-3);
 textAttr(Window, WHITE);
 Tagged = 0;
 Tmp = (tTag *)TagList.Base;
 for (j=0;j<5;j++)
 for (i=0;i<LINES-3;i++){
  if (Tmp != NULL){
   mvwaddstr(Window, i+1, j*13+15, Tmp->Name);
   if (Tmp->Tag){
    Tagged++;
    MarkLine(Window, i+1, j*13+15, Integers[cMenuHot]);
   }
   Tmp = (tTag *)Tmp->Next;
  }
 }
 textAttr(Window, Integers[cMenuOptions]);
 sprintf(TempStr, "%03d", Tagged);
 mvwaddstr(Window, 10, 9, TempStr);
 textAttr(Window, WHITE);
}

void SendTags()
{
 int y, x, Key;
 char TempStr[200];
 WINDOW *TagScreen;
 tTag *Tmp, *Next;
 
 TagScreen = newwin(LINES-1, COLS, 0, 0);
 DrawTag(TagScreen);
 MarkLine(TagScreen, 1, 15, (int)((int)WHITE << 4));
 wrefresh(TagScreen);
 Key = 0;
 y = 1;
 x = 0;
 while (Key != '\e'){
   Key = getEscChar();
   switch(Key){
   case ' ':
     Tmp = GetTag(y, x);
     if (Tmp != NULL){
      Tmp->Tag ^= 1;
      if (Tmp->Tag){
       MarkLine(TagScreen, y, x*13+15, Integers[cMenuHot]);
       Tagged++;
      }
      else
      {
       Tagged--;
       MarkLine(TagScreen, y, x*13+15, WHITE);
      }
     textAttr(TagScreen, Integers[cMenuOptions]);
     sprintf(TempStr, "%03d", Tagged);
     mvwaddstr(TagScreen, 10, 9, TempStr);
     textAttr(TagScreen, WHITE);
     if (Integers[TagAdvance]) goto TAGDOWN;
     }
     break;
   case 'a':
   case 'A':
     Tmp = new tTag;
     Tmp->Name[0] = 0;
     TagList.Add(Tmp);
     break;
   case 'c':
   case 'C':
     DeleteTagList();
     DrawTag(TagScreen);
     redrawwin(TagScreen);
     x = 0;
     y = 1;
     MarkLine(TagScreen, 1, 15, (int)((int)WHITE << 4));
     wrefresh(TagScreen);
     break;
   case 'd':
   case 'D':
     Tmp = (tTag *)TagList.Base;
     while (Tmp != NULL){
      Next = (tTag *)Tmp->Next;
      if (Tmp->Tag){
       TagList.Del(Tmp);
      }
      Tmp = Next;
     }
     DrawTag(TagScreen);
     redrawwin(TagScreen);
     x = 0;
     y = 1;
     MarkLine(TagScreen, 1, 15, (int)((int)WHITE << 4));
     wrefresh(TagScreen);
     break;
   case 'e':
   case 'E':
     Tmp = GetTag(y, x);
     if (Tmp != NULL){
      nl();
      if (getstring(TagScreen, y, x*13+15, Tmp->Name, ValidEdit, "\e\n", 12, -1, TempStr) == '\n'){
        strcpy(Tmp->Name, TempStr);
        DrawTag(TagScreen);
        wrefresh(TagScreen);
      }
      nonl();
      MarkLine(TagScreen, y, x*13+15, (int)((int)WHITE << 4));
     }
     break;
   case 'u':
   case 'U':
     Tmp = GetTag(y, x);
     if (Tmp != NULL && Tmp->Tag){
      Tmp->Tag = 0;
      Tagged--;
     }
     textAttr(TagScreen, Integers[cMenuOptions]);
     sprintf(TempStr, "%03d", Tagged);
     mvwaddstr(TagScreen, 10, 9, TempStr);
     textAttr(TagScreen, WHITE);
     break;
   case KEY_DC:
     Tmp = GetTag(y, x);
     if (Tmp != NULL)
      TagList.Del(Tmp);
     DrawTag(TagScreen);
     MarkLine(TagScreen, y, x*13+15, (int)((int)WHITE << 4));
     redrawwin(TagScreen);
     wrefresh(TagScreen);
     break;
   case '+':
     Tmp = (tTag *)TagList.Base;
     while (Tmp != NULL){
      if (Tmp->Tag){
       mwriteblock(Tmp->Name, strlen(Tmp->Name));
       msendchar(13);
      }
      Tmp = (tTag *)Tmp->Next;
     }
     break;
   case '-':
     Tmp = (tTag *)TagList.Base;
     while (Tmp != NULL){
      if (Tmp->Tag){
       mwriteblock(Tmp->Name, strlen(Tmp->Name));
       msendchar(32);
      }
      Tmp = (tTag *)Tmp->Next;
     }
     break;
   case KEY_DOWN:
   TAGDOWN:
     if (y<LINES-3){
      Tmp = GetTag(y, x);
      if (Tmp != NULL)
      if (Tmp->Tag)
       MarkLine(TagScreen, y, x*13+15, Integers[cMenuHot]);
      else
       MarkLine(TagScreen, y, x*13+15, WHITE);
      else
       MarkLine(TagScreen, y, x*13+15, WHITE);
      y++;
      MarkLine(TagScreen, y, x*13+15, (int)((int)WHITE << 4));
     }
     break;
   case KEY_UP:
     if (y>1){
      Tmp = GetTag(y, x);
      if (Tmp != NULL)
      if (Tmp->Tag)
       MarkLine(TagScreen, y, x*13+15, Integers[cMenuHot]);
      else
       MarkLine(TagScreen, y, x*13+15, WHITE);
      else
       MarkLine(TagScreen, y, x*13+15, WHITE);
      y--;
      MarkLine(TagScreen, y, x*13+15, (int)((int)WHITE << 4));
     }
     break;
   case KEY_RIGHT:
     if (x<4){
      Tmp = GetTag(y, x);
      if (Tmp != NULL)
      if (Tmp->Tag)
       MarkLine(TagScreen, y, x*13+15, Integers[cMenuHot]);
      else
       MarkLine(TagScreen, y, x*13+15, WHITE);
      else
       MarkLine(TagScreen, y, x*13+15, WHITE);
      x++;
      MarkLine(TagScreen, y, x*13+15, (int)((int)WHITE << 4));
     }
     break;
   case KEY_LEFT:
     if (x>0){
      Tmp = GetTag(y, x);
      if (Tmp != NULL)
      if (Tmp->Tag)
       MarkLine(TagScreen, y, x*13+15, Integers[cMenuHot]);
      else
       MarkLine(TagScreen, y, x*13+15, WHITE);
      else
       MarkLine(TagScreen, y, x*13+15, WHITE);
      x--;
      MarkLine(TagScreen, y, x*13+15, (int)((int)WHITE << 4));
     }
     break;
   case KEY_NPAGE:
     Tmp = GetTag(y, x);
     if (Tmp != NULL)
     if (Tmp->Tag)
      MarkLine(TagScreen, y, x*13+15, Integers[cMenuHot]);
     else
      MarkLine(TagScreen, y, x*13+15, WHITE);
     else
      MarkLine(TagScreen, y, x*13+15, WHITE);
     y = LINES-3;
     MarkLine(TagScreen, y, x*13+15, (int)((int)WHITE << 4));
     break;
   case KEY_PPAGE:
     Tmp = GetTag(y, x);
     if (Tmp != NULL)
     if (Tmp->Tag)
      MarkLine(TagScreen, y, x*13+15, Integers[cMenuHot]);
     else
      MarkLine(TagScreen, y, x*13+15, WHITE);
     else
      MarkLine(TagScreen, y, x*13+15, WHITE);
     y = 1;
     MarkLine(TagScreen, y, x*13+15, (int)((int)WHITE << 4));
     break;
   case KEY_HOME:
     Tmp = GetTag(y, x);
     if (Tmp != NULL)
     if (Tmp->Tag)
      MarkLine(TagScreen, y, x*13+15, Integers[cMenuHot]);
     else
      MarkLine(TagScreen, y, x*13+15, WHITE);
     else
      MarkLine(TagScreen, y, x*13+15, WHITE);
     y = 1;
     x = 0;
     MarkLine(TagScreen, y, x*13+15, (int)((int)WHITE << 4));
     break;
   case KEY_END:
     Tmp = GetTag(y, x);
     if (Tmp != NULL)
     if (Tmp->Tag)
      MarkLine(TagScreen, y, x*13+15, Integers[cMenuHot]);
     else
      MarkLine(TagScreen, y, x*13+15, WHITE);
     else
      MarkLine(TagScreen, y, x*13+15, WHITE);
     y = LINES-3;
     x = 4;
     MarkLine(TagScreen, y, x*13+15, (int)((int)WHITE << 4));
     break;
   default:break;
   }
   wrefresh(TagScreen);
 }
 delwin(TagScreen);
 clearok(Screen, TRUE);
 redrawwin(Screen);
 wrefresh(Screen);
}

void DumpEmu()
{
 int i;
 
 for (i=0;i<Emupos;i++)
 {
  swaddch(Screen, Emu[i]);
  Emu[i] = 0;
 }
 Emupos = 0;
}

void Help()
{
 WINDOW *Window;
 
 Window = newwin(LINES-1, COLS, 0, 0);
 
 curs_set(0);
 textAttr(Window, Integers[cHelp]);
 wbkgdset(Window, Window->_attrs);
 wclear(Window);
 box(Window, SVLINE, SHLINE);
 mvwaddstr(Window, 1, 1, "Help");
 mvwaddstr(Window, 2, 1, "Scrollback");
 mvwaddstr(Window, 3, 1, "Clear screen");
 mvwaddstr(Window, 4, 1, "Phonebook");
 mvwaddstr(Window, 5, 1, "Local Echo");
 mvwaddstr(Window, 6, 1, "Hangup");
 mvwaddstr(Window, 7, 1, "Image save");
 mvwaddstr(Window, 8, 1, "Capture");
 mvwaddstr(Window, 9, 1, "Send tags");
 mvwaddstr(Window, 10, 1, "Options");
 mvwaddstr(Window, 11, 1, "Device change");
 mvwaddstr(Window, 12, 1, "Quit");
 mvwaddstr(Window, 13, 1, "Strip high");
 mvwaddstr(Window, 14, 1, "Terminal type");
 mvwaddstr(Window, 15, 1, "Tag files");
 mvwaddstr(Window, 16, 1, "Quit + reset");
 mvwaddstr(Window, 17, 1, "EMSI chat");
 mvwaddstr(Window, 18, 1, "Send break");
 textAttr(Window, Integers[cHelpBold]);
 if (Integers[EscType] == 1)
 {
   mvwaddstr(Window, 1, 20, "Ctrl-Alt-A");
   mvwaddstr(Window, 2, 20, "Ctrl-Alt-B");
   mvwaddstr(Window, 3, 20, "Ctrl-Alt-C");
   mvwaddstr(Window, 4, 20, "Ctrl-Alt-D");
   mvwaddstr(Window, 5, 20, "Ctrl-Alt-E");
   mvwaddstr(Window, 6, 20, "Ctrl-Alt-H");
   mvwaddstr(Window, 7, 20, "Ctrl-Alt-I");
   mvwaddstr(Window, 8, 20, "Ctrl-Alt-L");
   mvwaddstr(Window, 9, 20, "Ctrl-Alt-N");
   mvwaddstr(Window, 10, 20, "Ctrl-Alt-W");
   mvwaddstr(Window, 11, 20, "Ctrl-Alt-P");
   mvwaddstr(Window, 12, 20, "Ctrl-Alt-Q");
   mvwaddstr(Window, 13, 20, "Ctrl-Alt-R");
   mvwaddstr(Window, 14, 20, "Ctrl-Alt-T");
   mvwaddstr(Window, 15, 20, "Ctrl-Alt-V");
   mvwaddstr(Window, 16, 20, "Ctrl-Alt-X");
   mvwaddstr(Window, 17, 20, "Ctrl-Alt-Y");
   mvwaddstr(Window, 18, 20, "Ctrl-Alt-Z");
 }
 else
 {
   mvwaddstr(Window, 1, 20, "Alt-A");
   mvwaddstr(Window, 2, 20, "Alt-B");
   mvwaddstr(Window, 3, 20, "Alt-C");
   mvwaddstr(Window, 4, 20, "Alt-D");
   mvwaddstr(Window, 5, 20, "Alt-E");
   mvwaddstr(Window, 6, 20, "Alt-H");
   mvwaddstr(Window, 7, 20, "Alt-I");
   mvwaddstr(Window, 8, 20, "Alt-L");
   mvwaddstr(Window, 9, 20, "Alt-N");
   mvwaddstr(Window, 10, 20, "Alt-W");
   mvwaddstr(Window, 11, 20, "Alt-P");
   mvwaddstr(Window, 12, 20, "Alt-Q");
   mvwaddstr(Window, 13, 20, "Alt-R");
   mvwaddstr(Window, 14, 20, "Alt-T");
   mvwaddstr(Window, 15, 20, "Alt-V");
   mvwaddstr(Window, 16, 20, "Alt-X");
   mvwaddstr(Window, 17, 20, "Alt-Y");
   mvwaddstr(Window, 18, 20, "Alt-Z");
 }
 textAttr(Window, Integers[cHelp]);
 mvwaddstr(Window, 5, 40, "Termite v0.2.0b1, 8-19-97");
 mvwaddstr(Window, 7, 40, "Programmer:");
 mvwaddstr(Window, 10, 40, "Credits:");
 textAttr(Window, Integers[cHelpBold]);
 mvwaddstr(Window, 8, 45, "Nadav Cohen");
 mvwaddstr(Window, 11, 45, "Nimrod Zimerman");
 mvwaddstr(Window, 12, 45, "Gil Megidish");
 mvwaddstr(Window, 13, 45, "Shay Cohen");
 mvwaddstr(Window, 14, 45, "Nimrod Carmi");
 wrefresh(Window);
 getEscChar();
 delwin(Window);
 clearok(Screen, TRUE);
 redrawwin(Screen);
 wrefresh(Screen);
 curs_set(1);
}

void InitVars()
{
 textAttr(Screen, Integers[cTerminal]);
 CurrentTerm.LocalEcho = Integers[GlobalLocalEcho];
 CurrentTerm.StripHigh = Integers[GlobalStripHigh];
 CurrentTerm.RcvdBSdest = Integers[BSdest];
 CurrentTerm.Baud = Integers[Bauds1+CurrentTerm.Device-1];
 CurrentTerm.Parity = Integers[Parities1+CurrentTerm.Device-1];
 CurrentTerm.Stopbits = Integers[Stopbits1+CurrentTerm.Device-1];
 CurrentTerm.Databits = Integers[Databits1+CurrentTerm.Device-1];
 Warned = 0;
 strcpy(RingStr, Strings[Ring]);
 ReplaceM(RingStr);
 RING[0]=strlen(RingStr);
 RING[1]=0;
}

void DoTerminal(int Device)
{
 int i, Key, Quit, j, k;
 char EmulationReturn[3];
 time_t Time, LastTime;
 int LastStatus;
 char Temp[250];
 int y, x;

 OpenLogfile();
 ClosePort = 1;
 LastStatus = linestatus();
 Printlogln("Start");
 if (CAR){
  Online = 1;
  Offline = 0;
  Printlogln("Carrier detected");
 }
 else
 {
  Online = 0;
  Offline = 1;
 }
 textAttr(Screen, Integers[cTerminal]);
 wclear(Screen);
 wmove(Screen, 0, 0);
 time(&Starttime);
 CurrentTerm.Device = Device;
 CurrentTerm.Parity = Integers[Parities1+Device-1];
 CurrentTerm.Baud = Integers[Bauds1+Device-1];
 CurrentTerm.Stopbits = Integers[Stopbits1+Device-1];
 CurrentTerm.Databits = Integers[Databits1+Device-1];
 strncpy(CurrentTerm.Name+1,"Manual mode", 11);
 CurrentTerm.Name[0] = 11;
 CurrentTerm.Terminal = 0;
 CurrentTerm.User = 1;
 CurrentTerm.LocalEcho = Integers[GlobalLocalEcho];
 CurrentTerm.StripHigh = Integers[GlobalStripHigh];
 CurrentTerm.RcvdBSdest = Integers[BSdest];
 nl();
 PrintStatus();
 nonl();
 CurrentBufferFilled = 0;
 wmove(Screen, 0, 0);
 waddstr(Screen, "Termite version 0.2.0b1, Copyright (C) 1996-1997 Nadav Cohen\n");
 waddstr(Screen, "Termite comes with ABSOLUTELY NO WARRANTY; for details read COPYING.\n");
 waddstr(Screen, "This is free software, and you are welcome to redistribute it\n");
 waddstr(Screen, "under certain conditions; read COPYING for details.\n\n");
 waddstr(Screen, "PLEASE REPORT BUGS TO: nadavc@ibm.net\n");
 if (Offline || Integers[ForceInit])
 for (i=0;i<10;i++){
   if (DeviceInit[CurrentTerm.Device-1][i] >= 0){
    if (DeviceInit[CurrentTerm.Device-1][i] == 0)
     strcpy(Temp, Strings[InitStr0]);
    else
     strcpy(Temp, Strings[InitStr1+DeviceInit[CurrentTerm.Device-1][i]-1]);
    ReplaceM(Temp);
    mwriteblock(Temp, strlen(Temp));
   }
 if (charwaiting() == TRUE){
  BytesRead = mreadblock((char *)&CurrentBuffer[CurrentBufferFilled], 128-CurrentBufferFilled);
  for (j=CurrentBufferFilled;j<CurrentBufferFilled+BytesRead;j++)
   if (CurrentBuffer[j] != '\r') swaddch(Screen, CurrentBuffer[j]);
  if (BytesRead == 128) CurrentBufferFilled = 0;
  wrefresh(Screen);
 }
 }
 Emupos = 0;
 Quit = 0;
 Warned = 0;
 CurrentBufferFilled = 0;
 time(&Time);
 LastTime = Time;
 Ansi = 1;
 Avatar = 1;
 vt100 = 0;
 strcpy(EMSI_CHT+1, "**EMSI_CHTF5D4\15");
 strcpy(EMSI_TCH+1, "**EMSI_TCH3C60\15");
 strcpy(EMSI_ACK+1, "**EMSI_ACKA490\15");
 EMSI_CHT[0]=0;
 EMSI_TCH[0]=0;
 EMSI_ACK[0]=0;
 strcpy(RingStr, Strings[Ring]);
 ReplaceM(RingStr);
 RING[0]=strlen(RingStr);
 RING[1]=0;
 CaptureOn = 0;
 CurrentEntry = NULL;
 PhonebookChanged = 0;
 Remote = newwin(LINES/2, COLS, 0, 0);
 Local = newwin(LINES/2-1, COLS, LINES/2+1, 0);
 textAttr(Remote, Integers[cRemote]);
 textAttr(Local, Integers[cLocal]);
 wbkgdset(Remote, ' ' | Remote->_attrs);
 wbkgdset(Local, ' ' | Local->_attrs);
 wmove(Remote, 0, 0);
 wmove(Local, 0, 0);
 scrollok(Remote, TRUE);
 scrollok(Local, TRUE);
 clearok(Remote, TRUE);
 clearok(Local, TRUE);
 CurrentTerm.Password[0] = 5;
 strcpy(CurrentTerm.Password+1, "Nopwd");
 j=0;
 if (NameOfDialEntry != NULL) CommandLineDialer();
 if (DialIndex > 0) CommandLineIndexDialer();
 if (NameOfDialEntry != NULL || DialIndex > 0){ 
  Dialer(TermHeader.Num);
  clearok(Screen, TRUE);
  redrawwin(Screen);
  wrefresh(Screen);
 }
 if (Integers[StartInPhone]) Phone();
 while (!Quit){
 if (chwait() == TRUE){
  Key = getEscChar();
  CommandKey = 1;
  switch(Key){
  case KEY_HOME:
    if (Online){
     Integers[StatusStartOn]++;
     if (Integers[StatusStartOn] == 3) Integers[StatusStartOn] = 0;
    }
    else{
     Integers[StatusStartOff]++;
     if (Integers[StatusStartOff] == 3) Integers[StatusStartOff] = 0;
    }
    PrintStatus();
    break;
  case KEY_END:
    if (Online){
     Integers[StatusEndOn]++;
     if (Integers[StatusEndOn] == 2) Integers[StatusEndOn] = 0;
    }
    if (Offline){
     Integers[StatusEndOff]++;
     if (Integers[StatusEndOff] == 1) Integers[StatusEndOff] = 0;
    }
    PrintStatus();
    break;
  case KEY_PPAGE:
    Upload();
    break;
  case KEY_NPAGE:
    Download();
    break;
  case KEY_UP:
    if (Ansi == 1 || vt100 == 1)
    {
     msendchar(0x1B);
     msendchar('[');
     msendchar('A');
    }
    else if (Avatar == 1)
    {
     msendchar(3);
    }
    break;
  case KEY_DOWN:
    if (Ansi == 1 || vt100 == 1)
    {
     msendchar(0x1B);
     msendchar('[');
     msendchar('B');
    }
    else if (Avatar == 1)
    {
     msendchar(4);
    }
    break;
  case KEY_RIGHT:
    if (Ansi == 1 || vt100 == 1)
    {
     msendchar(0x1B);
     msendchar('[');
     msendchar('C');
    }
    else if (Avatar == 1)
    {
     msendchar(6);
    }
    break;
  case KEY_LEFT:
    if (Ansi == 1 || vt100 == 1)
    {
     msendchar(0x1B);
     msendchar('[');
     msendchar('D');
    }
    else if (Avatar == 1)
    {
     msendchar(5);
    }
    break;
  case KEY_F(1):
  case KEY_F(2):
  case KEY_F(3):
  case KEY_F(4):
  case KEY_F(5):
  case KEY_F(6):
  case KEY_F(7):
  case KEY_F(8):
  case KEY_F(9):
  case KEY_F(10):
    for (j=1;j<11;j++)
     if (Key == KEY_F(j)){
       strcpy(Temp, Strings[Macro1+j-1]);
       MacroParser(Temp, Strings[User1+CurrentTerm.User-1], CurrentTerm.Password);
       mwriteblock(Temp, strlen(Temp));
     }
    break;
  case 27: 
     Key = getEscChar();
     if (Integers[EscType] == 0) 
      if (Key > 64 && Key < 91) Key -= 64;
      else if (Key > 96 && Key < 123) Key -= 96;
     CommandKey = 1;
     if (Key == 25){
      if (EMSI_CHT[0] != 15){
       EMSI_CHT[0] = 15;
       mwriteblock(EMSI_CHT+1, 15);
       wmove(Remote, 0 ,0);
       wmove(Local, 0, 0);
       textAttr(Remote, Integers[cRemote]);
       textAttr(Local, Integers[cLocal]);
       wbkgdset(Remote, ' ' | Remote->_attrs);
       wbkgdset(Local, ' ' | Local->_attrs);
       werase(Remote);
       werase(Local);
       wrefresh(Remote);
       wrefresh(Local);
      }
      else
      {
       EMSI_CHT[0]=0;
       EMSI_TCH[0]=15;
       mwriteblock(EMSI_TCH+1, 15);
       clearok(Screen, TRUE);
       redrawwin(Screen);
       wrefresh(Screen);
      }
     }
     else if (EMSI_CHT[0] != 15) 
     switch(Key){
     case 1:
       Help();
       break;
     case 2:
       Scrl = ScrollStart;
       for (j=0;j<LINES;j++)
        for (k=0;k<COLS;k++)
         mvwaddch(ScrollScreen, j, k, *Scrl++);
       Scrl = ScrollStart;
       clearok(ScrollScreen, TRUE);
       redrawwin(ScrollScreen);
       wrefresh(ScrollScreen);
       while (Key != ' '){
        Key = getEscChar();
        switch(Key){
        case 't':
        case 'T':
          Tagger(ScrollScreen);
          break;
        case 27:
          Key = getEscChar();
          if (Key == 9){
            nl();
            ImageSave(ScrollScreen);
            nonl();
          }
          break;
        case KEY_DOWN:
          if (Scrl+(LINES-1)*COLS<ScrollBuffer+ScrollBackLength) Scrl += COLS;
          break;
        case KEY_UP:
          if (Scrl>ScrollBuffer) Scrl -= COLS;
          break;
        case KEY_HOME:
        HOME:
          Scrl = ScrollBuffer;
          break;
        case KEY_END:
        END:
          Scrl = ScrollBuffer+ScrollBackLength-(LINES-1)*COLS;
          break;
        case KEY_PPAGE:
          if (Scrl>=ScrollBuffer+(LINES-1)*COLS) Scrl -= (LINES-1)*COLS;
          else goto HOME;
          break;
        case KEY_NPAGE:
          if (Scrl+(LINES-1)*COLS*2<ScrollBuffer+ScrollBackLength) Scrl += (LINES-1)*COLS;
          else goto END;
          break;
        default:break;
       }
       for (j=0;j<LINES-1;j++)
        for (k=0;k<COLS;k++)
         mvwaddch(ScrollScreen, j, k, *(Scrl+j*COLS+k));
       wrefresh(ScrollScreen);
       }
       redrawwin(Screen);
       wrefresh(Screen);
       break;
     case 3:
       swclear(Screen);
       swmove(Screen, 0, 0);
       wrefresh(Screen);
       break;
     case 4:
       Phone();
       break;
     case 5:
       CurrentTerm.LocalEcho ^= 1;
       Integers[GlobalLocalEcho] ^= 1;
       ShowEcho();
       break;
     case 8: 
       if (Integers[AskOnHang]){
        nl();
        curs_set(0);
        if (ChooseYesno(Screen, " Hangup? "))
         Hangup();
        curs_set(1);
        nonl();
       }
       else
        Hangup();
       break;
     case 9:
       nl();
       ImageSave(Screen);
       nonl();
       break;
     case 12:
       nl();
       OpenCapture(Screen);
       nonl();
       break;
     case 14:
       curs_set(0);
       SendTags();
       curs_set(1);
       break;
     case 16:
       nl();
       curs_set(0);
       ChangeDevice();
       curs_set(1);
       nonl();
       break;
     case 17:
       if (Integers[AskOnExit]){
        nl();
        curs_set(0);
        Quit = ChooseYesno(Screen, " Exit without closing port? ");
        curs_set(1);
        nonl();
       }
       else Quit=1;
       ClosePort = Quit ^ 1;
       break;
     case 18:
       CurrentTerm.StripHigh ^= 1;
       Integers[GlobalStripHigh] ^= 1;
       ShowStrip();
       break;
     case 20:
       nl();
       curs_set(0);
       CurrentTerm.Terminal = ChooseTerminal(Screen, CurrentTerm.Terminal);
       curs_set(1);
       nonl();
       SetTerminal();
       wrefresh(Screen);
       break;
     case 22:
       Tagger(Screen);
       break;
     case 23:
       nl();
       curs_set(0);
       Setup();
       InitVars();
       curs_set(1);
       nonl();
       break;
     case 24:
       if (Integers[AskOnExit]){
         nl();
         curs_set(0);
         Quit = ChooseYesno(Screen, " Exit? ");
         curs_set(1);
         nonl();
       }
       else Quit = 1;
       break;
     case 26:
       SendBreak();
       break;
     default:
       CommandKey = 0;
       break;
    }
    break;
  default:
    CommandKey = 0;
    break;
  }
  if (!CommandKey)
  if (EMSI_CHT[0] != 15){ 
   if (Integers[CaptureKeys] && CaptureOn)
    write(CaptureHandle, &Key, 1);
   if (Key == KEY_BACKSPACE) Key = '\b';
   if (CurrentTerm.LocalEcho){ 
    if (Key == '\r') swaddch(Screen, '\n');
    else if (Key != '\b') swaddch(Screen, Key);
   }
   msendchar(Key);
  }
  else
  {
   if (Key == KEY_BACKSPACE || Key == '\b'){
    getyx(Local, k, j);
    if (j>0){
     j--;
     wmove(Local, k, j);
     if (CurrentTerm.RcvdBSdest){
      waddch(Local, ' ');
      wmove(Local, k, j);
     }
     wrefresh(Local);
     msendchar('\b');
    }
   }
   else
   {
    if (Key != '\r') waddch(Local, Key);
    else{ 
     waddch(Local, '\n');
     msendchar('\r');
     Key = '\n';
    }
    msendchar(Key);
   }
  }
  }
 if (CAR && !Online){
  Online = 1;
  Offline = 0;
  time(&Starttime);
  SetTerminal();
  if (CurrentTerm.Device != Device){
   resetModem();
   closeModem();
   unlock(Strings[Device1+Device-1]);
   Device = CurrentTerm.Device;
   initModem(Strings[Device1+Device-1], Integers[Bauds1+Device-1]);
   setparms(CurrentTerm.Baud, CurrentTerm.Parity, CurrentTerm.Databits, CurrentTerm.Stopbits, 
            Integers[FlowCtrl1+Device-1]);
  }
  Printlogln("Online");
  Warned = 0;
 }
 if (!CAR && Online){
  Online = 0;
  Offline = 1;
  CurrentTerm.Parity = Integers[Parities1+CurrentTerm.Device-1];
  CurrentTerm.Baud = Integers[Bauds1+CurrentTerm.Device-1];
  CurrentTerm.Stopbits = Integers[Stopbits1+CurrentTerm.Device-1];
  CurrentTerm.Databits = Integers[Databits1+CurrentTerm.Device-1];
  strncpy(CurrentTerm.Name+1,"Manual mode", 11);
  setparms(CurrentTerm.Baud, CurrentTerm.Parity, CurrentTerm.Databits, CurrentTerm.Stopbits,
           Integers[FlowCtrl1+CurrentTerm.Device-1]);
  CurrentTerm.Name[0] = 11;
  Printlogln("Offline");
  Warned = 0;
  if (CaptureOn){
   write(CaptureHandle, CaptureStart, CapturePtr-CaptureStart);
   close(CaptureHandle);
  }
  CaptureOn = 0;
  if (CurrentEntry != NULL){
    CurrentEntry->Connects++;
    CurrentEntry->SecUsed+=Time-Starttime;
    CurrentEntry->Calcmin=(Time-Starttime) / 60;
    CurrentEntry = NULL;
    PhonebookChanged = 1;
  }
  if (Integers[PhoneBookOnDrop]) Phone();
 }
 if (charwaiting() == TRUE){
  BytesRead = mreadblock((char *)&CurrentBuffer[CurrentBufferFilled], 128-CurrentBufferFilled);
  j+=BytesRead;
  if (Integers[SaveControl] && CaptureOn){
   write(CaptureHandle, &CurrentBuffer[CurrentBufferFilled], BytesRead);
  }
  CurrentBufferFilled += BytesRead;
  for (BufferCounter=CurrentBufferFilled-BytesRead;BufferCounter<CurrentBufferFilled;BufferCounter++)
  {
   if (CurrentTerm.StripHigh) CurrentBuffer[BufferCounter] &= 0x7F;
   if (EMSI_CHT[0] == 15){
    if (EMSI_TCH[EMSI_TCH[0]+1] == CurrentBuffer[BufferCounter])
     EMSI_TCH[0]++;
    else
     EMSI_TCH[0]=0;
    if (CurrentBuffer[BufferCounter] != '\r') waddch(Remote, CurrentBuffer[BufferCounter]);
    wrefresh(Remote);
    wrefresh(Local);
    if (EMSI_TCH[0] == 15){ 
      EMSI_CHT[0]=0;
      clearok(Screen, TRUE);
      redrawwin(Screen);
      wrefresh(Screen);
     }
   }
   else
   {
   EmulationReturn[0]=SEND;
   EmulationReturn[1]=SEND;
   EmulationReturn[2]=SEND;
   UsedGetChar=0;
   if (Ansi){ 
    EmulationReturn[0] = ANSI(CurrentBuffer[BufferCounter]);
    if (EmulationReturn[0] == DUMP) DumpEmu();
   }
   if (Avatar){ 
    EmulationReturn[1] = AVATAR(CurrentBuffer[BufferCounter]);
    if (EmulationReturn[1] == DUMP) DumpEmu();
    if (UsedGetChar) BufferCounter--;
   }
   if (vt100){
    EmulationReturn[2] = VT100(CurrentBuffer[BufferCounter]);
    if (EmulationReturn[2] == DUMP) DumpEmu();
   }
   if ((Ansi && EmulationReturn[0] == SAVE) || (Avatar && EmulationReturn[1] == SAVE)
     || (vt100 && EmulationReturn[2] == SAVE)){
     Emu[Emupos++] = CurrentBuffer[BufferCounter];
     if (Emupos == 20) DumpEmu();
   }
   if (EmulationReturn[0] == SEND && EmulationReturn[1] == SEND && EmulationReturn[2] == SEND){
     for (j=0;j<5;j++){
      if (Signature[j][Signature[j][1]+2] == CurrentBuffer[BufferCounter])
       Signature[j][1]++;
      else
       Signature[j][1]=0;
      if (Signature[j][0] == Signature[j][1] && Signature[j][0] != 0){
        AutoDownload(j);
      }
     }
     if (*(RingStr+RING[1]) == CurrentBuffer[BufferCounter])
      RING[1]++;
     else
      RING[1]=0;
     if (EMSI_CHT[EMSI_CHT[0]+1]==CurrentBuffer[BufferCounter])
       EMSI_CHT[0]++;
     else
       EMSI_CHT[0]=0;
     if (EMSI_CHT[0] == 15){
       werase(Remote);
       werase(Local);
       wmove(Remote, 0, 0);
       wmove(Local, 0, 0);
       wrefresh(Remote);
       wrefresh(Local);
       mwriteblock(EMSI_ACK, 15);
     }
     if (CurrentBuffer[BufferCounter] == '\b'){
       getyx(Screen, k, j);
       if (j>0){
        j--;
        wmove(Screen, k, j);
        if (CurrentTerm.RcvdBSdest){
         waddch(Screen, ' ');
         wmove(Screen, k, j);
        }
       }
     }
     else
     if (CurrentBuffer[BufferCounter] != '\r' && CurrentBuffer[BufferCounter] != '\n') 
       swaddch(Screen, CurrentBuffer[BufferCounter]);
     if (CurrentBuffer[BufferCounter] == '\n'){
      getyx(Screen, y, x);
      x = 0;
      if (y < LINES-2){
       y++;
       swmove(Screen, y, x);
      }
      else swaddch(Screen, '\n');
     }
     }
   }
  }
  if (charwaiting() == FALSE && EMSI_CHT[0] != 15 || CurrentBufferFilled == 128){ 
   wrefresh(Screen);
  }
 }
 if (CurrentBufferFilled == 128){
  CurrentBufferFilled = 0;
 }
 time(&Time);
 if (Time != LastTime || LastStatus != linestatus()){
  PrintStatus();
  if ((Time-Starttime) / 60 >= Integers[OnlineWarn] && Online && !Warned) WarnOnline();
  LastTime = Time;
  LastStatus = linestatus();
 }
 if (RNG && RING[0] == RING[1]){
  RING[1]=0;
  linestatus();
  strcpy(Temp, Strings[AutoAnswerStr]);
  ReplaceM(Temp);
  if (Integers[AutoAnswer]){
   mwriteblock(Temp, strlen(Temp));
  }
  else
  {
   nl();
   curs_set(0);
   if (ChooseYesno(Screen, "Answer incomming call?") == 1)
    mwriteblock(Temp, strlen(Temp));
   RNG = 0;
   curs_set(1);
   nonl();
  }
 }
 }
 delwin(Local);
 delwin(Remote);
 Printlogln("End");
 CloseLog();
 if (CaptureOn){
  write(CaptureHandle, CaptureStart, CapturePtr-CaptureStart);
  close(CaptureHandle);
 }
 if (ClosePort) sethup(TRUE);
 else sethup(FALSE);
 if (PhonebookChanged) 
   switch(WhichBook){
   case 0:
    if (WriteTerminate(CurrentPhonebook) < 0) 
     WindowError(PhoneScreen, "Unable to write phonebook");
     break;
   case 1:
    if (WriteTelix(CurrentPhonebook) < 0)
     WindowError(PhoneScreen, "Unable to write phonebook");
     break;
   default:break;
   }
}
