/* File     : xenmenu.c
 * Author   : Karyl F. Stein <xenon@xenos.net>
 * Purpose  : Program designed to allow the easy creation of text-based menus.
 *            This program can also be used as a UN*X shell.  For more
 *            information, see the README file that should have come with this
 *            release.
 *
 * Xenmenu is Copyright (C)1996 Karyl F. Stein
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc., 675
 * Mass Ave, Cambridge, MA 02139, USA.
 */

#include "config.h"
#include "xenmenu.h"
#include "menu_func.h"
#include "opt_func.h"
#include "get_input.h"
#include "dochoice.h"
#include "rpmenu.h"
#include <stdio.h>
#include <unistd.h>      /* getopt(), geteuid() */
#include <pwd.h>         /* getpwuid()          */
#include <sys/types.h>   /* getpwuid()          */

/* Function Prototypes */
void usage(char *);
void version(void);


/* Function: usage
 * Input   : The command name of the program.
 * Output  : Print a usage summary and exit.
 */
void usage (char *name) {
  printf("Usage: %s [-v] [-h] [menu]\n", name);
  printf(" [menu] - Menu to load.\n");
  printf(" -v     - Print the version and default settings and exit.\n");
  printf(" -h     - Print this usage description and exit.\n");
  exit(0);
}


/* Function: version
 * Output  : Print a version message as well as default compile time settings
 *           and exit.
 */
void version (void) {
  puts("xenmenu v0.7a  COPYRIGHT (c)1996,1997 Karyl F. Stein");
#ifdef MENUDIR
  printf("Menu Directory             : %s\n", MENUDIR);
#endif
#ifdef MAINMENU
  printf("Default Menu               : %s\n", MAINMENU);
#endif
#ifdef SECURECONFIG
  printf("Secure Configuration File  : %s\n", SECURECONFIG);
#endif
#ifdef SYSCONFIG
  printf("System Configuration File  : %s\n", SYSCONFIG);
#endif
#ifdef USERCONFIG
  printf("User Configuration File    : %s\n", USERCONFIG);
#endif
#ifdef SECURERUN
  printf("Secure Executable Directory: %s\n", SECURERUN);
#endif
#ifdef VIEWDIR
  printf("Viewable Files Directory   : %s\n", VIEWDIR);
#endif
#ifdef SECURE
  printf("Security                   : %d\n", SECURE);
#endif
#ifdef DEFAULTPROMPT
  printf("Default Prompt             : %s\n", DEFAULTPROMPT);
#endif
#ifdef MENUHEAD
  printf("Default Menu Header        : %s\n", MENUHEAD);
#endif
  exit(0);
}


int main (int argc, char *argv[]) {
  char *choice, *input, *oldmenu = NULL, *menu = NULL;
  int opt;
  option_head options = NULL;
  struct passwd *pw;

  /* Parse the arguments */
  while ((opt = getopt(argc, argv, "hv")) != EOF) {
    switch (opt) {
    case 'v':
      version();
    default:
      usage(argv[0]);
    }
  }
  if (optind < argc) {
    if ((menu = (char *) malloc(strlen(argv[optind]) + 1)) == NULL) {
      fprintf(stderr, "Out Of Memory\n");
      exit(1);
    }
    strcpy(menu, argv[optind]);
  } else {
#ifndef MAINMENU
    fprintf(stderr, "You must specify a menu\n");
    exit(1);
#else
    if ((menu = (char *) malloc(strlen(MAINMENU) + 1)) == NULL) {
      fprintf(stderr, "Out Of Memory\n");
      exit(1);
    }
    strcpy(menu, MAINMENU);
#endif  /* MAINMENU */
  }

  /* Read the environment configure files */
#ifdef SYSCONFIG
  parseenv(SYSCONFIG);
#endif  /* SYSCONFIG */
#ifdef USERCONFIG
  if ((pw = getpwuid(geteuid())) == NULL) {
    fprintf(stderr, "Unable to find your user name\n");
    exit(1);
  }
  if ((input = (char *) malloc(strlen(pw->pw_dir) + strlen(USERCONFIG) + 2))
      == NULL) {
    fprintf(stderr, "Out Of Memory\n");
    exit(1);
  }
  sprintf(input, "%s/%s", pw->pw_dir, USERCONFIG);
  parseenv(input);
  free(input);
#endif  /* USERCONFIG */
#ifdef SECURECONFIG
  parseenv(SECURECONFIG);
#endif  /* SECURECONFIG */

  /* Main loop */
  while (1) {
    free(clearlist(options));
    if ((options = rpmenu(menu)) == NULL) {

#ifdef MAINMENU
      /* Exit if on the default menu, (see config.h to set MAINMENU). */
      /* We test here to avoid possible infinate looping.             */
      if (strcmp(menu, MAINMENU) == 0)
	exit(1);
#endif  /* MAINMENU */

      /* Reload known good menu */
      free(menu);
      if (oldmenu == NULL) {
#ifdef MAINMENU
	if ((menu = (char *) malloc(strlen(MAINMENU) + 1)) == NULL) {
	  fprintf(stderr, "Out Of Memory\n");
	  exit(1);
	}
	strcpy(menu, MAINMENU);
#else
	exit(1);
#endif  /* MAINMENU */
      } else {
	menu = oldmenu;
	oldmenu = NULL;
      }
    } else {

      /* Get user input */
#ifdef DEFAULTPROMPT
      if (options->prompt == NULL)
	choice = get_input(DEFAULTPROMPT, MAXINPUT, 0, TEXT);
      else choice = get_input(options->prompt, MAXINPUT, 0, TEXT);
#else
      choice = get_input(options->prompt, MAXINPUT, 0, TEXT);
#endif

      /* Act on input */
      if ((choice != NULL) && ((input = dochoice(options, choice)) != NULL)) {
	oldmenu = menu;
	if ((menu = (char *) malloc(strlen(input) + 1)) == NULL) {
	  fprintf(stderr, "Fatal Error: Out Of Memory\n");
	  exit(1);
	}
	strcpy(menu, input);
      }
    }
  }  /* while (1) */
}
