/* File     : rpmenu.c
 * Author   : Karyl F. Stein <xenon@xenos.net>
 * Purpose  : Functions for reading and printing xenmenu menus.  The
 *            individual functions are detailed below.
 *
 * Xenmenu is Copyright (C)1996 Karyl F. Stein
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc., 675
 * Mass Ave, Cambridge, MA 02139, USA.
 */

#include "config.h"     /* Configuration       */
#include "rpmenu.h"     /* Function prototypes */
#include "xenmenu.h"    /* TRUE/FALSE          */
#include "menu_func.h"  /* Various prototypes  */

/* Set defaults if not defined previously or not valid.  Do not change */
/* here, but in config.h.                                              */
#ifndef SECURE
#define SECURE 0
#endif  /* SECURE */
#ifndef DEF_OPTTAIL
#define DEF_OPTTAIL "."
#endif  /* DEF_OPTTAIL */


/* Function: rpmenu
 * Input   : Filename of menu to parse
 * Output  : Menu printed to screen
 * Return  : TRUE if there were no errors, otherwise FALSE.
 */
option_head rpmenu (char *name) {
  char *arguments, *filename, *input, *mheader = NULL, *mname = NULL;
  FILE *ifp = NULL;
  int choice = 0, columns = 1, rerror = FALSE, number = 0, pos = 0;
  option_head retval = newlist();
  option_node new_node;

  /* Build the file name to open */
  if (SECURE & 2)
    input = stripdots(name);
  else input = name;

#ifdef MENUDIR
  if ((SECURE & 2) || (*input != '/')) {
    if ((filename = (char *) malloc(strlen(MENUDIR) + strlen(input) + 2)) ==
	NULL) {
      fprintf(stderr, "Out Of Memory\n");
      exit(1);
    }
    sprintf(filename, "%s/%s", MENUDIR, input);
  } else {
    if ((filename = (char *) malloc(strlen(input) + 1)) == NULL) {
      fprintf(stderr, "Out Of Memory\n");
      exit(1);
    }
    strcpy(filename, input);
  }
#else
  if ((filename = (char *) malloc(strlen(input) + 1)) == NULL) {
    fprintf(stderr, "Out Of Memory\n");
    exit(1);
  }
  strcpy(filename, input);
#endif

  if (SECURE & 2)
    free(input);

  /* Try to open menu first in the current directory, (if security */ 
  /* permits), and then in the default menu directory.             */
  if ((~SECURE & 2) && ((ifp = fopen(name, "r")) != NULL)) {
    free(filename);
    if ((filename = (char *) malloc(strlen(name) + 1)) == NULL) {
      fprintf(stderr, "Out Of Memory\n");
      exit(1);
    }
    strcpy(filename, name);
  } else if ((ifp = fopen(filename, "r")) == NULL) {
    error("ERROR: Unable to load menu", name, 0);
    free(filename);
    return(FALSE);
  }

  /* Print the menu */
  line = 0;
  clearscr();
  while ((input = get_option(ifp)) != NULL) {
    arguments = get_arguments(ifp);

    /* Set the menu header value */
    if (strcmp(input, "header") == 0) {
      free(mheader);
      mheader = arguments;
    }

    /* Set the menu name */
    else if (strcmp(input, "name") == 0) {
      free(mname);
      mname = arguments;
    }

    /* Set the option tail */
    else if (strcmp(input, "opttail") == 0) {
      free(retval->opttail);
      retval->opttail = arguments;
    }

    /* Set the menu prompt */
    else if (strcmp(input, "prompt") == 0) {
      free(retval->prompt);
      retval->prompt = arguments;
    }

    else {

      /* Center the passed string */
      if (strcmp(input, "center") == 0) {
	if (pos <= choice)
	  if ((pos = printfrom(retval, pos, columns)) == -1) {
	    rerror = TRUE;
	    break;
	  }
	center(arguments);
      }

      /* Set option scanning to be case sensitive */
      else if (strcmp(input, "checkcase") == 0) {
	if (retval->list != NULL) {
	  error("Syntax Error: Option checkcase must be defined before any options", filename, line);
	  rerror = TRUE;
	  break;
	}
	retval->flag |= CHECKCASE;
      }

      /* Set the number of columns */
      else if (strcmp(input, "columns") == 0) {
	if (arguments == NULL) {
	  error("Syntax Error: Columns declared but not defined",
		filename, line);
	  rerror = TRUE;
	  break;
	}
	if ((columns = atoi(arguments)) == 0) {
	  error("Syntax Error: Invalid column argument", filename, line);
	  rerror = TRUE;
	  break;
	}
      }

      /* Set option scanning to be case insensitive */
      else if (strcmp(input, "nocheckcase") == 0) {
	if (retval->list != NULL) {
	  error("Syntax Error: Option nocheckcase must be defined before any options", filename, line);
	  rerror = TRUE;
	  break;
	}
	retval->flag &= !CHECKCASE;
      }

      /* Do not print the option type */
      else if (strcmp(input, "notype") == 0)
	retval->flag |= NO_TYPE;

      else if (strcmp(input, "option") == 0) {
	if ((arguments == NULL) || (strcmp(arguments, "{") != 0)) {
	  error("Syntax Error: Invalid Option Declaration", filename, line);
	  rerror = TRUE;
	  break;
	}
	if ((new_node = read_option(ifp)) == NULL) {
	  error("Syntax Error: Invalid Option Declaration", filename, line);
	  rerror = TRUE;
	  free(clearnode(new_node));
	  break;
	}

	if (~new_node->flag & NO_PRINT)
	  ++choice;

	/* Create a option value for the new option if it does not already */
	/* exist otherwise check to make sure the option value is valid.   */
	if (new_node->value == NULL) {
	  new_node->value = itoa(++number);
	  while (scanvalue(retval, new_node->value) != NULL) {
	    free(new_node->value);
	    new_node->value = itoa(++number);
	  }
	} else if (scanvalue(retval, new_node->value) != NULL) {
	  error("Error in menu: Option value exists",
		new_node->value, line);
	  rerror = TRUE;
	  free(clearnode(new_node));
	  break;
	}

	retval = addoption(retval, new_node);
	++retval->size;
      }

      /* Print a string */
      else if (strcmp(input, "print") == 0) {
	if (pos <= choice)
	  if ((pos = printfrom(retval, pos, columns)) == -1) {
	    rerror = TRUE;
	    break;
	  }
	printstr(arguments);
      }

      /* Print a file */
      else if (strcmp(input, "printfile") == 0) {
	if (pos <= choice)
	  if ((pos = printfrom(retval, pos, columns)) == -1 ) {
	    rerror = TRUE;
	    break;
	  }
	printfile(arguments, FALSE);
      }

      /* Print a menu header */
      else if (strcmp(input, "printheader") == 0) {
	if (pos <= choice)
	  if ((pos = printfrom(retval, pos, columns)) == -1) {
	    rerror = TRUE;
	    break;
	  }
	printhead(mheader, mname);
      }

      /* Print a line */
      else if (strcmp(input, "printline") == 0) {
	if (pos <= choice)
	  if ((pos = printfrom(retval, pos, columns)) == -1) {
	    rerror = TRUE;
	    break;
	  }
	printlineof(getcols(), arguments);
      }

      /* Run an external program */
      else if (strcmp(input, "run") == 0) {
	if (pos <= choice)
	  if ((pos = printfrom(retval, pos, columns)) == -1) {
	    rerror = TRUE;
	    break;
	  }
	runprog(arguments, FALSE);
      }

      /* Print the option type */
      else if (strcmp(input, "type") == 0) {
	retval->flag &= ~NO_TYPE;
      }

      else { 
	error("Syntax Error: Invalid option", input, line);
	rerror = TRUE;
	break;
      }

      free(arguments);
    }

    free(input);
  }

  if ((pos <= choice) && (rerror == FALSE))
    if ((pos = printfrom(retval, pos, columns)) == -1)
      rerror = TRUE;
  close(ifp);
  free(filename);
  free(mheader);
  free(mname);
  free(input);
  if (rerror == TRUE) {
    free(clearlist(retval));
    return(NULL);
  }
  return(retval);
}


/* Function: printfrom
 * Input   : A pointer to a header node, a position from which to start
 *           printing, and the number of columns in which to print options.
 * Output  : Options printed to screen.
 * Return  : New position value.
 */
int printfrom (option_head options, int pos, int cols) {
  option_node ptr, sptr;
  int nspace = 0, space = 0, tmpcol = 0, tmpint;

  if (pos < 0) {
    fprintf(stderr, "Corrupt Option List\n");
    return(-1);
  }

  if ((options == NULL) || (options->list == NULL))
    return(0);
  ptr = options->list;
  if (pos == 0)
    pos = 1;

  /* Find the position to start printing from */
  tmpint = pos;
  while (--tmpint > 0) {
    if (ptr == NULL) {
      error("ERROR in menu: Invalid position", NULL, -1);
      return(-1);
    }
    ptr = ptr->next;
  }

  tmpint = plistsize(ptr);
  if (tmpint == 0)
    return(listsize(ptr) + pos);
  if (tmpint < cols)
    cols = tmpint;
  space = (getcols() - cols) / cols;
  if (options->opttail == NULL)
    space -= strlen(DEF_OPTTAIL);
  else space -= strlen(options->opttail);
  tmpcol = cols;
  tmpint = listsize(ptr);

  /* Find maxium name size */
  for (sptr = ptr; sptr != NULL; sptr = sptr->next) {
    if ((sptr->flag != 1) && (sptr->name != NULL))
      if (strlen(sptr->name) > nspace)
	nspace = strlen(sptr->name);
  }

  /* Print the options */
  while (tmpint-- > 0) {
    ++pos;
    if (ptr->flag != 1) {

      /* Print the option value */
      if (ptr->value == NULL) {
	error("Fatal Error: Corrupt Option List", NULL, -1);
	return(-1);
      } else {
	printf("%s", ptr->value);
	if (options->opttail == NULL)
	  printf("%s ", DEF_OPTTAIL);
	else printf("%s ", options->opttail);
      }

      /* Print the option name and pad with spaces */
      if (space < nspace)
	nspace = space;
      printopt(ptr, space - 3, nspace, options->flag);
      if (--tmpcol <= 0) {
	putc('\n', stdout);
	tmpcol = cols;
      }
      else putc(' ', stdout);
    }

    ptr = ptr->next;
  }

  /* Add a new line if necessary and return the new position */
  if (tmpcol != cols)
    putc('\n', stdout);
  return(pos);
}


/* Function: printopt
 * Input   : An option node to print, the number of columns available in which
 *           to print the option, the number of columns available in which to
 *           print the option name, and a flag.
 * Output  : The option is printed to the screen.
 * Return  : TRUE if successful, otherwise FALSE.
 */
int printopt (option_node node, int space, int nspace, int flag) {
  char *outstr;

  if (space <= 0)
    return(FALSE);

  if (strlen(node->name) > nspace) {
    if ((outstr = (char *) malloc(nspace + 1)) == NULL) {
      fprintf(stderr, "Fatal Error: Out Of Memory\n");
      exit(1);
    }
    strncpy(outstr, node->name, nspace);
  } else {
    if ((outstr = (char *) malloc(strlen(node->name) + 1)) == NULL) {
      fprintf(stderr, "Fatal Error: Out Of Memory\n");
      exit(1);
    }
    strcpy(outstr, node->name);
  }

  printf("%s", outstr);
  space -= strlen(outstr);
  nspace -= strlen(outstr);
  while ((nspace-- > 0) && (space-- > 0))
    putc(' ', stdout);

  /* Print other fields if possible */
  if (space >= 7) {
    if (~flag & NO_TYPE) {
      if (node->type == MENU_T)
	printf(" (menu)");
      else if (node->type == FILE_T)
	printf(" (file)");
      else if (node->type == RUN_T)
	printf(" (prog)");
      else printf("       ");
      space -= 7;
    }
    if ((space > 5) && (node->comment != NULL)) {
      free(outstr);
      if ((outstr = (char *) malloc(space + 1)) == NULL) {
	fprintf(stderr, "Fatal Error: Out Of Memory\n");
	exit(1);
      }
      strncpy(outstr, node->comment, space);
      if (strlen(node->comment) >= space) {
	outstr[space] = '\0';
	outstr[space - 1] = '.';
	outstr[space - 2] = '.';
	outstr[space - 3] = '.';	
      }
      printf(" %s", outstr);
      space -= strlen(outstr);
      space -= 1;
    }
  }

  while (space-- > 0)
    putc(' ', stdout);

  free(outstr);
  return(TRUE);
}
