/* File     : opt_func.c
 * Author   : Karyl F. Stein <steinkf@cs.purdue.edu>
 * Purpose  : Functions for handling option lists for xenmenu.  The individual
 *            functions are described below.
 *
 * Xenmenu is Copyright (C)1996 Karyl F. Stein
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc., 675
 * Mass Ave, Cambridge, MA 02139, USA.
 */

#include "opt_func.h"
#include "xenmenu.h"
#include "menu_func.h"


/* Function: addoption
 * Purpose : Add a new option node to the options list.
 * Input   : A node to add to the option list.
 */
option_head addoption (option_head head, option_node node) {
  option_node ptr;

  if ((head == NULL) || (node == NULL))
    return(NULL);

  if (head->list == NULL)
    head->list = node;
  else {
    for (ptr = head->list; ptr->next != NULL; ptr = ptr->next);
    ptr->next = node;
  }

  return(head);
}


/* Function: clearlist
 * Input   : A list to clear.
 * Output  : The passed option list is cleared.
 */
option_head clearlist (option_head head) {
  option_node aptr, bptr;

  if (head == NULL)
    return(NULL);
  aptr = head->list;

  /* Clear each node in the option list */
  while (aptr != NULL) {
    bptr = aptr;
    aptr = aptr->next;
    free(clearnode(bptr));
  }

  /* Clear the option list header node */
  free(head->prompt);
  head->prompt = NULL;
  free(head->opttail);
  head->opttail = NULL;
  head->list = NULL;
  head->size = 0;
  head->flag = 0;
}


/* Function: clearnode
 * Input   : A node to clear.
 * Output  : The passed node is freed.
 */
option_node clearnode (option_node node) {
  if (node == NULL)
    return(NULL);

  free(node->name); 
  node->name = NULL;
  free(node->value);
  node->value = NULL;
  free(node->comment);
  node->comment = NULL;
  free(node->file);
  node->file = NULL;
  return(node);
}


/* Function: listsize
 * Purpose : Return the size of the passed option list.
 * Input   : The list of which to find the size.
 * Return  : The size of the list.
 */
int listsize (option_node list) {
  option_node ptr = list;
  int retval = 0;

  while (ptr != NULL) {
    ptr = ptr->next;
    ++retval;
  }

  return(retval);
}


/* Function: newlist
 * Output  : The option list is initialized.
 */
option_head newlist (void) {
  option_head retval;

  if ((retval = (option_head) malloc(sizeof(struct option_head_t))) == NULL) {
    fprintf(stderr, "Out Of Memory\n");
    exit(1);
  }

  retval->opttail = NULL;
  retval->prompt = NULL;
  retval->flag = 0;
  retval->size = 0;
  retval->type = NO_T;
  retval->list = NULL;

  return(retval);
}


/* Function: newnode
 * Return  : A new and initialized option node.
 */
option_node newnode (void) {
  option_node retval;

  if ((retval = (option_node) malloc(sizeof(struct option_node_t))) == NULL) {
    fprintf(stderr, "Out Of Memory\n");
    exit(1);
  }

  retval->comment = NULL;
  retval->file = NULL;
  retval->name = NULL;
  retval->value = NULL;
  retval->type = NO_T;
  retval->flag = 0;
  retval->next = NULL;

  return(retval);
}


/* Function: plistsize
 * Purpose : Count the number of printable options in the passed list.
 * Input   : The list to scan.
 * Return  : The number of printable options in the list.
 */
int plistsize (option_node list) {
  option_node ptr = list;
  int retval = 0;

  while (ptr != NULL) {
    if (~ptr->flag & NO_PRINT)
      ++retval;
    ptr = ptr->next;
  }

  return(retval);
}


/* Function: read_option
 * Input   : An open file pointer from which to read.
 * Return  : An option node filled with the new data or NULL if error.
 */
option_node read_option (FILE *ifp) {
  char *input;
  option_node retval = newnode();

  while ((input = get_option(ifp)) != NULL) {
    if (strcmp(input, "}") == 0) {
      if (testopt(retval) == FALSE)
	break;
      else {
	free(input);
	return(retval);
      }
    }

    /* Input the option name.  Rewrite over the old name value, (if any). */
    if (strcmp(input, "name") == 0) {
      free(retval->name);
      retval->name = get_arguments(ifp);
    }

    /* Input the option comment.  Overwrite over the old value if any. */
    else if (strcmp(input, "comment") == 0) {
      free(retval->comment);
      retval->comment = get_arguments(ifp);
    }

    /* Set the option to be an exit condition */
    else if (strcmp(input, "exit") == 0)
      retval->type = EXIT_T;

    /* Input the option filename.  Overwrite the old value if any. */
    else if (strcmp(input, "file") == 0) {
      free(retval->file);
      retval->file = get_arguments(ifp);
      retval->type = FILE_T;
    }

    /* Set the option to be a menu to load.  Overwrite the old value if any. */
    else if (strcmp(input, "menu") == 0) {
      free(retval->file);
      retval->file = get_arguments(ifp);
      retval->type = MENU_T;
    }

    /* Set the option to not be printed to the screen */
    else if (strcmp(input, "noprint") == 0)
      retval->flag |= NO_PRINT;

    /* Set the option as a program to run.  Overwrite the old value if any. */
    else if (strcmp(input, "run") == 0) {
      free(retval->file);
      retval->file = get_arguments(ifp);
      retval->type = RUN_T;
    }

    /* Input the option value.  Overwrite the old value if any. */
    else if (strcmp(input, "value") == 0) {
      free(retval->value);
      retval->value = get_arguments(ifp);
    }

    else break;

    free(input);
  }

  free(input);
  free(clearnode(retval));
  return(NULL);
}


/* Function: scanvalue
 * Purpose : Scan for a passed value in a passed list.
 * Input   : A list to scan and a value to scan for.
 * Return  : Pointer to node containing the value or NULL if not found.
 */
option_node scanvalue (option_head head, char *scanval) {
  char *aval = NULL, *bval = NULL;
  option_node ptr;

  if (scanval == NULL)
    return(NULL);

  /* Test if list is empty */
  if ((head == NULL) || (head->list == NULL))
    return(NULL);

  /* Scan the list */
  for (ptr = head->list; ptr != NULL; ptr = ptr->next) {
    if (ptr->value == NULL)
      continue;

    /* Convert the test values to lowercase if needed */
    if (~head->flag & CHECKCASE) {
      if ((aval = (char *) malloc(strlen(scanval) + 1)) == NULL) {
	fprintf(stderr, "Out Of Memory\n");
	exit(1);
      }
      strcpy(aval, scanval);
      aval = lowercase(aval);
      if ((bval = (char *) malloc(strlen(ptr->value) + 1)) == NULL) {
	fprintf(stderr, "Out Of Memory\n");
	exit(1);
      }
      strcpy(bval, ptr->value);
      bval = lowercase(bval);
      if (strcmp(aval, bval) == 0)
	break;
      free(aval);
      free(bval);
      aval = NULL;
      bval = NULL;
    } else if (strcmp(ptr->value, scanval) == 0)
      break;
  }

  free(aval);
  free(bval);
  return(ptr);
}


/* Function: testopt
 * Input   : The option node to test.
 * Return  : TRUE if it is a valid node, otherwise FALSE.
 */
int testopt (option_node node) {
  if (node->type == NO_T)
    return(FALSE);
  if ((node->name == NULL) && (~node->flag & NO_PRINT))
    return(FALSE);
  if ((node->type != EXIT_T) && (node->file == NULL))
    return(FALSE);

  return(TRUE);
}
