/* File     : menufunc.c
 * Author   : Karyl F. Stein <xenon@xenos.net>
 * Purpose  : Functions for use with the xenmenu program.  Descriptions of the
 *            individual functions are given before the function.
 *
 * Xenmenu is Copyright (C)1996 Karyl F. Stein
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc., 675
 * Mass Ave, Cambridge, MA 02139, USA.
 */

#include "config.h"
#include "xenmenu.h"
#include "menu_func.h"
#include "get_input.h"
#include <stdio.h>
#include <stdlib.h>      /* getenv()/realloc() */
#include <string.h>      /* strchr()           */
#include <ctype.h>       /* tolower()          */
#include <unistd.h>      /* fork()             */
#include <sys/types.h>   /* waitpid()          */
#include <sys/wait.h>    /* waitpid()          */
#include <sys/ioctl.h>   /* ioctl()            */
#include <sys/termios.h> /* struct winsize     */

/* Values for the flag variable (used in printstr) */
#define NOCR    0x01
#define INQUOTE 0x02

/* Set defaults if not defined previously or not valid.  Do not change */
/* here, but in the file config.h.                                     */
#ifndef SECURE
#define SECURE 0
#endif  /* SECURE */
#ifndef INBUF
#define INBUF 512
#elif (INBUF < 2)
#define INBUF 512
#endif  /* INBUF */


/* Function: center
 * Input   : String to center.
 * Output  : The passed string printed on the center of the screen.  Word
 *           wrapping is done if necessary.
 */
void center(char *str) {
  char *ptr;
  char c;
  int cols, tmpint;

  /* Return if no string passed to print */
  if (str == NULL)
    return;

  /* Wrap the string onto multiple lines of needed */
  cols = getcols();
  if ((tmpint = cols - strlen(str)) < 0) {
    c = str[cols - 1];
    str[cols - 1] = '\0';
    if (((ptr = strrchr(str, ' ')) == NULL) ||
	((ptr = strrchr(str, '\t')) == NULL) ||
	(ptr = strrchr(str, '\n')));
    str[cols - 1] = c;

    /* Break the string up as needed and recursively center each section */
    if (ptr != NULL) {
      *ptr = '\0';
      center(str);
      *ptr = ' ';
      center(ptr + 1);
      return;
    } else {
      if (((ptr = strchr(str, ' ')) == NULL) &&
	  ((ptr = strchr(str, '\t')) == NULL) &&
	  ((ptr = strchr(str, '\n')) == NULL))
	printf("%s\n", str);
      else {
	c = *ptr;
	*ptr = '\0';
	center(str);
	*ptr = c;
	center(ptr + 1);
      }
    }
  } else {
    tmpint /= 2;
    while (tmpint-- > 0)
      putchar(' ');
    printf("%s\n", str);
  }
}


/* Function: clearscr
 * Output  : Clear the screen
 */
void clearscr(void) {
#ifdef CLEAR
  int pid, status;

  if ((pid = fork()) == -1) {
    fprintf(stderr, "Unable to fork\n");
    exit(1);
  }
  if (pid == 0) {
    if (execl(CLEAR, get_name(CLEAR), NULL) == -1)
      fprintf(stderr, "Unable to execute %s\n", CLEAR);
  } else waitpid(pid, &status, 0);
#endif  /* CLEAR */
}


/* Function: error
 * Input   : Message to print, a debugging message, and a line number.
 * Output  : Print the passed message and debugging message.  If the debugging
 *           message is NULL, it and the line number will not be printed.
 */
void error (char *message, char *where, int line) {
  clearscr();
  printf("Menu Error\n");
  printline(getcols());
  putc('\n', stdout);
  center(message);
  putc('\n', stdout);
  if (where != NULL)
    printf("Menu: %s\n", where);
  if (line != 0)
    printf("Line: %d\n", line);
  putc('\n', stdout);
  printline(getcols());
  waitkey();
}


/* Function: getcols
 * Purpose : Return the number of columns on a user's screen.
 * Return  : Number of columns on the user's scren.
 */
int getcols (void) {
  struct winsize win;

  if ((ioctl(1, TIOCGWINSZ, &win)) == -1)
    return(80);
  return(win.ws_col);
}


/* Function: get_arguments
 * Input   : An open file pointer from which to read.
 * Return  : Everything until EOF, a new line, or a comment is reached.
 */
char *get_arguments (FILE *ifp) {
  char c;
  char *inbuf = NULL, *retval;
  extern int line;
  int count = 0, flag = 0, size = 0;

  /* Skip over any leading white space */
  skip(ifp, SPACE);

  /* Read and store input until whitespace or EOF is found */
  while ((c = getc(ifp)) != EOF) {
    if (c == '\n')
      break;
    if ((flag) && (c == '#')) {
      skip(ifp, TO_EOL);
      break;
    }
    if ((c == ' ') || (c == '\t'))
      flag = 1;
    else flag = 0;
    if ((++count >= size) &&
        ((inbuf = (char *) realloc(inbuf, size += INBUF)) == NULL)) {
      fprintf(stderr, "Out of Memory\n");
      exit(1);
    }
    inbuf[count - 1] = c;
  }

  if (count == 0)
    return(NULL);

  ++line;
  inbuf[count] = '\0';

  /* Return the data read */
  if ((retval = (char *) malloc(strlen(inbuf) + 1)) == NULL) {
    fprintf(stderr, "Out Of Memory\n");
    exit(1);
  }
  strcpy(retval, inbuf);
  free(inbuf);
  return(retval);
}


/* Function: get_name
 * Input   : A file name including a path
 * Return  : A pointer to the command name, (the path is stripped).
 */
char *get_name (char *str) {
  char *retval;

  if (str == NULL)
    return(NULL);

  if ((retval = strrchr(str, '/')) == NULL)
    return(str);
  else return(retval);
}


/* Function: get_option
 * Input   : A file pointer to read data from.
 * Ouptut  : The passed file pointer is moved over all comments and white
 *           space, and the first option found.
 * Return  : The next command in the file, or NULL if error or no option
 *           found.
 */
char *get_option (FILE *ifp) {
  char c;
  int count = 0, size = 0;
  char *inbuf = NULL, *retval;

  /* Skip over any leading comments or white space */
  skip(ifp, COMMENTS | WHITE);

  /* Read and store input until whitespace or EOF is found */
  while ((c = getc(ifp)) != EOF) {
    if ((c == ' ') || (c == '\n') || (c == '\t'))
      break;
    if ((++count >= size) &&
        ((inbuf = (char *) realloc(inbuf, size += INBUF)) == NULL)) {
      fprintf(stderr, "Out Of Memory\n");
      exit(1);
    }
    inbuf[count - 1] = c;
  }

  if (count == 0)
    return(NULL);

  inbuf[count] = '\0';
  if (ungetc(c, ifp) == EOF) {
    fprintf(stderr, "Failed ungetc() call\n");
    exit(1);
  }

  /* Return the data read */
  if ((retval = (char *) malloc(strlen(inbuf) + 1)) == NULL) {
    fprintf(stderr, "Fatal Error: Out of Memory\n");
    exit(1);
  }
  strcpy(retval, inbuf);
  free(inbuf);
  return(retval);
}


/* Function: itoa
 * Input   : Integer and string
 * Return  : String representation of the passed integer
 */
char *itoa (int n) {
  char *str = NULL;
  char *retval;
  int count = 0, sign = n, size = 0;

  if (sign < 0)
    n = -n;

  /* Convert the int to a string starting with 1's place */
  do {
    if (count >= size)
      if ((str = (char *) realloc(str, size += 5)) == NULL) {
	fprintf(stderr, "Out Of Memory\n");
	exit(1);
      }
    str[count++] = n % 10 + '0';
  } while ((n /= 10) > 0);

  /* Add the unary - if any */
  if (sign < 0) {
    if ((count >= size) &&
	((str = (char *) realloc(str, size += 1)) == NULL)) {
      fprintf(stderr, "Out Of Memory\n");
      exit(1);
    }

    str[count++] = '-';
  }

  str[count] = '\0';
  if ((retval = (char *) malloc(strlen(str) + 1)) == NULL) {
    fprintf(stderr, "Out Of Memory\n");
    exit(1);
  }
  strcpy(retval, str);
  free(str);
  reverse(retval);
  return(retval);
}


/* Function: lowercase
 * Input   : A string
 * Output  : The passed string is converted to all lowercase.
 */
char *lowercase (char *instr) {
  int tmpint;

  if (instr == NULL)
    return(NULL);

  tmpint = strlen(instr);
  while (tmpint-- > 0)
    instr[tmpint] = tolower(instr[tmpint]);

  return(instr);
}


/* Function: my_setenv
 * Input   : The name of an input variable, the value for that variable, and
 *           a setenv() flag, (only used if setenv() is available).
 * Output  : The passed enviroment variable is set to the passed value.
 * Return  : TRUE if it is safe to free the passed arguments, otherwise FALSE.
 */
int my_setenv (char *var, char *value, int flag) {
#ifdef HAVE_SETENV
  if (setenv(var, value, flag) == -1) {
    fprintf(stderr, "Out Of Environment Memory\n");
    exit(1);
  }
  return(FALSE);
#else
  char *envstr;

  if ((envstr = (char *) malloc(strlen(var) + strlen(value) + 2)) == NULL) {
    fprintf(stderr, "Out Of Memory\n");
    exit(1);
  }
  sprintf(envstr, "%s=%s", var, value);
  if (putenv(envstr) == -1) {
    fprintf(stderr, "Out Of Environment Memory\n");
    exit(1);
  }
  return(TRUE);
#endif  /* HAVE_SETENV */
}


/* Function: parseenv
 * Input   : An environment file to parse
 * Output  : The user's current environment is changed accordingly.
 * Notes   : The environment file is in the form ENV VALUE where ENV is the
 *           environment variable to set and VALUE is the value for the
 *           variable.  Blank lines and lines begining with a # are ignored.
 *           If the first word on the line is run, then the rest of the line,
 *           (before any # characters), is taken as a program to run.
 */
void parseenv (char *file) {
  char *arg, *str;
  FILE *ifp;

  if ((file == NULL) || ((ifp = fopen(file, "r")) == NULL))
    return;

  /* Parse the environment file */
  while ((str = get_option(ifp)) != NULL) {
    arg = get_arguments(ifp);

    /* If it is a command to run, execute it, otherwise set the environment */
    if (strcmp(str, "run") == 0)
      runprog(arg, FALSE);
    else if (my_setenv(str, arg, 1) == FALSE)
      continue;
    free(str);
    free(arg);
  }
}


/* Function: printfile
 * Input   : Filename to print and print flag
 * Output  : If flag equals TRUE, print the file and then wait for the user
 *           to press a key, otherwise just print the file.
 */
void printfile (char *filename, int flag) {
  char c;
  char *infile, *pager, *tmpstr;
  FILE *ifp;
  int pid, status;

#ifdef VIEWDIR
  /* Build the filename to print */
  if (SECURE & 4) {
    tmpstr = stripdots(filename);
    infile = (char *) malloc(strlen(VIEWDIR) + strlen(tmpstr) + 1);
    strcpy(infile, VIEWDIR);
    strcat(infile, tmpstr);
    free(tmpstr);
  } else {
    infile = (char *) malloc(strlen(filename) + 1);
    strcpy(infile, filename);
  }
#else
  infile = (char *) malloc(strlen(filename) + 1);
  strcpy(infile, filename);
#endif  /* VIEWDIR */

  /* Open the file */
  if ((ifp = fopen(infile, "r")) == NULL) {
    error("ERROR: File not accessable", infile, 0);
    free(infile);
    return;
  }

  /* Print the file */
  if ((flag == TRUE) && ((pager = getenv("PAGER")) == NULL))
#ifdef PAGER
    pager = PAGER;
#else
    ;
#endif  /* PAGER */

  if ((flag == TRUE) && (pager != NULL)) {
    if ((pid = fork()) == -1) {
      fprintf(stderr, "Unable to fork\n");
      exit(1);
    }
    if (pid == 0) {
      clearscr();
      if (execl(pager, get_name(pager), infile, NULL) == -1)
	fprintf(stderr, "Unable to execute %s\n", pager);
    } else {
      waitpid(pid, &status, 0);
      waitkey();
    }
  } else {
    while ((c = getc(ifp)) != EOF)
      putc(c, stdout);
    fclose(ifp);
  }

  free(infile);
}


/* Function: printhead
 * Input   : Header and name to print.
 * Ouptut  : Header on far left of screen and name on far ride side.  Only the
 *           portion of the header and the name that can fit on the screen are
 *           printed.
 */
void printhead (char *head, char *name) {
  char *tmpstr;
  int cols = getcols(), tmpint;

  /* Print the header */
  if (head == NULL)
#ifdef MENUHEAD
    tmpstr = MENUHEAD;
#else
    return;
#endif
  else tmpstr = head;
  tmpint = strlen(tmpstr);
  printf("%.*s", cols - 1, tmpstr);
  if (((cols -= tmpint) < 0) || (name == NULL)) {
    putc('\n', stdout);
    return;
  }

  /* Print the name (if any) */
  for (tmpint = cols - strlen(name); tmpint > 1; tmpint--)
    putc(' ', stdout);
  if (tmpint <= 0)
    for (tmpint = 0; tmpint < cols; ++tmpint)
      putc(name[tmpint], stdout);
  else printf("%s\n", name);
}


/* Function: printline
 * Input   : Length of line to print.
 * Output  : A line of dashes followed by a newline.
 * Notes   : Same as printlineof(len, "-"); but faster.
 */
void printline (int len) {
  while (--len)
    putc('-', stdout);
  putc('\n', stdout);
}


/* Function: printlineof
 * Input   : Length of line to print and a string to print.
 * Output  : A line consisting of the passed string, (the string is repeated
 *           as many times as needed to fill the line).
 */
void printlineof (int len, char *str) {
  int count;

  if (str == NULL)
    printline(len);
  else {
    count = strlen(str);
    while (len > count) {
      len -= count;
      printf("%s", str);
    }

    for (count = 0; len > 1; ++count, --len)
      putc(str[count], stdout);
    putc('\n', stdout);
  }
}


/* Function: printstr
 * Input   : A string to print
 * Output  : The string parsed for arguments and printed to the screen
 * Notes   : The string must have the following format:
 *           [-n] ["] [string [`program to run`] [string] ...] ["]
 *           The -n argument will cause a newline not to be appended to the
 *           end of the passed string.  Things enclosed in `` marks will be
 *           treated as programs to run and can be scattered about within the
 *           string to print.  To print a ` mark, enter two ` marks like so:
 *           ``.  Environment variables can be printed by putting a $VAR where
 *           VAR is the environment variable you want to print into the print
 *           string.  To print a $, put two $ marks like so: $$.  To print
 *           spaces as the begining of the string, or to print new lines in a
 *           string, you need to begin the string with a double quote.  Any
 *           double quote found within the string, (not in the first or last
 *           position), is printed.  For example,
 *
 *              "    this "string" is indented
 *                    and continues on a new line"
 *
 *           would be displayed as:
 *
 *                   this "string" is indented
 *                   and continues on a new line
 *
 *           You should also use double quotes to enclose the string if you
 *           want to print a -n as the first couple characters.
 */
void printstr (char *str) {
  char *ptr;
  char savechar;
  int count = 0;
  register flag = 0;

  /* Check if there is anything to print */
  if (str == NULL) {
    putc('\n', stdout);
    return;
  }

  /* Check for a leading -n argument */
  if (strncmp(str, "-n", 2) == 0) {
    if ((str[2] == '\0') || (str[2] == ' ')) {
      flag |= NOCR;
      if (str[2] == ' ')
	count = 3;
      else count = 2;
    }
  }

  /* Check for a double quote */
  if (str[count] == '"') {
    flag |= INQUOTE;
    ++count;
  }

  /* Parse and print the passed string */
  while (str[count] != '\0') {

    /* Print output of a program */
    if (str[count] == '`') {
      fflush(stdout);
      ptr = &str[count + 1];
      while (str[++count] != '`') {
	if (((str[count] == '\n') && (~flag & INQUOTE)) ||
	    (str[count] == '\0')) {
	  error("Error in menu: Malformed print string", str, 0);
	  return;
	}
      }
      if (ptr == &str[count])
	putc(str[count++], stdout);
      else {
	savechar = str[count];
	str[count] = '\0';
	runprog(ptr, FALSE);
	str[count++] = savechar;
      }
    }

    /* Print a variable */
    else if (str[count] == '$') {
      fflush(stdout);
      ptr = &str[++count];
      while ((str[count] != ' ') && (str[count] != '\n') &&
	     (str[count] != '\0'))
	  ++count;
      if (ptr == &str[count])
	putc(str[count++], stdout);
      else {
	savechar = str[count];
	str[count] = '\0';
	printf("%s", getenv(ptr));
	str[count] = savechar;
      }
    }

    /* Print a character */
    else {
      if ((str[count] == '"') && (str[count + 1] == '\0') && (flag & INQUOTE));
      else putc(str[count], stdout);
      ++count;
    }
  }

  if (~flag & NOCR)
    putc('\n', stdout);
}


/* Function: reverse
 * Input   : String
 * Output  : The passed string in reversed order: "0123" becomes "3120".
 */
void reverse (char *str) {
  char *tmpstr = (char *) malloc(strlen(str) + 1);
  int count = 0, place;

  if (tmpstr == NULL) {
    fprintf(stderr, "Out Of Memory\n");
    exit(1);
  }

  strcpy(tmpstr, str);
  for (place = strlen(str) - 1; tmpstr[count] != '\0'; --place, ++count)
    str[place] = tmpstr[count];
  free(tmpstr);
}


/* Function: runprog
 * Input   : Program to run and flag
 * Output  : Run the passed program.  If flag is TRUE, wait for user to press
 *           a key after program is done running.
 */
void runprog (char *filename, int flag) {
  char *command, *ifs, *path, *pathenv, *save_ifs = NULL, *save_path = NULL;
  int result;

  /* Test if anything was passed to run */
  if (filename == NULL)
    return;

  if (SECURE & 1) {
#ifndef SECURERUN
    return;
#endif  /* SECURERUN */

    /* Strip any leading / marks */
    if (*command == '/')
      *command = ' ';

    /* Save the old path */
    if ((path = getenv("PATH")) != NULL) {
      if ((save_path = (char *) malloc(strlen(path) + 1)) == NULL) {
	fprintf(stderr, "Out Of Memory\n");
	exit(1);
      }
      strcpy(save_path, path);
    }

    /* Set the approved path */
    my_setenv("PATH", SECURERUN, 1);

    /* Remove IFS from the environment */
    if ((ifs = getenv("IFS")) != NULL) {
      if ((save_ifs = (char *) malloc(strlen(ifs) + 1)) == NULL) {
	fprintf(stderr, "Out Of Memory\n");
	exit(1);
      }
      strcpy(save_ifs, ifs);
      ifs = '\0';
    }

  } else {
    if ((command = (char *) malloc(strlen(filename) + 1)) == NULL) {
      fprintf(stderr, "Out Of Memory");
      exit(1);
    }
    strcpy(command, filename);
  }

  result = system(command);

  if (save_ifs != NULL) {
    strcpy(ifs, save_ifs);
    free(save_ifs);
  }
  if ((save_path != NULL) && (my_setenv("PATH", save_path, 1) == TRUE))
    free(save_path);

  if ((flag == TRUE) && (result != 127) && (result != -1))
    waitkey();
}


/* Function: skip
 * Input   : An open file to read from and a flag telling what to skip over.
 *           The flag may be an or of any of the following values:
 *
 *           COMMENTS - Skip lines starting with a # character
 *           SPACE    - Skip over spaces and tabs, but not new lines
 *           TO_EOL   - Skip over everything to and including the first new
 *                      line
 *           WHITE    - Skip over white space including new lines
 *
 * Output  : File pointer is moved over things to "skip."
 */
void skip (FILE *ifp, int flag) {
  char c;
  extern int line;

  /* Make sure a valid flag and non-null file pointer was given */
  if (!((flag & COMMENTS) || (flag & SPACE) || (flag & TO_EOL) ||
        (flag & WHITE)) || (ifp == NULL))
    return;

  /* Loop until EOF is found */
  while ((c = getc(ifp)) != EOF) {

    /* Skip over comments */
    if ((flag & COMMENTS) && (c == '#')) {
      skip_line(ifp, 1);
      continue;
    }

    /* Skip over white space */
    if (((flag & WHITE) || (flag & SPACE)) &&
        ((c == ' ') || (c == '\t') || (c == '\r')))
      continue;

    /* Test for EOL */
    if (c == '\n') {
      ++line;
      if (flag & WHITE)
        continue;
      else if (flag & TO_EOL)
        return;
    }

    /* Ignore everything if TO_EOL (new line is detected above) */
    if (flag & TO_EOL)
      continue;

    /* Place the last character read back into stream */
    if (ungetc(c, ifp) == EOF) {
      fprintf(stderr, "Fatal Error: Unable to unget\n");
      exit(1);
    }
    return;
  }
}


/* Function: skip_line
 * Input   : A file pointer to read the data from and the number of lines
 *           to skip.
 * Ouptut  : File pointer is moved appropiately
 */
void skip_line (FILE *ifp, int count) {
  char tmpin;
  int line = 0;

  while (line++ < count)
    skip(ifp, TO_EOL);
}


/* Function: stripdots
 * Input   : String
 * Output  : All instances of ../ are removed from the first word in the
 *           passed string.
 * Return  : The stripped string
 */
char *stripdots (char *str) {
  char *aptr = str, *retval;
  int count = 0;

  if (aptr == NULL)
    return;

  /* Count the size of the return string */
  while (*aptr != '\0') {
    if ((*aptr == ' ') || (*aptr == '\t')) {
      while (*aptr != '\0') {
        ++count;
	++aptr;
      }
    } else if (strncmp(aptr, "../", 3) == 0)
        aptr = aptr + 3;
    else {
      ++count;
      ++aptr;
    }
  }

  /* Create and fill the return value */
  if ((retval = (char *) malloc(count + 1)) == NULL) {
    fprintf(stderr, "Fatal Error: Out Of Memory\n");
    exit(1);
  }
  aptr = str;
  count = 0;
  while (*aptr != '\0') {
    if ((*aptr == ' ') || (*aptr == '\t')) {
      while (*aptr != '\0') {
        retval[count++] = *aptr;
        ++aptr;
      }
    } else if (strncmp(aptr, "../", 3) == 0)
        aptr = aptr + 3;
    else {
      retval[count++] = *aptr;
      ++aptr;
    }
  }
  retval[count] = '\0';
  return(retval);
}


/* Function: waitkey
 * Output  : Prompt to press ENTER and wait for user to do so.
 */
void waitkey(void) {
  char *tmpstr = NULL;

  tmpstr = get_input("Press ENTER to Continue", 1, getcols(), LETTER | NOSHOW);
  free(tmpstr);
}
