/* File     : getinput.c
 * Author   : Karyl F. Stein <xenon@xenos.net>
 * Purpose  : An multi-purpose input function.  This function strives to be
 *            as "bullet-proof" as possible.
 *
 * Usage    : char *getinput(char *prompt, int places, int col, int mode)
 *
 *            * prompt defines the prompt to print.  If this is NULL, no
 *              prompt is given.
 *            * places defines the number of characters to allow the user to
 *              input.  If this is 0, (not suggested), the user my enter an
 *              unlimited number of characters.
 *            * col defines the number of columns on the terminal.  If this is
 *              0, it is ignored.
 *            * mode takes at least one of the following values defined in
 *              getinput.h.  Multiple flags should be or'ed togeather.
 *
 *              LETTER    - Only letters are accepted.
 *              LOWERCASE - Any uppercase letters are transformed to lower
 *                          case.
 *              NUMBER    - Only numbers are accepted.
 *              TEXT      - Any text characters are accepted, (NUMBER and
 *                          LETTER do not need to be defined).
 *              ANY       - Any character printable or not is accepted.
 *              FIELD     - The prompt is followed by a field width
 *                          message, (either enclosed in | marks or a message
 *                          explaining how many lines are allowed to be
 *                          input).
 *              HELP      - Specify that help requests are allowed.  A help
 *                          request is when the user enters HELPCHAR as the
 *                          first character of the input.  A NULL is
 *                          immeditely returned if this is the case.
 *              NOSHOW    - Do not display the actual input, (useful if you
 *                          are inputing a password).
 *
 *            The string entered is returned or NULL if help is defined and
 *            HELPCHAR was the first character entered.
 *
 * Notes    : Be sure to edit get_input.h and change anything you want.
 *
 * get_input.c is Copyright (C)1996 Karyl F. Stein
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc., 675
 * Mass Ave, Cambridge, MA 02139, USA.
 *
 * TODO: Get the field stuff working.
 */

#include "get_input.h"
#include <termios.h>
#include <unistd.h>
#include <stdio.h>

/* Set defaults if not defined previously or not valid.  Do not change. */
#ifndef INBUF
# define INBUF 512
#elif (INBUF < 2)
# define INBUF 512
#endif
#ifndef HELPCHAR
# define HELPCHAR  '?'
#endif
#ifndef ERROR
# define ERROR 007
#endif
#ifndef NOSHOWCHAR
# define NOSHOWCHAR '*'
#endif


char *get_input (char *prompt, int places, int col, int mode) {
  char c;
  char *inbuf, *retval;
  int bufsize = 1, count = 0;
  struct termios save_scr, secure_scr;

  /* Initialize the input buffer */
  if (places == 0) {
    places = -1;
    if ((inbuf = (char *) malloc(INBUF)) == NULL) {
      fprintf(stderr, "Fatal Error: Out of Memory\n");
      exit(1);
    }
  } else if ((inbuf = (char *) malloc(places + 1)) == NULL) {
    fprintf(stderr, "Fatal Error: Out of Memory\n");
    exit(1);
  }

  /* Print the prompt before going into raw mode in case it has \n in it */
  if (prompt != NULL)
    printf("%s", prompt);

  /* Put the terminal into raw mode */
  if (tcgetattr(0, &save_scr) == -1) {
    fprintf(stderr, "Fatal Error: Unable to get terminal settings\n");
    exit(1);
  }
  secure_scr = save_scr;
#ifdef HAVE_CFMAKERAW
  cfmakeraw(&secure_scr);
#else
  secure_scr.c_iflag &= ~(IGNBRK|BRKINT|PARMRK|ISTRIP|INLCR|IGNCR|ICRNL|IXON);
  secure_scr.c_oflag &= ~OPOST;
  secure_scr.c_lflag &= ~(ECHO|ECHONL|ICANON|ISIG|IEXTEN);
  secure_scr.c_cflag &= ~(CSIZE|PARENB);
  secure_scr.c_cflag |= CS8;
#endif  /* HAVE_CFMAKERAW */
  secure_scr.c_cc[VMIN] = 1;
  secure_scr.c_cc[VTIME] = 0;
  if (tcsetattr(0, TCSANOW, &secure_scr) == -1) {
    fprintf(stderr, "Fatal Error: Unable to set terminal settings\n");
    exit(1);
  }

  /* Is there a better way to search for \n? */
  while (((c = getc(stdin)) != 13) && (c != 10)) {

    /* Test if help was requested */
    if ((mode & HELP) && (count == 0) && (c == HELPCHAR)) {

      /* Reset the terminal settings */
      if (tcsetattr(0, TCSANOW, &save_scr) == -1) {
	fprintf(stderr, "Fatal Error: Unable to set terminal settings\n");
	exit(1);
      }
      free(inbuf);
      return(NULL);
    }

    /* Resize the input buffer if necessary */
    if ((places == -1) && (count >= (INBUF * bufsize)))
      if ((inbuf = (char *) realloc(inbuf, INBUF * ++bufsize)) == NULL) {
	fprintf(stderr, "Fatal Error: Out of Memory\n");
	exit(1);
      }

    if ((c == secure_scr.c_cc[VERASE]) || (c == 127) || (c == 8)) {
      if (count > 0) {
	putc('\b', stdout);
	putc(' ', stdout);
	putc('\b', stdout);
	--count;
	inbuf[count] = '\0';
      } else putc(ERROR, stdout);
      continue;
    }

    else if ((mode & ANY) && ((count < places) || (places == -1))) {
      if (mode & NOSHOW)
	putc(NOSHOWCHAR, stdout);
      else putc(c, stdout);
      inbuf[count++] = c;
      continue;
    }

    if ((mode & LETTER) || (mode & TEXT)) {
      if ((((c >= 'a') && (c <= 'z')) || ((c >= 'A') && (c <= 'Z'))) &&
	  ((count < places) || (places == -1))) {
	if ((mode & LOWERCASE) && ((c >= 'A') && (c <= 'Z')))
	  c = (c - 'A') + 'a';
	inbuf[count++] = c;
	if (mode & NOSHOW)
	  putc(NOSHOWCHAR, stdout);
	else putc(c, stdout);
	continue;
      }
    }

    if ((mode & NUMBER) || (mode & TEXT)) {
      if ((c >= '0') && (c <= '9') && ((count < places) || (places == -1))) {
	inbuf[count++] = c;
	if (mode & NOSHOW)
	  putc(NOSHOWCHAR, stdout);
	else putc(c, stdout);
	continue;
      }
    }

    /* Relies on the use of ASCII */
    if (mode & TEXT) {
      if ((c >= 32) && ((count < places) || (places == -1))) {
	inbuf[count++] = c;
	if (mode & NOSHOW)
	  putc(NOSHOWCHAR, stdout);
	else putc(c, stdout);
	continue;
      }
    }
  
    putc(ERROR, stdout);
  }

  /* Reset the terminal settings */
  if (tcsetattr(0, TCSANOW, &save_scr) == -1) {
    fprintf(stderr, "Fatal Error: Unable to set terminal settings\n");
    exit(1);
  }

  /* Build the return value */
  inbuf[count] = '\0';
  if ((retval = (char *) malloc(strlen(inbuf) + 1)) == NULL) {
    fprintf(stderr, "Fatal Error: Out of Memory\n");
    exit(1);
  }
  strcpy(retval, inbuf);
  free(inbuf);

  putc('\n', stdout);
  return(retval);
}
