/*
 * StdModem.C - source file for class StdModem
 * Copyright (c) 1999 Joe Yandle <yandle@cs.unc.edu>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 */

#include "StdModem.h"

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/file.h>
#include <sys/select.h>
#include <selectbits.h>
#include <fcntl.h>
#include <unistd.h>
#include <termios.h>
#include <stdio.h>
#include <signal.h>
#include <string.h>
#include <stdlib.h>

const int MAX=255;
const int BAUDRATE=38400;

bool StdModem::openPort()
{
  if( (fd = open( MODEM, O_RDWR | O_SYNC | O_NOCTTY | O_NONBLOCK )) == -1 ) {
    return false;
  }
  flock(fd, LOCK_EX);
  
  /*
    This is taken straight from the Serial-Programming HOWTO
  */
  //int c, res;
  /*
    Open modem device for reading and writing and not as controlling tty
    because we don't want to get killed if linenoise sends CTRL-C.
  */
  
  tcgetattr(fd,&oldtio); /* save current serial port settings */
  //bzero(newtio, sizeof(newtio)); /* bzero gave me seg faults */
  memset(&newtio, 0, sizeof(newtio)); /* clear the struct for new port settings */
  
  /*
    BAUDRATE: Set bps rate. You could also use cfsetispeed and cfsetospeed.
    CRTSCTS : output hardware flow control (only used if the cable has
    all necessary lines. See sect. 7 of Serial-HOWTO)
    CS8     : 8n1 (8bit,no parity,1 stopbit)
    CLOCAL  : local connection, no modem contol
    CREAD   : enable receiving characters
  */
  newtio.c_cflag = BAUDRATE | CRTSCTS | CS8 | CLOCAL | CREAD;
  
  /*
    IGNPAR  : ignore bytes with parity errors
    ICRNL   : map CR to NL (otherwise a CR input on the other computer
    will not terminate input)
    otherwise make device raw (no other input processing)
  */
  newtio.c_iflag = IGNPAR | ICRNL;
  
  /*
    Raw output.
  */
  newtio.c_oflag = 0;
  
  /*
    ICANON  : enable canonical input
    disable all echo functionality, and don't send signals to calling program
  */
  newtio.c_lflag = ICANON;
  
  /*
    initialize all control characters
    default values can be found in /usr/include/termios.h, and are given
    in the comments, but we don't need them here
  */
  newtio.c_cc[VINTR]    = 0;     /* Ctrl-c */
  newtio.c_cc[VQUIT]    = 0;     /* Ctrl-\ */
  newtio.c_cc[VERASE]   = 0;     /* del */
  newtio.c_cc[VKILL]    = 0;     /* @ */
  newtio.c_cc[VEOF]     = 4;     /* Ctrl-d */
  newtio.c_cc[VTIME]    = 0;     /* inter-character timer unused */
  newtio.c_cc[VMIN]     = 1;     /* blocking read until 1 character arrives */
  newtio.c_cc[VSWTC]    = 0;     /* '\0' */
  newtio.c_cc[VSTART]   = 0;     /* Ctrl-q */
  newtio.c_cc[VSTOP]    = 0;     /* Ctrl-s */
  newtio.c_cc[VSUSP]    = 0;     /* Ctrl-z */
  newtio.c_cc[VEOL]     = 0;     /* '\0' */
  newtio.c_cc[VREPRINT] = 0;     /* Ctrl-r */
  newtio.c_cc[VDISCARD] = 0;     /* Ctrl-u */
  newtio.c_cc[VWERASE]  = 0;     /* Ctrl-w */
  newtio.c_cc[VLNEXT]   = 0;     /* Ctrl-v */
  newtio.c_cc[VEOL2]    = 0;     /* '\0' */
  
  /*
    now clean the modem line and activate the settings for the port
  */
  tcflush(fd, TCIOFLUSH);
  tcsetattr(fd,TCSANOW,&newtio);
  
  /*
    terminal settings done, now handle input
    In this example, inputting a 'z' at the beginning of a line will
    exit the program.
  */
  
  writeLine(MODEM_RESET);
  writeLine(MODEM_INIT);

  return true;
}

void StdModem::closePort()
{
  tcsetattr(fd, TCSANOW, &oldtio);
  flock(fd, LOCK_UN);
  close(fd);
  fd = -1;
}

void StdModem::readLine(char* buffer)
{
  int i = read(fd,buffer,MAX);
  if(i>0) {
    while(buffer[i-1]=='\n')
      i--;
    buffer[i] = '\0';
  }
  else {
    buffer[0]='\0';
  }
}

void StdModem::writeLine(char* buffer)
{
    char buf1[255];
    char buf2[255];
    struct fd_set readFds, writeFds;
    FD_ZERO(&readFds);
    FD_ZERO(&writeFds);
    FD_SET(fd, &readFds);
    FD_SET(fd, &writeFds);
    sprintf(buf1,"%s\r",buffer);
    select(fd+1, NULL, &writeFds, NULL, NULL);
    write(fd, buf1, strlen(buf1));
    while(!strstr(buf2,buffer)) {
	select(fd+1, &readFds, NULL, NULL, NULL);
	readLine(buf2);
    }
    while(!strstr(buf2,"OK")) {
	select(fd+1, &readFds, NULL, NULL, NULL);
	readLine(buf2);
    }
}

int StdModem::getDes()
{
  return fd;
}

