/****************************************************************************
ttnews.cpp 99-05-02
Version: 1.0.0
Authors: Simon Kgstrm
         Carl Smith
	 Nicholas Daley
	 Rickard Molin (testing)
Description: Displays headlines and articles in X11.

(C) 1999-2000 Simon Kgstrm
This software is released under the GNU General Public License.
Please read the included file COPYING for more information.
This software comes with no warranty of any kind, use at you own risk!
*****************************************************************************/

#include <qbitmap.h>
#include <qpainter.h>
#include <qcursor.h>
#include <qevent.h>
#include <qapplication.h>
#include <qpushbutton.h>
#include <qstring.h>
#include <stdio.h>
#include <qdir.h>
#include <qfile.h>
#include <qdatastream.h>
#include <qtextstream.h>
#include <qpopupmenu.h>
#include <qmessagebox.h>
#include <qimageio.h>

// Includes written for the program
#include "config.h"
#include "ttnews.h"
#include "readConfig.h"
#include "newsWidget.h"
#include "allNewsWidget.h"
#include "pixmapButton.h"
#include "parseNews.h"

ConfigStruct *config; // Global varaiables for storing the config options.
SkinStruct *skinConfig;

/*
Class DesktopWidget - The main class, displays the headlines on the desktop.
----------------------------------------------------------------------------
*/
/** Constructs a new main window.*/
DesktopWidget::DesktopWidget(QWidget *parent, const char *name )
#ifdef DESKTOP_NO_WM_CONTROL // WIthouot WM-decorations
  : QWidget( parent, name, WStyle_Customize | WStyle_NoBorder | WStyle_Tool)
#else // Or with
  : QWidget( parent, name, WStyle_NormalBorder)
#endif
{ 
  headlineFont = QFont( skinConfig->headlineFontFamily, skinConfig->headlineFontSize );
  //qInitImageIO();

  QString tempSkinPath = "skins/" + QString(config->skin).stripWhiteSpace() + "/";
  if(!(QFile(tempSkinPath)).exists()) // Else, use the one in the users home catalogue.
    tempSkinPath = dirPath.homeDirPath()+"/.ttnews/skins/"+ QString(config->skin).stripWhiteSpace() + "/";
  if (!(QFile(tempSkinPath)).exists()) // Else, use the global one,
    tempSkinPath = QString(INSTALL_DIR) + "/skins/" + QString(config->skin).stripWhiteSpace() + "/";
  QString tempSkinSuffix = ".png"; 
  if (!QFile::exists(tempSkinPath+"mainWindow"+tempSkinSuffix))
      tempSkinSuffix = ".bmp"; 

  if (backgroundImage.load((tempSkinPath+"mainWindow"+tempSkinSuffix).data()) == false)
    {
      fprintf (stderr, "This skin is wrong! Could not load mainWindow.[png/bmp]\n");
      exit (2);
    }

  setFixedSize(backgroundImage.width(),backgroundImage.height());

  // create push-buttons. This (and skins) should be converted to PNG...
  prevButton = new PixmapButton (tempSkinPath+"prevNormalButton"+tempSkinSuffix,tempSkinPath+"prevPressedButton"+tempSkinSuffix,this);
  nextButton = new PixmapButton (tempSkinPath+"nextNormalButton"+tempSkinSuffix,tempSkinPath+"nextPressedButton"+tempSkinSuffix,this);
  allNewsButton = new PixmapButton (tempSkinPath+"allNewsNormalButton"+tempSkinSuffix,tempSkinPath+"allNewsPressedButton"+tempSkinSuffix,this);
  if (skinConfig->showQuitButton)
    quitButton = new PixmapButton (tempSkinPath+"closeNormalButton"+tempSkinSuffix,tempSkinPath+"closePressedButton"+tempSkinSuffix,this);
  else quitButton = new PixmapButton(NULL,NULL, this);
  connect ( prevButton, SIGNAL(clicked()), SLOT(prevButtonClicked()) );
  connect ( nextButton, SIGNAL(clicked()), SLOT(nextButtonClicked()) );
  connect ( allNewsButton, SIGNAL(clicked()), SLOT (allNewsButtonClicked()) );
  if (skinConfig->showQuitButton)
    connect ( quitButton, SIGNAL(clicked()), qApp, SLOT (quit()) );  

  prevButton->move(width()-prevButton->width()-nextButton->width(),2);
  nextButton->move(width()-nextButton->width(),2);
  allNewsButton->move(width()-allNewsButton->width(),nextButton->height());
  if (skinConfig->showQuitButton)
    quitButton->move(2,2);

  //  articleNr=0;
  showingBody = false;

  // Start some timers
  switchNewsTimer = startTimer (config->switchNewsTime*1000);
  readNewsTimer = startTimer (config->readNewsTime*1000);
  getNewsTimer = startTimer (config->getNewsTime*1000); // Not used.
  checkMouseTimer = startTimer (config->checkMouseOverTime);
  readCommandFileTimer = startTimer (config->readCommandFileTime);

  popup = new QPopupMenu; // Create a popup menu with some options.
  popup->setFont( QFont("Helvetica", 12, QFont::Normal));
  popup->insertItem("&Next item", this,SLOT(nextButtonClicked()));
  popup->insertItem("&Previous item", this, SLOT(prevButtonClicked()));
  popup->insertItem("&First item", this,SLOT(firstItemButtonClicked()));
  popup->insertItem("Next &Source", this,SLOT(nextNewsSourceButtonClicked()));
  popup->insertItem("&All news", this,SLOT(allNewsButtonClicked()));
  popup->insertItem("&Read news", this,SLOT(readNewsButtonClicked()));
  popup->insertSeparator();
  popup->insertItem("&Help",this, SLOT(helpClicked()));
  popup->insertSeparator();
  popup->insertItem("&Quit", qApp, SLOT(quit()));

  newsFilenames = config->newsPath;
  
  windowMetricsFile = new QFile(dirPath.homeDirPath()+"/.ttnews/ttnews.windowMetrics");

  table = new NewsTable(); // BUGWARNING:

  if (readNews() != 0)
    {
      fprintf(stderr, "Could not find any valid news sources, check NewsPath in\n"
	              "$HOME/.ttnews/ttnews.conf\n"
	              "or the default installation path for TT-News.\n"
	              "Exiting...\n ");
      exit (3);
    }

#ifdef READ_COMMAND_FILE 
  commandFile = new QFile (dirPath.homeDirPath()+"/.ttnews/command");
  commandFile->remove();
#else
  commandFile = NULL;
#endif
}

DesktopWidget::~DesktopWidget()
{
  delete prevButton;
  delete nextButton;
  delete allNewsButton;
  delete quitButton;
  delete newsBody;
  delete allNewsBody;
  delete commandFile;
  delete windowMetricsFile;
  delete table; 
}

// -----------------------Methods------------------------------
/**
Draw the headlines in the widget. (Make it more configurable)
*/
void DesktopWidget::paintEvent( QPaintEvent * )
{
  QPixmap offscreen( width(), height() );
  QPainter p;

  p.begin( &offscreen );
  p.setFont( headlineFont );

  if ( table->getCurrent().getColour().isEmpty() ) {
    p.setPen( skinConfig->headlineFontColor );
  } else {
    p.setPen( table->getCurrent().getColour().data());
  }

  p.drawPixmap(0,0,backgroundImage,0,0,-1,-1);

  p.drawText( quitButton->width()+6,14, table->getCurrent().getDate()); 
  p.drawText( 32+p.fontMetrics().width(table->getCurrent().getDate()), 14, table->getCurrent().getTime()); // Change
  int h = quitButton->height()+2;
  if (p.fontMetrics().height() > h)
    h = p.fontMetrics().height();
  p.drawText( 6,h, width()-allNewsButton->width()-20, height()-p.fontMetrics().height(), AlignLeft | WordBreak, table->getCurrent().getTitle());
 
  p.end();

  bitBlt( this, 0, 0 , &offscreen );
  
  prevButton->repaint();
  nextButton->repaint();
  allNewsButton->repaint();
  if (skinConfig->showQuitButton)
    quitButton->repaint();
}

/** Display the headline [dir], if dir==30000, show next. -30000 shows prev. */
void DesktopWidget::switchNews(int dir)
{ 
  bool tmp = false;
  int itmp=0;
  
  if (showingBody)
    {
      showBody(false);
      tmp = true;
    }

  if (dir == -30000 ) // Step backwards.
    NewsItem tmp2 = table->getPrev();
  else if (dir == 30000) // Step forward. (Buggy?)
    NewsItem tmp2 = table->getNext();
  else // Switch to news nr dir.
    table->getArticleNr(dir);
  if (newsBody != NULL) // If it has been created, update the news when they switch
    newsBody->update(table->getCurrent());
  if (tmp)
    showBody(true);
  killTimer(switchNewsTimer); // Restart the timer if the news has switched
  switchNewsTimer = startTimer (config->switchNewsTime*1000);
  repaint();
}

/** Display the article-body. Boolean argument if the body should be showed or hided.*/
void DesktopWidget::showBody(bool onOff)
{
  QString showArticle=config->showArticleBody;
  QString mouseClickShowArticle=config->mouseClickShowArticle;
  if (showArticle.stripWhiteSpace().lower() == "true")
    {
      if (onOff)
	{
	  if (newsBody == NULL)
	    newsBody = new NewsWidget(config, skinConfig, table->getCurrent(), 0); // 0 = top-level window.
	  
	  // Move the newsBody somewhere nice.
	  int moveXPos;
	  int moveYPos;
	  if (x()+newsBody->width() > qApp->desktop()->width()) // Outside right border of the screen
	    moveXPos = qApp->desktop()->width()-newsBody->width();
	  else
	    moveXPos = x();
	  if (y()-newsBody->height()-10 < 0) // Above the top.
	    moveYPos = y()+height()+10;
	  else if ( y()-newsBody->height() > qApp->desktop()->height() ) // Below the bottom.
	    moveYPos = qApp->desktop()->height()-newsBody->height();
	  else
	    moveYPos = y()-newsBody->height()-10;
	  newsBody->move(moveXPos, moveYPos);
	  if (!table->getCurrent().getBody().simplifyWhiteSpace().isEmpty())
	    newsBody->show();
	  showingBody = true;	  
	}
      if (!onOff) // Hide.
	{
	  if (newsBody != NULL)
	    {
	      if (newsBody->isVisible()) //	  != NULL
		{
		  newsBody->hide();
		  showingBody = false;
		}	
	    }
	}
    }
}


/** Show all news articles in a separate window */
void DesktopWidget::showAllArticles()
{
  if ((allNewsBody == NULL) || (allNewsBody->isVisible() == false))
    {
      if ((allNewsBody != NULL) && (allNewsBody->isVisible() == false))
	allNewsBody->show();
      else
	{
	  allNewsBody = new AllNewsWidget(config, skinConfig, table, 0, "News for today:");
	  connect ( allNewsBody, SIGNAL(sel(int)), SLOT (switchNews( int )) ); // If title is clicked, switch to corresponding news-item.
	  allNewsBody->show();
	}
    }
  else if (allNewsBody != NULL)
    {
      delete allNewsBody;
      allNewsBody = NULL;
    }
}

/** Checks where the mouse is, and displays a body-window if the mouse is inside the window. */
void DesktopWidget::checkMousePos()
{
  QCursor q;
  QPoint p = q.pos(); // Get the global mouse-pos
  QString tmp = config->mouseClickShowArticle; // Show article only if window clicked.
  if (tmp != "true") // Ugly?
    {
      bool inOut = ( (p.x() > x()) && (p.y() > y()) && (p.x() < x()+frameGeometry().width()) && (p.y() < y()+frameGeometry().height()) );
      showBody(inOut);
    }
}


/** Executed if the mouse enters the window */
void DesktopWidget::enterEvent(QEvent *e)
{
  if (QString(config->mouseClickShowArticle) != "true")
    showBody(true);
}

/** Executed when the mouse leaves the window */
void DesktopWidget::leaveEvent(QEvent *e)
{
  if (QString(config->mouseClickShowArticle) != "true")
    showBody(false);
}

/** move to the next news source (eg. BBC) */
void DesktopWidget::nextNewsSourceButtonClicked()
{
  switchNews(table->getNextNewsSource());
}

/** Goto next news item */
void DesktopWidget::nextButtonClicked()
{
  switchNews(30000);
}

/** Goto previous news item */
void DesktopWidget::prevButtonClicked()
{
  switchNews(-30000);
}

/** Goto first news item */
void DesktopWidget::firstItemButtonClicked()
{
  switchNews(0);
}

/** Show all news articles (catch the button clicked) */
void DesktopWidget::allNewsButtonClicked()
{
  showAllArticles();
}

/** read the news files again */
void DesktopWidget::readNewsButtonClicked()
{
  int rn = readNews();
  if (rn != 0)
    {
      fprintf(stderr, "Could not find any valid news sources, check NewsPath in\n"
	              "$HOME/.ttnews/ttnews.conf\n"
	              "or the default installation path for TT-News.\n"
	              "Exiting...\n ");
      exit (3);
    }
}

/** Show some help for the program */
void DesktopWidget::helpClicked()
{
  QMessageBox::information(0, "TT-News Help",
		     "Usage of TT-News\n\n"
                     "The program shows news in it's main window, which is\n"
		     "switching news every few seconds. If there is an\n"
		     "article associated with the news item, it can be shown\n"
		     "by placing the mouse over the TT-News window, or\n"
		     "(optionally) by clicking in the window.\n\n"
		     "A menu with some options is shown if you press the\n"
		     "right mouse button in the window.\n\n"
		     "The news-button shows all news in store, and the \n"
		     "arrow-buttons moves to next and previous item.\n\n"
		     "Version 1.0.0\n\n"
		     "TT-News is released under the GNU GPL, see the file\n"
		     "COPYING for details\n\n"
		     "TT-News was coded by (in order of appearance):\n"
		     "Simon Kgstrm, simon.kagstrom.864@student.lu.se\n"
		     "Carl Smith\n"
                     "Nicholas Daley\n"
		     "Thanks to Rickard Molin for testing.\n"); 
}

/** Reads the news from the news-file(s). */
int DesktopWidget::readNews()
{
  int nrOfSources=0;
  int tokenPos;
  int tmpNrOfNodes=0;
  int rn = 0; // return value
  QString newsArray = newsFilenames;

  if (!table->isEmpty())
    {
      delete table;
      table = new NewsTable();
    }  
  do // merge the lists in newsArray...
    {      
      tokenPos = newsArray.find(',');
      QString tmp = newsArray.left(tokenPos);
      bool fileFound=false;
      if (QFile(tmp).exists())
	fileFound = true;
      else if (QFile(dirPath.homeDirPath()+"/.ttnews/"+tmp).exists())
	{
	  tmp=(dirPath.homeDirPath()+"/.ttnews/"+tmp);
	  fileFound=true;
	}
      newsArray = (newsArray.right(newsArray.length()-tokenPos-1)).simplifyWhiteSpace();
      if (fileFound)
	{
	  newsParser = new ParseNews(tmp, table); // Read generic News.      
	  int out = newsParser->readNews();
	  if (out == 0) // News file found.
	    nrOfSources++;
	}
    }  
  while ((newsArray.length() != 0) && (tokenPos != -1));
  if (nrOfSources==0) // No news-files found. Exit gracefully.
    rn = 1;
  else
    firstItemButtonClicked(); // move to the first news-item.
  return rn;
}

/** Read the command-file and act accordingly. This would probably be better with a pipe, but I don't know how to make those (haven't found out how in QT at least)... :-/ You HELP! */
int DesktopWidget::parseCommand()
{
  if (!commandFile->open(IO_ReadOnly))
    return 1;
  QTextStream t(commandFile);
  QString s = t.readLine();
  if (s.lower() == "next")
    switchNews(30000);
  else if (s.lower() == "prev" || s.lower() == "previous")
    switchNews(-30000);
  else if (s.lower() == "allnews" || s.lower() == "all news")
    showAllArticles();
  else if (s.lower() == "showbody" || s.lower() == "show body" || s.lower() == "show article")
    {
      if (!showingBody)
	showBody(true);
      else
	showBody(false);
    }
  else if (s.lower() == "next source" || s.lower() == "nextsource" || s.lower() == "next agency")
    nextNewsSourceButtonClicked();
  else if (s.lower() == "read news" || s.lower() == "readnews")
     readNewsButtonClicked();
  else if (s.lower() == "quit" || s.lower() == "exit" || s.lower() == "close")
    qApp->quit();
  else
    {
      bool tst;
      int tmp=s.toInt(&tst);
      if ((tst == true) && (tmp > 0))
	switchNews(tmp);
    }
  commandFile->close();
  commandFile->remove();
  return 0;
}

/** Catches clicks in the main window. (Show the article) */
void DesktopWidget::mousePressEvent (QMouseEvent *e)
{
  QString tmp = config->mouseClickShowArticle;
  if ((e->button() == LeftButton) && (tmp == "true")) // tmp != true if mouse in window shows article
    {
      showingBody = !showingBody;
      showBody(showingBody); 
    }
  if (e->button() == RightButton)
    popup->popup(mapToGlobal(e->pos()));
}

/** This is executed if a moveEvent occurs (window moved). */
void DesktopWidget::moveEvent (QMoveEvent *e)
{
  if (windowMetricsFile->open(IO_WriteOnly)) // Open the windowMetrics file.
    {
      QDataStream d (windowMetricsFile);
      d << e->pos(); // Write windowMetrics to file.
      windowMetricsFile->close();
    }
}

/** This is executed if a timeEvent occurs. */
void DesktopWidget::timerEvent( QTimerEvent *e)
{
  if (e->timerId() == switchNewsTimer)
    {
      //      checkMousePos();
      if ((showingBody == false)) // Don't switch news if we're showing the newsbody
	switchNews(30000);
      else if (table->getCurrent().getBody().simplifyWhiteSpace().isEmpty()) // But switch if we have an empty body.
	switchNews(30000);
    }
  if (e->timerId() == readNewsTimer)
    {
      int rn=readNews(); // If the user reads an article causes this a jump to the first article.
      if (rn != 0)
	{
	  fprintf(stderr, "Could not find any valid news sources, check NewsPath in\n"
		      "$HOME/.ttnews/ttnews.conf\n"
	              "or the default installation path for TT-News.\n"
	              "Exiting...\n ");
	  exit (3);
	}
    }
  //  if (e->timerId() == checkMouseTimer)
  //    checkMousePos();
  if (e->timerId() == readCommandFileTimer)
    {
#ifdef READ_COMMAND_FILE 
      parseCommand();
#endif 
    }
}

/*
Main program
------------
*/
int main( int argc, char **argv )
{
  DesktopWidget *t;
  QDir homePath;
  QString tmp;

  tmp = "ttnews.conf"; // Create path to ttnews.conf
  if(!(QFile(tmp)).exists()) // Else, use the global one,
    tmp = homePath.homeDirPath() + "/.ttnews/ttnews.conf";
  if(!(QFile(tmp)).exists()) // Else, use the global one,
    tmp = QString(INSTALL_DIR)+"/ttnews.conf";
  config = readConfig(tmp.data());
  
  if ((config == NULL))
    return 1;

  tmp = QString("skins/") + config->skin + "/skin.conf"; // Create the skin-path
  if(!(QFile(tmp)).exists()) // Else, use the global one,
    tmp = homePath.homeDirPath()+"/.ttnews/skins/" + config->skin + "/skin.conf";
  if(!(QFile(tmp)).exists()) // Else, use the global one,
    tmp = QString(QString(INSTALL_DIR)+"/skins/") + config->skin + "/skin.conf";

  skinConfig = readSkinConfig(tmp.data());
  if (skinConfig == NULL)
    return 1;

  QApplication app( argc, argv );
  QFont f( "Helvetica", 12, QFont::Normal ); // Default font
  //  f.setStyleHint( QFont::Times );
  app.setFont( f,false ); 

  t = new DesktopWidget(); 

  app.setMainWidget( t);
  t->update();
  // Place the window somewhere.
  QPoint p;
  if (strcmp(config->placement,"topright") == 0) { 
    p.setX(qApp->desktop()->width()-t->width());
    p.setY(0);
  }
  else if (strcmp(config->placement,"topleft") == 0) { 
    p.setX(0);
    p.setY(0);
  }  
  else if (strcmp(config->placement,"bottomright") == 0) {  
    p.setX(qApp->desktop()->width()-t->width());
    p.setY(qApp->desktop()->height()-t->height());
  }
  else if (strcmp(config->placement,"bottomleft") == 0) {  
    p.setX(0);
    p.setY(qApp->desktop()->height()-t->height());
  }
  else {
    if (t->windowMetricsFile->open(IO_ReadOnly)) // Move the window to where it was last session.
      {
	QDataStream d (t->windowMetricsFile);
	d >> p; // Read last window pos from file.
	// p is outside screen, move it inside! POSSIBLE BUG!
	if ((p.x() <= -5 || p.x()+t->width() >= qApp->desktop()->width()+5) || (p.y() <= -5 || p.y()+t->height() >= qApp->desktop()->height()+5))
	  {
	    p.setX(qApp->desktop()->width()-t->width()+2);
	    p.setY(qApp->desktop()->height()-t->height()+2);
	  }
	
	t->windowMetricsFile->close();
      }
    else // If windowMetrics not found, place it in the lower right corner...
      {
	p.setX(qApp->desktop()->width()-t->width()+2);
	p.setY( qApp->desktop()->height()-t->height()+2);
      }
  }

  t->show();
  t->move (p);

  app.exec();

  delete t;
  return 0;
}

