/**********************************************************************
Generic news Feeder.
99-10-04
Authors:  Simon Kgstrm
          Carl Smith
Description: Fetches news from various sources, uses ReadNews which 
             should be implemented separately for each source.
**********************************************************************/

#include <qapplication.h>
#include <qdir.h>
#include <qstring.h>
#include <stdlib.h>
#include <unistd.h>     
#include <signal.h>
#include <time.h>
#include <sys/wait.h>

#include "../config.h"
#include "readNews.h"
#include "fetchNews.h"

/**
Class FetchNews - Fetches news every 25(?) minutes
--------------------------------------------------
*/
bool readOnly;

FetchNews::FetchNews( QString iForkString, QString iOutFile)
{
  outFile = iOutFile;
  forkString = iForkString;
  // Do a initial fetch of the news.
  QDir tmp;
  newsFile = outFile+".ticker";
  int status = 0;
  if (!readOnly)
    status = getNews();
  if (status!=0)
    {
      fprintf (stderr,QString("Could not fetch news from "+forkString+". Exiting...\n").data());
      cleanUp();
      exit (status);
    }
  ReadNews *news = new ReadNews (newsFile, outFile);
  int rn= news->generateNewsfile();
  if (rn==2)
    {
     fprintf (stderr,"The outfile is writeprotected. Exiting...\n");
     cleanUp();
     exit(rn);
    }
  delete news;
}

/** Overloaded constructor if you want the feeder to quit directly.*/
FetchNews::FetchNews( QString iForkString, QString iOutFile, bool quit)
{
  outFile = iOutFile;
  forkString = iForkString;
  QDir tmp;
  newsFile = outFile+".ticker";
  int status = 0;
  if (!readOnly)
     status = getNews();
  if (status!=0)
    {
      fprintf (stderr,QString("Could not fetch news from "+forkString+". Exiting...\n").data());
      cleanUp();
      exit (status);
    }
  ReadNews *news = new ReadNews (newsFile, outFile);
  int rn = news->generateNewsfile();
  if (rn==2)
    {
     fprintf (stderr,"The outfile is write protected. Exiting...\n");
     cleanUp();
     exit(rn);
    }
  delete news;
  if (!readOnly)
    cleanUp();
  exit(0);
}

FetchNews::~FetchNews()
{
  cleanUp();
}

/*----------------------------Methods----------------------*/

/** Get news from the source. Spawns lynx (or another program)*/
int FetchNews::getNews()
{
  int status=3000;
  int dummy_check = -5;
  time_t dummy_time=0;
  pid_t pid;

  QString dir;
  dir = forkString + " > " + newsFile;
  pid = fork ();
  if (pid == 0)      /* This is the child process.*/
    {
      status = execl (SHELL, SHELL, "-c", dir.data(), NULL);
      _exit (EXIT_FAILURE);
    }
  else if (pid < 0)    /* The fork failed. Report failure.*/
    status = -1;
  
  //  dummy_time = time(NULL);
  while (dummy_check == -5 && status == 3000)
    {
      waitpid (WAIT_ANY, &status, WNOHANG); /*Wait for forked process.*/
      /*      if (time(NULL) >= dummy_time+20) 
	      {
	      dummy_check = 0;
	      kill (0,0); //If not ready by now - kill lynx!

	      }
      */
 
      sleep(1);
      dummy_time++;
      if (dummy_time >=20)
	{
	  dummy_check = 0;
	  kill(0,0);
	}
  
    }
  return status;
}

/** Fetch the news and generate news-file. */
int FetchNews::doFetch()
{
  int status = getNews();
  if (status != 0)
    return status;
  ReadNews *news = new ReadNews (newsFile, outFile);
  int rn = news->generateNewsfile();
  if (rn==2)
    {
     fprintf (stderr,"The outfile is write protected. Exiting...\n");
     cleanUp();
     status=1;
    }
  delete news;
  return status;
}

// --- Private methods ---
void FetchNews::cleanUp()
{
  if (QFile(newsFile).remove() == false)
    fprintf (stderr,"Could not remove tmp-file\n");
}

/* Main. */
int main (int argc, char **argv)
{
  QDir dir;
  QString tmp;  
  QString tmpArgv = argv[1];
  int timeBetweenFetches = 1500; // Default time = 25 mins.
  bool quit=false;
  readOnly = false;

  /* Parse the arguments to the feeder. */
  int argPos=1; // Which argv to access. 
  if ((argc <= 1) || (tmpArgv == "--help") || (tmpArgv == "-h"))
    {
      printf("Usage:%s [OPTION] FILE\nStarts a server for fetching news. FILE is where you'd like to output\nthe parsed news.\n  -h, --help\t\tthis text.\n  -q, --quit\t\tquit after the news has been fetched once.\n  -r, --read\t\tjust read the source file (FILE+"".ticker""), no news\n\t\t\tfetching is done. Useful for testing offline.\n  -t, --time [int]\tSpecify the time (in seconds) to wait between fetches.\n",argv[0]);
      exit(0);
    }
  else
    while(argPos < argc-1) // Options
      {
	if ((tmpArgv == "--quit") || (tmpArgv == "-q"))
	  {
	    quit = true;
	  }
	if ((tmpArgv == "--read") || (tmpArgv == "-r"))
	  {
 	    readOnly = true;
	  }
	if ((tmpArgv == "--time") || (tmpArgv == "-t"))
	  {
	    argPos++;
	    QString tmp=argv[argPos];
	    bool tst=true;
	    timeBetweenFetches = tmp.toInt(&tst);
	    if (tst == false)
	      {
		fprintf(stderr, "Enter time (integer) between fetches\n");
		exit(2);
	      }
	  }
	argPos++;
	tmpArgv=argv[argPos];
	tmpArgv=tmpArgv.stripWhiteSpace();
      }
  if (tmpArgv.simplifyWhiteSpace().isEmpty() || tmpArgv.at(0) == '-')
    {
      fprintf (stderr,"Specify a destination, please\n");
      exit (2);
    }


  tmp = tmpArgv;
  ReadNews *tmpNews = new ReadNews();
  FetchNews *myApp;
  if(quit)
    myApp = new FetchNews(tmpNews->getForkCommand(),tmp, quit);
  else
    myApp = new FetchNews(tmpNews->getForkCommand(),tmp);

  int rn=0;
  if (timeBetweenFetches < 600) // Minimum 10 minutes between fetches
    timeBetweenFetches = 600;
    
  while (rn == 0)
    {
      rn = myApp->doFetch();
      if (rn != 0)
	{
	  fprintf (stderr, "Could not fetch the news...\nExiting...\n");
	}
      sleep (timeBetweenFetches);
    }
  delete myApp;
  delete tmpNews;
}
