/*
 *   Tm  -  Turing Machine
 *
 *   Placed in public domain by Alvaro De Amicis, 1999.
 *
 *   tm.c
 */

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include "tm.h"
#include "auxfunc.h"
#include "msg.h"


SQUARE *mark_A ;
SQUARE *mark_B ;
int    verbose, mverbose ;


SQUARE *next_square( SQUARE *sq )
{
    if ( sq->next == NULL ) {
        SQUARE *p ;
        p = (SQUARE*) malloc( sizeof( SQUARE ) ) ;
        if ( p == NULL ) {
        	printf( MSG_ERR_NOMEM ) ; exit (-1) ;
        }
        sq->next = p ;
        p->symbol = BLANK ;
        p->next = NULL ;
        p->previous = sq ;
        mark_B = p ;
        return p ;
    } else 
        return ( sq->next ) ;
}


SQUARE *prev_square( SQUARE *sq )
{
    if ( sq->previous == NULL ) {
        SQUARE *p ;
        p = (SQUARE*) malloc( sizeof( SQUARE ) ) ;
        if ( p == NULL ) {
        	printf( MSG_ERR_NOMEM ) ; exit (-1) ;
        }
        sq->previous = p ;
        p->symbol = BLANK ;
        p->previous = NULL ;
        p->next = sq ;
        mark_A = p ;
        return p ;
    } else 
        return sq->previous ;
}



SQUARE *read_tape( char *filename )
{
    FILE    *fp ;
    SQUARE  *p, *p2 ; 
    int     c ;
    
    p = (SQUARE*) malloc( sizeof( SQUARE ) ) ;
    if ( p == NULL ) {
      	printf( MSG_ERR_NOMEM ) ; exit (-1) ;
    }
    p->symbol = BLANK ;
    p->next = p->previous = NULL ;
    mark_A = mark_B = p ;
    
    p2 = p ;
    
    if ( ( fp = fopen( filename, "r") ) == NULL ) {
    	if ( errno == ENOENT )
    		fprintf( stderr, "%s: %s", filename, MSG_ERR_FILENOTFOUND ) ;
    	else if ( errno == EACCES )
    		fprintf( stderr, "%s: %s", filename, MSG_ERR_PERMDENIED ) ;
    	else fprintf( stderr, "%s: %s", filename, MSG_ERR_IOERR ) ; 
    	exit(-1) ;
    }

    while ( (c = getc(fp)) != EOF && c != '\n' ) {
        p->symbol = c ;
        p = next_square(p);
    }
    
    fclose(fp);
    
    return p2 ;
}


RULE *read_ruleset( char *filename )
{
    FILE  *fp ;
    RULE  *r, *ir, *p ;
    char  s[MAXCHAR] ;
    char  sst[3] ;
    char  sym, nsym, m ;
    int   st ;
    
    ir = r = (RULE*) malloc( sizeof( RULE ) ) ;

    
    if( ( fp = fopen( filename, "r") ) == NULL ) {
    	if ( errno == ENOENT )
    		fprintf( stderr, "%s: %s", filename, MSG_ERR_FILENOTFOUND ) ;
    	else if ( errno == EACCES )
    		fprintf( stderr, "%s: %s", filename, MSG_ERR_PERMDENIED ) ;
    	else fprintf( stderr, "%s: %s", filename, MSG_ERR_IOERR ) ; 
    	exit(-1) ;

    }
    
    while ( fgets( s, MAXCHAR, fp ) != NULL ) {
    	if ( s[0] == '#' || s[0] < 33 ) continue ;
        sscanf( s, "%i %c %c %c %s", &st, &sym, &nsym, &m, sst ) ;
        r->state = st ;
        if ( sym == '!' ) sym = BLANK ;
        r->symbol = sym ;
        if ( sst[0] == 'H' ) r->new_state = HALT ;
        else r->new_state = atoi( sst ) ;
        r->new_symbol = nsym ;
        switch ( m ) {
            case 'L' : 
                r->shift = LEFT ;
                break ;
            case 'R' : 
                r->shift = RIGHT ;
                break ;
            case 'N' : 
                r->shift = NONE ;
                break ;
            default : 
                fprintf(stderr, MSG_ERR_RULESET ) ;
                exit(-1);
        }
    	r->next = (RULE*) malloc( sizeof( RULE ) ) ;
    	if ( r->next == NULL ) {
      		printf( MSG_ERR_NOMEM ) ; exit (-1) ;
    	}
    	
    	p = r ;
    	r = r->next ;
    }
    
    free( p->next ) ;
    p->next = NULL ;
    
    fclose( fp ) ;
    
    return ir ;
}



int print_tape( void )
{
	SQUARE* sq ;
	
	sq = mark_A ;
	
	while (sq != NULL ) {
		printf( "%c", sq->symbol ) ;
		sq = sq->next ;
	}
	printf("\n") ;
	return 0 ;	
}

	
void print_rule(RULE *r)
{
	char st[10];
	char sh ;
	
	if (r->new_state != HALT ) itoa( r->new_state, st ) ;
	switch ( r->shift ) {
		case NONE:
			sh = 'N' ;
			break ;
		case LEFT:
			sh = 'L' ;
			break ;
		case RIGHT:
			sh = 'R' ;
	}
	printf("%i %c %c %c %s\n",
		r->state,
		r->symbol == BLANK ? '!' : r->symbol,
		r->new_symbol,
		sh,
		r->new_state == HALT ? "H" : st
	) ;
}

	
int main ( int argc, char *argv[] )
{
    SQUARE  *sq ;
	RULE    *r, *ir ;    
    int     c, st, i ;
    int     nargt = -1 ;
    int     nargr = -1 ;
    
    verbose  = FALSE ;
    mverbose = FALSE ;
    
    i = 1 ;
    while ( i < argc && argv[i][0] == '-' ) {
    	c = argv[i][1] ;
    	switch ( c ) {
    		case 'v' :
    			verbose = TRUE ;
    			break ;
    		case 'V' :
    			verbose  = TRUE ;
    			mverbose = TRUE ;
    			break ;
    		case 't' :
    			nargt = ++i ;
    			break ;
    		case 'r' :
    			nargr = ++i ;
    			break ;
    		default :
    			printf( "%c: %s", c, MSG_ERR_ILLEGALOPT ) ;
    			printf( "%s", MSG_HELP ) ;
    			exit( -1 ) ;
    	}
    	++i ;
    }
    
    		 
    sq =     read_tape( nargt != -1 ? argv[nargt] : "tape" ) ;
    ir =  read_ruleset( nargr != -1 ? argv[nargr] : "ruleset" ) ;
    st = 1 ;
    r  = ir ;
    
    printf( "\n%s", MSG_TAPE ) ; print_tape() ; printf( "\n" ) ;
    
    do {
    	if ( r == NULL ) {
    		printf( MSG_ERR_NORULE ) ;
    		exit (-1) ;
    	}
    	if ( r->state == st && sq->symbol == r->symbol ) {
    		if (mverbose) {
    			printf( MSG_RULETOAPPLY ) ;
    			print_rule( r ) ;
    		}
    		st = r->new_state ;
    		sq->symbol = r->new_symbol ;
    		switch ( r->shift ) {
    			case LEFT :
    				sq = next_square( sq ) ;
    				break ;
    			case RIGHT :
    				sq = prev_square( sq ) ;
    				break ;
    			case NONE :
    				break ;
    		}
    		r = ir ;
    		if (verbose) {
    			printf( MSG_TAPESTATE ) ; print_tape() ;
    		}
    	} else {
    		r = r->next ;
    	}
    } while ( st != HALT ) ;
    
    printf( "\n%s", MSG_REACHEDHALT ) ;
    printf( MSG_TAPE ) ; print_tape() ; printf( "\n" ) ; 
    return 0 ;

}
