#!/usr/bin/perl
#-------------------------------------------------------------------------------
# Copyright (C) 2005 by Soumen Dass (soumen_dass@mtnl.net.in). 
# All rights reserved.
#
# This material was originally developed by Soumen Dass in his
# personal capacity/time at Mumbai, India during Apr 2005.
#
# This program is FREE SOFTWARE; you can redistribute it and/or 
# modify it under the terms of the GNU General Public License 
# as published by the Free Software Foundation; either version 2 
# of the License, or (at your option) any later version. 
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
#
# The author would be more than glad to know of ways to improve
# this program. Feel free to suggest/correct.
#
#-----------+-------------------------------------------------------------------
# Purpose   | Concurrnt exeuction of a given program at a given parallel degree.
#           | with variable parameters.
#-----------+-------------------------------------------------------------------
# Version   | 1.0.0
#-----------+-------------------------------------------------------------------
# Features/ | [** NOTE **] Run this program with -h option to see the help doc.
# Guide     |
#           |
#-----------+-------------------------------------------------------------------

use warnings;
use strict;
use File::Basename;
use Getopt::Std;
use POSIX qw(:sys_wait_h);
use File::Temp qw(tempfile);
use 5.6.0;

#-------------------------------------------------------------------------------
# Variables and prototypes
#-------------------------------------------------------------------------------
our	($zombies, $refresh, $fired, $plh, $uname) = (0, 0, 0, '@s', basename $0);
our	($inp_param_file, $par_level, $prg_name, $file_having_par_level, $auto_par_update);
our	($version, $slptime, $par_from_file) = (qw(1.0.0), 1/8, 0);
our ($prev_mtime, $curr_mtime);
our	(@prg_args, @inplist, @pids);
our ($quiet_mode, $slpctr) = (0, 0);

our $print_to_debug = ($ENV{'debug'}) ? 1 : 0;
$Getopt::Std::OUTPUT_HELP_VERSION = 1;

sub show_help_doc;
sub HELP_MESSAGE;
sub VERSION_MESSAGE;
sub par_level_format_ok;
sub print_to_debug;
sub reaper;
sub reap_all;
sub forky;

#-------------------------------------------------------------------------------
# Help messages
#-------------------------------------------------------------------------------

sub VERSION_MESSAGE {
	print "$uname : version $version \n";
	exit(2);
}

sub HELP_MESSAGE {
print <<'HELP';
Copyright (C) 2005 by Soumen Dass (soumen_dass@mtnl.net.in). 

runpar [-vhds] -i ifile -n plevel [-f lfile [-a]] -p prg [args...@s...args]

use option -h to See help doc.
HELP
exit(2);
}

#-------------------------------------------------------------------------------
# Take and process inputs
#-------------------------------------------------------------------------------
my %options;
$Getopt::Std::OUTPUT_HELP_VERSION = 1;
getopts(q(i:n:f:p:hvdsa), \%options);

$inp_param_file			= $options{'i'};
$par_level 				= $options{'n'};
$prg_name  				= $options{'p'};
$file_having_par_level	= $options{'f'};
$quiet_mode  			= $options{'s'};
$print_to_debug 		= $options{'d'} unless ($print_to_debug);

VERSION_MESSAGE	if (exists $options{'v'});
show_help_doc 	if (exists $options{'h'});
foreach (qw(i p n)) {
	HELP_MESSAGE unless ($options{$_});
}
if (exists $options{'f'}) {
	HELP_MESSAGE unless ($file_having_par_level);
	++$par_from_file;
}
++$auto_par_update if (($options{'a'}) and ($par_from_file));

#-------------------------------------------------------------------------------
# Input checks
#-------------------------------------------------------------------------------
(-e $inp_param_file)
	or die "Error : Could not find input file $inp_param_file with input values\n";
(-s _)
	or die "Error : Input file $inp_param_file is an empty file\n";
(-e $prg_name)
	or die "Error : Could not find prg file $prg_name\n";
(-x _)
	or die "Error : Prg file $prg_name not executable\n";
(par_level_format_ok)
	or die "Error : Invalid par level $par_level (valid range 1-99)\n";
@prg_args = @ARGV;


#-------------------------------------------------------------------------------
# Display the parameters for debugging
#-------------------------------------------------------------------------------
if ($print_to_debug) {
	print_to_debug(	"inp_param_file=$inp_param_file par_level=$par_level",
					"prg_name=$prg_name prg_args=@prg_args");
	#print "Hit return to continue..."; (my $val = <STDIN>);
}

#-------------------------------------------------------------------------------
# Read the input list
#-------------------------------------------------------------------------------
open PL, "< $inp_param_file"
	or die "Could not open : file $inp_param_file with input values: $!\n";
while (my $val = <PL>) {
	next if ($val =~ m/^\s*$/g);
	chomp($val);
	push @inplist, $val;
}
close PL;
die "Empty file $inp_param_file / no values found for input\n" unless (@inplist);
print_to_debug(@inplist);

#-------------------------------------------------------------------------------
# Record par level
#-------------------------------------------------------------------------------
$par_level = ($par_level < @inplist) ? $par_level : @inplist;
if ($par_from_file) {
	open PS, "> $file_having_par_level" 
		or die "Error : Could not open file $file_having_par_level to record initial par_level $par_level : $!\n";
	print PS $par_level;
	close PS;
}

#-------------------------------------------------------------------------------
# Core subs
#-------------------------------------------------------------------------------
sub show_help_doc {
	#my $doc_cmd="/usr/perl5/5.6.1/bin/perldoc -F $uname";
	my $doc_cmd="perldoc -F $uname";
	$ENV{'PERLDOC_PAGER'} = 'less -+C -E -P "Page down-Space / Page up-b / Quit-q <--(%P\%)-->"';
	exec("$doc_cmd") if (exists $options{'h'});
}

sub par_level_format_ok {
	return (($par_level =~ m/^[0-9]{1,2}$/o) and ($par_level >= 1 && $par_level <= 99)) ? 1 : 0;
}

sub print_to_debug {
	if ($print_to_debug) {
		print "$_\n" foreach (@_);
	}
}

sub reaper {
	my @livepids;

	foreach my $pid (@pids) {
		if (waitpid($pid, WNOHANG) > 0) {
			--$fired;
			--$zombies;
		} else {
			push @livepids, $pid;
		}
	}
	return @livepids;
}

sub reap_all {
	1 until (wait() == -1);
	exit(1);
}

sub slp {
	select(undef, undef, undef, $slptime);
	++$slpctr;
}

sub reload_par_level {
	--$refresh if ($refresh);
	my $dontchange = 0;
	if ( ($curr_mtime = (stat($file_having_par_level))[9]) > $prev_mtime ) {
		$prev_mtime = $curr_mtime;
		if (open PS, "< $file_having_par_level") {
			if (seek(PS, 0, 0)) {
				chomp($par_level = <PS>);
				close PS;

				unless (par_level_format_ok) {
					warn "Warn : Invalid par level $par_level (valid range 1-99)\n";
					++$dontchange;
				}
			} else {
				warn "Warn : Could not move seek pointer to beginning : $!\n";
				++$dontchange;
			}

		} else {
			warn "Warn : Could not open $file_having_par_level to read fresh par_level : $!\n";
			++$dontchange;
		}
		warn "Warn : Continuing with existing par_level = $par_level\n" if ($dontchange);
	}
}

#-------------------------------------------------------------------------------
# The Core
#-------------------------------------------------------------------------------
$prev_mtime = (stat($file_having_par_level))[9] if ($par_from_file);
sub forky {

	local $SIG{'INT'}  = sub { reap_all };
	local $SIG{'QUIT'} = sub { reap_all };
	local $SIG{'CHLD'} = sub { $zombies++ };
	local $SIG{'USR2'} = sub { $refresh++ };

	my ($pid, $cmd, $entry);
	my $cmdline = ($quiet_mode)	?	"$prg_name @prg_args 1>/dev/null"
								:	"$prg_name @prg_args";
	while (1) {
		last unless(@inplist);
		print_to_debug("Fired count : $fired");
		while($fired >= $par_level) {
			($zombies) ? (@pids = reaper) : slp;
		}
		reload_par_level if ($par_from_file and ($refresh or $auto_par_update));

		$entry = shift @inplist;
		($cmd = $cmdline) =~ s/$plh/$entry/g;
		if ($pid = fork) {
			++$fired;
			push @pids, $pid;
			print_to_debug("Fired $cmd");
		} else {
			die "Could not fork : $cmd : $!\n" unless defined $pid;
			exec("$cmd") or die "Could not exec : $!\n";
		}
	}
	print_to_debug("Reaping leftovers");
	1 until (wait() == -1);
	print_to_debug("Sleep counter : $slpctr\n");
}

forky;

__END__

#-----------------------------------------------------------------------------------------------------
# Program Documentation
#-----------------------------------------------------------------------------------------------------

=pod

=head1 NAME

runpar - Perl program for concurrent exeuction of a given program at a given parallel degree

=head1 SYNOPSIS

B<runpar [-vhds] -i ifile -n plevel [-f lfile [-a]] -p prg [args...@s...args]>


=head1 OVERVIEW

This program enables concurrnt exeuction of a given program at a given parallel degree with variable parameters. It also provides the functionality to alter the degree of parallelism during runtime in order to fine tune system resource usage.

=head1 DESCRIPTION

runpar can be used for execution of a given program (say B<prg>) for a list of values in a flat file (say B<ifile>) with B<n> number of values at a time. The moment the execution is completed for any value, next one is picked from B<ifile>. The idea is keep B<prg> running for exactly B<n> number of values (no more and no less) until it has been executed for all the values in B<ifile>. 

Firing B<prg> in the background for all the entries in B<ifile> simultaneously may overload the system. On the other hand, sequential execution results in wastage of time. This program attempts to be a middle approach.

=head1 INSTALLATION

It can be called from any directory and does not require write permissions (except where B<Option -p> is used. See section OPTIONS for details.)


=head1 OPTIONS

The following options are supported:


=over 4

=item B<-i>

B<ifile> Contains the list of values which are picked up parallely. 

E.g. If program "a" has to execute 100 entries with 5 degrees of parallelism, those 100 entries are placed in B<ifile>. These entries are picked from B<ifile> to run 5 copies of "a" with 5 different entries at a time, till all entries are exhausted.

=item B<-n>

B<plevel> is the parallel level i.e. max number of concurrent instances of B<prg> running at any given time. The program tries to maintain this level throughout its execution (except when the number of entries left is less than B<plevel>). Currently the valid range is 1-99.

=item B<-f>

This option provides a way to alter the degree of parallelism runtime. This is done by overwriting/creating B<lfile> upon startup by placing B<plevel> (the argument to B<-n>) as initial value. To alter the degree, the new value should be put in B<lfile>. After this, there are two ways to let the running process know of the change:

B<Manual>

=over 6

=item 

Find the process_id of runpar and issue:

=item 

B<kill -USR2 <runpar_process_idE<gt>>

=item 

The process would immediately read the change and use the new degree for further spawning of remainig entries in B<ifile>.

=back

B<Automatic>

=over 6

=item

See B<Option -a>

=back

Note : User must ensure that B<lfile> is writable. Else the process would fail to start.

=item B<-a>

If this option is provided, the new degree is picked up automatically as and when the change ocurrs in B<lfile>. Since the script has to keep a tab on the B<lfile>, this option has a marginal overhead. 

Note : It is (theoritically) possible that the time, at which script is checking on B<lfile>, conincides with the time the new value is written (which may lead to invalid data being read.) However, the possibilities of this really happening are miniscule. Further, the script has several in-built checks which would simply discard any invalid data and continue with the existing level.

This option is redundant without -f option.

=item B<-p>

B<prg> is the program which is executed parallelly. This could be any non-interactive executable.

=item B<-d>

displays debuging data. Verbosity has its own overhead so please be judicious in usage.

=item B<-v>

displays program version

=item B<-h>

displays this help

=item B<-s>

Silent mode. The output from B<prg> is suppressed. However errors (if any) continue to be displayed.

=item B<arg>

These arguments are passwd to prg.

=item B<@s>

B<@s> is the placeholder which is replaced with the entries from B<ifile>.

=back

=head1 EXAMPLES

Suppose, B<ifile> has following entries (say srl nos.):

	0001
	0002
	0003
	0004
	0005
	0006

Suppose, program B<prg> takes two agruments, srl no. and current date.

=over 4

=item 1

To run B<prg> with 3 entries at a given point:

$ runpar -i B<ifile> -n 3 -p B<prg> @s 30-04-2005

=item 2

To run B<prg> with the option to change the B<plevel> runtime (manually):

$ runpar -i B<ifile> -n 3 -f B<lfile> -p B<prg> @s 30-04-2005

If you need to change the B<plevel> to 6, run following command:

$ echo "6" > B<lfile> 

Then find the process_id of runpar using ps and issue:

$ B<kill -USR2 <runpar_process_idE<gt>>

=item 3

To run B<prg> with the option to change the B<plevel> runtime (automatically):

runpar -i B<ifile> -n 3 -f B<lfile> -a -p B<prg> @s 30-04-2005

If you need to change the B<plevel> to 6, run following commands from the command prompt: 

echo "6" > B<lfile> 

runpar would alter the B<plevel> during it's next read of B<lfile> (which is reasonably
prompt.)

==over 4

=head1 NOTES

B<Strucutre>

Typical while loop based approaches lose time, speed and resource by:

=over 4

=item *

Excessive sleeping

=item *

Hog CPU resources by not sleeping at all.

=item *

Fire repeated external commands to check process count/status.

=back

This program uses signal handling to balance on all the fronts
resulting in:

=over 4

=item *

Minimal sleep (currently 0.125 secs per sleep; configurable)

=item *

Low resource consumption vis-a-vis speed

=item *

Low I/O and process overhead. Only fast syscalls are used.  Not a single external unix binary is invoked (except for displaying this man page :)

=back

B<Speed estimate>

At the par level 12 for 500 entries and assuming that the runner comes out instantly, the (THEORITICALLY) max loss of time due to the program logic has been brought down to 1 minute.  In actual runs with the above parameters, the real loss was found to be beteen 0 to 16 seconds.

=head1 AUTHOR

Soumen Dass (B<soumen_dass@mtnl.net.in>)

=head1 COPYRIGHT NOTICE

COPYRIGHT NOTICE

Copyright (C) 2005 by Soumen Dass (soumen_dass@mtnl.net.in). 

Permission is granted to make and distribute verbatim  copies  of  this manual  provided  the  copyright  notice and this permission notice are preserved on all copies.

Permission is granted to copy and distribute modified versions of  this manual  under  the  conditions for verbatim copying, provided that the entire resulting derived work is distributed under the terms of a  permission notice identical to this one.

Permission  is granted to copy and distribute translations of this manual into another language, under the above conditions for modified versions,  except  that this permission notice may be stated in a translation approved by the Foundation.

=cut
