/*
Copyright (c) 2000 David A. Caswell

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
*/

/* 
 * A program to read data from the Radioshack model 22-805
 * voltmeter by Dave Caswell, davec@apocalypse.org, September 2000
 */

/*
 *$Id: meter.c,v 1.1 2000/10/11 06:39:20 davec Exp $
 */

#include <getopt.h>
#include <fcntl.h>
#include <termios.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/ioctl.h>
#include <time.h>
#include <sys/timeb.h>
#include <string.h>
#include <signal.h>

#define VERSION_MAJOR 1 
#define VERSION_MINOR 0

/*
 *#include <asm/termios.h>
 * Where should these values come from, for portability?
 */
#define TIOCM_RTS	0x004
#define TIOCMGET	0x5415
#define TIOCMBIS	0x5416
#define TIOCMBIC	0x5417
#define TIOCMSET	0x5418

int clock_time_flag=0;
int elapsed_time_flag=0;
int multiple_flag=0;
int echo_flag=0;
int scientific_flag=0;
int count=1;
int interval=10;
char * outputfile=0;
char * inputdevice="/dev/ttyS0";

int sigint=0;
int sigalrm=0;

typedef void (*sighandler_t)(int);
void sigint_handler( int i)
{
    sigint=1;
}
void sigalrm_handler(int i)
{
    sigalrm=1;
}

int main( int argc, char ** argv)
{
    int fd;
    FILE* outfile=stdout;
    struct termios oldtermattr;
    struct termios termattr;
    int status;
    int oldstatus;
    char buf[255];
    int i;
    struct timeb start_time;
    struct timeb test_time;
    long elapsed, timenext;
    sighandler_t int_handler;
    sighandler_t alrm_handler;


    start_time.time=0;

    while(1){
	int c;
	static struct option long_options[] =
	{
	    /* These options set a flag. */
	    {"multiple", 0, &multiple_flag, 1},
	    {"clock-time", 0, &clock_time_flag, 1},
	    {"elapsed-time", 0, &elapsed_time_flag, 1},
	    {"echo", 0, &echo_flag, 1},
	    {"strip", 0, &scientific_flag, 1},
	    /* These options don't set a flag.
	       We distinguish them by their indices. */
	    {"output-file", 1, 0, 'f'},
	    {"count", 1, 0, 'n'},
	    {"interval", 1, 0, 'i'},
	    {"device", 1, 0, 'd'},
	    {"version", 0, 0, 'v'},
	    {"help", 0, 0, 'h'},
	    {0, 0, 0, 0}
	};
	int option_index = 0;
	
	c = getopt_long (argc, argv, "?cd:ef:hi:mn:stv",
			 long_options, &option_index);
     
	/* Detect the end of the options. */
	if (c == -1)
	    break;
	switch(c){
	case 'c':
	    clock_time_flag=1;
	    break;
	case 'd':
	    inputdevice=optarg;
	    break;
	case 'e':
	    echo_flag=1;
	    break;
	case 'f':
	    outputfile=optarg;
	    break;
	case 'h':
	case '?':
	    printf("Meter reads a value from a RadioShack multi-meter with a PC interface.\n");

    	    printf("With no options meter will place one value from the ");
            printf("multi-meter on stdout.\n");

	    printf("options:\n");
	    printf("  %6s, %-22s ","-c", "--clock-time ");    
	    printf("show the times when readings were made\n");

	    printf("  %6s, %-22s ","-t", "--elapsed-time ");    
	    printf("show the seconds elapsed since the first reading.\n");

	    printf("  %6s, %-22s ","-d DEV", "--device=DEV");    
	    printf("serial device the meter is connected to\n");    
	    printf("                                 ");
	    printf("defaults to /dev/ttyS0\n");    

	    printf("  %6s, %-22s ","-e", "--echo");    
	    printf("echoes the data to stdout as well as to the\n");    
	    printf("                                 ");
	    printf("specified output file.\n");

	    printf("  %6s, %-22s ","-n N", "--count=N");    
	    printf("take N samples\n");

	    printf("  %6s, %-22s ","-f FN", "--output-file=FN");    
	    printf("the file to write data to.\n");

	    printf("  %6s, %-22s ","-h", "--help");    
	    printf("this help message\n");

	    printf("  %6s, %-22s ","-i N", "--interval=N");    
	    printf("interval between samples in seconds\n");    

	    printf("  %6s, %-22s ","-m", "--multiple");    
	    printf("take samples continuously\n");

	    printf("  %6s, %-22s ","-s", "--scientific");    
	    printf("output values in scientific notation.\n");

	    printf("  %6s, %-22s ","-v", "--version");    
	    printf("show version.\n");

	    exit(0);
	    break;
	case 'i':
	    interval=atoi(optarg)*10;
	    if(interval<=5) interval=10;
	    break;
	case 'm':
	    multiple_flag=1;
	    break;
	case 'n':
	    count=atoi(optarg);
	    if(count<=0)count=1;
	    break;
	case 's':
	    scientific_flag=1;
	    break;
	case 't':
	    elapsed_time_flag=1;
	    break;
	case 'v':
	    printf("Version %d.%d\n", VERSION_MAJOR, VERSION_MINOR);
	    exit(0);
	    break;
	}
    }

    int_handler=signal(SIGINT, sigint_handler);
    alrm_handler=signal(SIGALRM, sigalrm_handler);
    /* linux will try and restart an interrupted system call by default */
    siginterrupt(SIGINT, 1); /* stop system calls on SIGINT */
    siginterrupt(SIGALRM, 1); /* stop system calls on SIGALRM */

    if( outputfile ){
	if(!(outfile=fopen(outputfile, "w"))){
	    perror("Error opening output file");
	    exit(1);
	}
    }
    if( !outputfile ) echo_flag=0;

    if( (fd=open( inputdevice, O_RDWR|O_NOCTTY|O_NDELAY))<0){
	perror("Opening serial port");
	exit(1);
    }
    fcntl(fd, F_SETFL, 0);
    tcgetattr(fd, &oldtermattr);
    tcgetattr(fd, &termattr);

    termattr.c_lflag &= ~(ISIG|XCASE|ECHO|ECHOE|ECHOK|ECHONL|NOFLSH|IEXTEN|
			  ECHOCTL|ECHOPRT|ECHOKE|FLUSHO|PENDIN|TOSTOP);
    termattr.c_lflag |= (ICANON);
    termattr.c_iflag &= ~(INPCK|PARMRK|IXON|IXOFF|BRKINT|INLCR|IGNCR|
			  IUCLC|IMAXBEL);
    termattr.c_iflag |= IGNPAR|ISTRIP|IXANY|IGNBRK|ICRNL;
    termattr.c_oflag &= ~(OPOST);
    termattr.c_oflag |= 0;
    termattr.c_cflag &= ~(CSIZE|PARENB|PARODD|CRTSCTS);
    termattr.c_cflag |= (CS7|CSTOPB|CREAD|HUPCL|CLOCAL);
    cfsetispeed(&termattr, B600);
    cfsetospeed(&termattr, B600);
    termattr.c_cc[VINTR]=0;
    termattr.c_cc[VQUIT]=0;
    termattr.c_cc[VERASE]=0;
    termattr.c_cc[VKILL]=0;
    termattr.c_cc[VEOF]=0;
    termattr.c_cc[VEOL]='\r';
    termattr.c_cc[VEOL2]='\n';
    termattr.c_cc[VMIN]=0;
    termattr.c_cc[VTIME]=0;
    tcsetattr( fd, TCSAFLUSH, &termattr);

    ioctl(fd, TIOCMGET, &oldstatus);
    ioctl(fd, TIOCMGET, &status);
    status&=~TIOCM_RTS;
    ioctl(fd, TIOCMSET, &status);
    usleep(10000); /* wait for meter interface to power up */

    while(multiple_flag || count-- >0){
	ftime(&test_time);
	if(start_time.time==0){
	    start_time=test_time;
	    timenext=0;
	}
	elapsed=1000*(test_time.time-start_time.time);
	elapsed+=test_time.millitm-start_time.millitm;

	buf[0]='D';
	if(write(fd, buf, 1)<0){
	    perror("Error writing to serial port \n");
	    exit(1);
	}
	/* set an alarm so that we don't loop waiting 
	 * for a meter which is off
	 */
	alarm(2); /* wait 2 seconds for response from meter */
	i=read(fd, buf, sizeof(buf)-1);
	alarm(0); /* clear any waiting alarm. */
	if(sigalrm){
	    sigalrm=0;
	    fprintf(stderr, "No response from meter.\n");
	    continue;
	}
	if(i<0){
	    perror("Error reading from serial port");
	}
	buf[i]=0;	if(i>1)buf[i-1]=0;
	if(scientific_flag){
	    double val;
	    char units[15];
	    if(sscanf(buf+3, "%lg %s", &val, units)!=2){
		val=0;
	    }else{
		switch(units[0]){
		case 'm':
		    val=val/1000.0;
		    break;
		case 'k':
		    val=val*1000.0;
		    break;
		case 'M':
		    val=val*1.0e6;
		    break;
		}
	    }
	    fprintf(outfile, "%11.5g", val);
	    if(echo_flag)printf("%11.5g", val);
	}else{
	    fprintf(outfile, "%s", buf);
	    if(echo_flag)printf("%s", buf);
	}
	if(clock_time_flag){
	    char * timestring;
	    char * ptr;
	    time_t time_now;
	    time_now=time(0);
	    timestring=ctime(&time_now);
	    ptr=strrchr(timestring, '\n');
	    *ptr=0;
	    fprintf(outfile, ", %s", timestring);
	    if(echo_flag)printf(", %s", timestring);
	}
	if(elapsed_time_flag){
	    fprintf(outfile, ", %ld.%01ld", elapsed/1000, elapsed%1000/100 );
	    if(echo_flag)printf(", %ld.%01ld", elapsed/1000, elapsed%1000/100 );
	}
	fprintf(outfile, "\n");
	if(echo_flag)printf("\n");
	fflush(outfile);
	if(count==0) break;
	ftime(&test_time);
	elapsed=1000*(test_time.time-start_time.time);
	elapsed+=test_time.millitm-start_time.millitm;
	timenext=timenext+interval*100;
	if(sigint)break;
	usleep((timenext-elapsed)*1000);
	if(sigint)break;
    }
    tcsetattr( fd, TCSAFLUSH, &oldtermattr);
    ioctl(fd, TIOCMSET, &oldstatus);
    close(fd);
    fclose(outfile);
    return 0;
}

/*
 *$Log: meter.c,v $
 *Revision 1.1  2000/10/11 06:39:20  davec
 *Initial revision
 *
 */

/* local variables: */
/* c-basic-offset: 4 */
/* end: */
