/*
This program is free software; you can redistribute it and/or modify
it under the ; (at your option) any later version.
This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

Copyright Jan panteltje 2009-always 

read and write routines to grab data from a 433.X MHz receiver, and send data to a 433.X MHz transmitter
*/


#include "lremote.h"



char *strsave(char *s) /* save char array s somewhere */
{
char *p;

p = malloc( strlen(s) +  1);
if(! p) return 0;

strcpy(p, s);
return p;
} /* end function strsave */



int print_usage()
{
fprintf(stderr, "\nPanteltje (c) lremote-%s\n", VERSION);
fprintf(stderr, "Usage: lremote [-d device_name] [-h] [-l] [-p] [-s command_filename] [z]\n");
fprintf(stderr, "-d device_name    name of output device for RTS, default /dev/ttyS0.\n");
fprintf(stderr, "-h                help, this help.\n");
fprintf(stderr, "-l                learn and record remote command to file, input always via parport at 0x378 pin 15.\n");
fprintf(stderr, "-p                output via parport D2 pin 4, not via serial device.\n");
fprintf(stderr, "-s filename       send a remote command file.\n");
fprintf(stderr, "-v                debug mode, prints functions and arguments.\n");
fprintf(stderr, "-z                set transmitter off, output to zero, or to minus for the serial RTS pin.\n");	
fprintf(stderr, "ATTENTION: When using RTS for output, put a diode in series with RTS, so the transmitter can only be driven positive, as RTS goes negative when low.\n");
fprintf(stderr, "Examples:\n");
fprintf(stderr, "lremote -l                         learns a command, connect receiver to par port 0x378 pin 15, follow instructions on screen.\n"); 
fprintf(stderr, "lremote -s on -p                   sends command via parport pin 4.\n");
fprintf(stderr, "lremote -s on -d /dev/ttyS1        sends command via serial port ttyS1, RTS.\n");
fprintf(stderr, "lremote -z -p                      sets parport pin 4 to zero, transmitter off, use in init.\n");
fprintf(stderr, "lremote -z                         sets RTS to negative on default serial port /dev/ttyS1, transmitter off, use in init.\n");  

return 1;
} /* end function print_usage */



int main(int argc, char **argv)
{
int a;
int transmit_flag;
char *transmit_command;
int option;

iopl(3);

/* defaults */
debug_flag = 0;
transmit_flag = 0;
par_port_address = 0x378;
output_mode = MODE_SERIAL_OUT;
//MODE_PARPORT_OUT


serial_device_name = strsave(DEFAULT_SERIAL_DEVICE);
if(! serial_device_name)
	{
	fprintf(stderr, "lremote: main(): could not allocate space for serial_device_name, abortin.\n");

	exit(1);
	}

transmit_command = 0;
/* end defaults */


fprintf(stderr, "Panteltje (c) lremote%s\n", VERSION);

option = 0;
while(1)
	{
	a = getopt(argc, argv, "d:hlps:vz");
	if(a == -1) break;
	option = 1;
	switch(a)
		{
		case 'd': // serial device
			free(serial_device_name);
			serial_device_name = strsave(optarg);
			if(! serial_device_name)
				{
				fprintf(stderr, "col_pic: could not allocate space for serial device, aborting.\n");

				exit(1);
				}

			break;
		case 'h': // help
			print_usage();
			exit(1);
			break;
		case 'l':	// learn mode
			//int ir_remote_learn()/*learn a remote command*/
			ir_remote_learn();
			exit(0);
			break;			
		case 'p': // output via parport D2 pin 4.
			output_mode = MODE_PARPORT_OUT;
			break;
		case 's': // send mode
			transmit_flag = 1;
			transmit_command = strsave(optarg);
			if(! transmit_command)
				{
				fprintf(stderr, "lremote: main(): could not allocate space for transmit_command, aborting.\n");
				
				exit(1);
				}
			break;		
		case 'v': // debug
			debug_flag = 1;
			break;
		case 'z': // output to 0V or engative for RTS on serial port.
			set_zero(output_mode);
			break;
        case -1:
        	break;
		case '?':
			if (isprint(optopt) )
 				{
 				fprintf(stderr, "Unknown option `-%c'.\n", optopt);
 				}
			else
				{
				fprintf(stderr, "Unknown option character `\\x%x'.\n",\
				optopt);
				}
			print_usage();

			exit(1);
			break;			
		default:
			print_usage();

			exit(1);
			break;
		}/* end switch a */
	}/* end while getopt() */

if(! option)
	{
	print_usage();
	}

if(transmit_flag)
	{
	if(output_mode == MODE_SERIAL_OUT)
		{
		serial_send(transmit_command);

		exit(0);	
		}
	else if(output_mode == MODE_PARPORT_OUT)
		{	
		//int ir_remote_send(char command[]) /*send a remote command */
		ir_remote_send(transmit_command);

		exit(0);
		}
	}

exit(1);
} /* end function main */



int set_zero(int output_mode)
{
if(output_mode == MODE_SERIAL_OUT)
	{
	set_rts_off();
	
	return 1;
	}

else if(output_mode == MODE_PARPORT_OUT)
	{
	ir_off();	

	return 1;	
	}

return 0;
}


