/************************************************/
/*						*/
/*	addrlist.c				*/
/*						*/
/*	This module contains functions that 	*/  
/*	build and manipulate the address list	*/
/*	which is a linked list.			*/
/*						*/
/*	Luther J. Maloney	Oct 1 1997	*/
/*						*/
/************************************************/
#include "addrbk.h"

static void move_addr_data(ADDR_LIST *output,IO_ADDR *input);

/*function to preform initial read of the address file*/
/*the file has already been opened in another module  */
/*the input file is in fixed length format previously */
/*written by this program. The file is read into a    */ 
/*linked list, so that searches are possible without  */
/*performing  extensive I/O. Also allows items to be  */
/*inserted easily. 				      */
/*parameters : address of the first item in the list  */
/*	       pointer to the opened file	      */
/*returns : a linked list of addresses		      */

int build_addr_list(ADDR_LIST *head_ptr,FILE *addr_file)
{
	ADDR_LIST *first_node;
	ADDR_LIST *current_node;
	ADDR_LIST *new_node;
	IO_ADDR input_addr;
	int	eof; 

	head_ptr->next_item = NULL;
	head_ptr->prev_item = NULL;
	first_node = current_node = head_ptr;
	
	memset(current_node,'\0',sizeof(ADDR_LIST));
	memset((void *)&input_addr,'\0',sizeof(IO_ADDR));

	fseek(addr_file,0,SEEK_SET);

	if(feof(addr_file)!=0)
		return(SUCCESS);

	do
	{
	   fread(&input_addr,sizeof(IO_ADDR),1,addr_file);

	   /*strip the blanks and move to list*/

	   move_addr_data(current_node,&input_addr);

	   strip_blanks(current_node);

	   if(feof(addr_file) == 0)
	   {
	      new_node = (ADDR_LIST *)malloc(sizeof(ADDR_LIST));

	      if(new_node == NULL)
	      {
	 	fprintf(stderr,"\ERROR PERFORMING MEMORY ALLOCATION : ABORT");
		exit(255);
	      }
		
	      memset(new_node,'\0',sizeof(ADDR_LIST));
	      new_node->next_item = NULL;
	      new_node->prev_item = current_node;
	      current_node->next_item = new_node;

	      current_node = current_node->next_item;
	   }
	}while(feof(addr_file)==0);

	return(SUCCESS);
}

/*this function inserts a node into the address list */
/*the list is searched in alphabetical order to      */  
/*the appropriate place to insert the new node	     */
/*data verification should have been preformed prior */
/*to entering this function.			     */
/*parameters : address of first item in the list     */
/*             pointer to the struct of data to be   */
/*		       inserted			     */
/*returns : linked list with the new item inserted   */

ADDR_LIST *insert_node_addr_list(ADDR_LIST *head_ptr,IO_ADDR *new_address)
{
	ADDR_LIST *current_node = NULL;
	ADDR_LIST *prev_node = NULL;
	ADDR_LIST *next_node = NULL;
	ADDR_LIST *new_node = NULL;
	ADDR_LIST *first_node;

	/*allocate memory for the new node and move data to the node*/

	new_node = (ADDR_LIST *)malloc(sizeof(ADDR_LIST));

	if(new_node == NULL)
	{
	   fprintf(stderr,"\nERROR ALLOCATING MEMORY : ABORT!");
	}

	memset(new_node,'\0',sizeof(ADDR_LIST));
	new_node->next_item = NULL;
	move_addr_data(new_node,new_address);

	/* go through the linked list until next_item == NULL */
	/* or the item has been ineserted */
	/* the list is sorted based on last_name,first_name*/

	/*check to see if the list is empty or only one item*/

	if(!strncmp(head_ptr->f_name,"EMPTY",sizeof("EMPTY"))) 
	{
	   /*the first node is empty, insert here*/

	   head_ptr = new_node;

	   head_ptr->prev_item = NULL;

	   return(head_ptr);   
	}
	
	/*if there is only one item in the list, determine*/
	/*to insert before or after it */

	if (head_ptr->next_item == NULL)
	{
	   /*there is one item in the list*/
	   if(strcmp(head_ptr->l_name,new_node->l_name) < 0)
	   {
		/*insert the new node after the first node*/
		new_node->prev_item = head_ptr;
		head_ptr->next_item = new_node;
		return(head_ptr);
	   }
	   else if (strcmp(head_ptr->l_name,new_node->l_name) > 0)
	   {
		/*insert the new node before the first node*/
		new_node->prev_item = NULL;
		head_ptr->prev_item = new_node;
		new_node->next_item = head_ptr;
		head_ptr  = new_node;
		return(head_ptr);
	   }
	   else if (strcmp(head_ptr->l_name,new_node->l_name) == 0)
	   {
		/*there is a match on last name, test first name*/
		if(strcmp(head_ptr->f_name,new_node->f_name) < 0)
		{
		   /*insert after the first node*/
		   new_node->prev_item = head_ptr;
		   head_ptr->next_item = new_node;
		   return(head_ptr);
		}
		else if (strcmp(head_ptr->f_name,new_node->f_name) > 0)
		{
		    /*insert before first node */
		    new_node->prev_item = NULL;
		    head_ptr->prev_item = new_node;
		    new_node->next_item = head_ptr;
		    head_ptr = new_node;
		    return(head_ptr);
		}
		else
		{
		   /*default, insert after first node*/
		   new_node->prev_item = head_ptr;
		   head_ptr->next_item = new_node;
		   return(head_ptr);
		}
	   }
	}
	
	/*if this point has been reached, then the list contains more */
	/*than two items. Use a while loop to insert the new node     */

	/*determine if the new item is to be inserted prior */            
	/*to the first node*/

	if (strcmp(head_ptr->l_name,new_node->l_name) > 0)
	{
	   /*insert the new node before the first node*/
	   next_node = head_ptr;
	   new_node->prev_item = NULL;
	   head_ptr->prev_item = new_node;
	   new_node->next_item = head_ptr;
	   head_ptr = new_node;
	   return(head_ptr);
	}
	else if (strcmp(head_ptr->l_name,new_node->l_name) == 0)
	{
	   /*there is a match on last name, test first name*/
	   if ((strcmp(head_ptr->f_name,new_node->f_name) > 0 ) ||
	       (strcmp(head_ptr->f_name,new_node->f_name) == 0))
	   {
	       /*insert before first node */
	       new_node->prev_item = NULL;
	       head_ptr->prev_item = head_ptr;
	       new_node->next_item = head_ptr;
	       head_ptr = new_node;
	       return(head_ptr);
	   }
	}

	current_node = head_ptr;
	
	while(current_node != NULL)
	{
	   next_node = current_node->next_item;

	   if(next_node == NULL)
	   {
		/*end of list was reached, insert at the end*/

	        new_node->prev_item = current_node;
		current_node->next_item = new_node;
		return(head_ptr);
	   }
	   /*determine if current < new (A<B) */
	   if (strcmp(current_node->l_name,new_node->l_name) < 0)
	   {
		/*check if new < next (B < C)*/
		if (strcmp(new_node->l_name,next_node->l_name) < 0)
		{
		   /*the new node is between current and next*/
		   new_node->prev_item = current_node;
		   new_node->next_item = next_node;
	  	   current_node->next_item = new_node;
		   return(head_ptr);
		}
	   }
	   else if(strcmp(current_node->l_name,new_node->l_name) == 0)
	   {
		/*if last names match*/
		/*check f_name*/
		if (strcmp(current_node->f_name,new_node->f_name) > 0)
		{
		   /*the new node is before current-> first name */
		   prev_node = current_node->prev_item;
		   new_node->prev_item = prev_node;
		   prev_node->next_item = new_node;
		   new_node->next_item = current_node;
		   current_node->prev_item = new_node;
		   return(head_ptr);
		}
		else
		{
		   /*insert the new node after the current node*/
		   new_node->prev_item = current_node;
		   new_node->next_item = next_node;
		   current_node->next_item = new_node;
		   return(head_ptr);
		}
		   
	   }

		prev_node = current_node;
		current_node = current_node->next_item;
	}

	/*the list was completely traversed, could not insert*/

	current_node->next_item = new_node;
	new_node->prev_item = current_node;

	return(head_ptr);

}

/*This function deletes a node from the address list */

int delete_item_addr_list(ADDR_LIST *delete_ptr,ADDR_LIST *prev_node)
{
	prev_node->next_item = delete_ptr->next_item;
	free((void *)delete_ptr);
	return(SUCCESS);
}

/*this function writes the entire list to the address file */

int write_addr_list(ADDR_LIST *head_ptr,FILE *addr_file)
{
	ADDR_LIST *current_node;
	IO_ADDR	  address;

	(void)fseek(addr_file,0,SEEK_SET);

	/*assign head_ptr (pointer to start of list) */
	/*to the current_node			     */

	current_node = head_ptr;

	/*enter loop to write address list to file*/ 

	do	
	{
	   /*assign values from the current node of the linked list*/
	   /* to the io structure (this struct does not have an pointer*/

	   memset((char *)&address,' ',sizeof(IO_ADDR));

	   strncpy(address.f_name,current_node->f_name,strlen(current_node->f_name));
	   strncpy(address.m_name,current_node->m_name,2);
	   strncpy(address.l_name,current_node->l_name,strlen(current_node->l_name));
	   strncpy(address.title,current_node->title,strlen(current_node->title));
	   strncpy(address.company,current_node->company,strlen(current_node->company));
	   strncpy(address.address,current_node->address,strlen(current_node->address));
	   strncpy(address.city,current_node->city,strlen(current_node->city));
	   strncpy(address.state,current_node->state,strlen(current_node->state));
	   strncpy(address.zip,current_node->zip,strlen(current_node->zip));
	   strncpy(address.phone,current_node->phone,strlen(current_node->phone));
	   strncpy(address.fax,current_node->fax,strlen(current_node->fax));
	   strncpy(address.email,current_node->email,strlen(current_node->email));
	   strncpy(address.comment,current_node->comment,strlen(current_node->comment));

	   fwrite((void *)&address,sizeof(IO_ADDR),1,addr_file);

	   fflush(addr_file);

   	   current_node = current_node->next_item;
	
	}while(current_node != NULL);
}
static void move_addr_data(ADDR_LIST *output,IO_ADDR *input)
{
	sprintf(output->f_name,"%.*s",L_FNAME,input->f_name);
	sprintf(output->l_name,"%.*s",L_LNAME,input->l_name);
	sprintf(output->m_name,"%.*s",L_MNAME,input->m_name);
	sprintf(output->title,"%.*s",L_TITLE,input->title);
	sprintf(output->company,"%.*s",L_COMPANY,input->company);
	sprintf(output->address,"%.*s",L_ADDR,input->address);
	sprintf(output->city,"%.*s",L_CITY,input->city); 
	sprintf(output->state,"%.*s",L_STATE,input->state);
	sprintf(output->zip,"%.*s",L_ZIP,input->zip);
	sprintf(output->phone,"%.*s",L_PHONE,input->phone);
	sprintf(output->fax,"%.*s",L_FAX,input->fax);
	sprintf(output->email,"%.*s",L_EMAIL,input->email);
	sprintf(output->comment,"%.*s",L_COMMENT,input->comment);

}
