/* Zgv v4.2 - GIF, JPEG and PBM/PGM/PPM viewer, for VGA PCs running Linux.
 * Copyright (C) 1993-1998 Russell Marks.
 *
 *  Changes by Matan Ziv-Av (zivav@cs.bgu.ac.il)
 *
 * zgv.c - This provides the zgv file selector, and interfaces to the
 *         vga display routines (vgadisp.c)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include <stdio.h>
#include <dirent.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include <signal.h>
#include <setjmp.h>
#include <sys/param.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/file.h>
#include <sys/ioctl.h>
#include <sys/vt.h>
#include <errno.h>
#include <vga.h>

#include "zgv.h"
#include "gifeng.h"
#include "vgadisp.h"
#include "readnbkey.h"
#include "font.h"
#include "3deffects.h"
#include "helppage.h"
#include "rc_config.h"
#include "rcfile.h"
#include "readjpeg.h"   /* we only use this for my_error_exit() */
#include "readpnm.h"	/* for the dithering - should be moved! */
#include "resizepic.h"
#include "mouse.h"
#include "magic.h"

#include "zgvlogopck.h"


char zgvhelp[][80]={
  "? (question mark)\\this help page",
  "/ (slash)\\other help page",
  "arrows\\move file selection cursor",
  "Enter\\display file or change directory",
  "Space\\toggle tagged",
  "t (T)\\tag file (all files)",
  "n (N)\\untag file (all files)",
  "Tab\\slideshow of tagged files",
  "i\\show all files / only image files",
  "v\\visual selection mode on/off",
  "V\\wide bars",
  "u\\create/update thumbnails in current directory",
  "U\\update directory thumbnails in current directory",
  "z\\toggle zgv logo",
  "r\\update thumbnails (including subdirectories)",
  "R\\update directory thumbnails (inc. subdirectories)",
  "Ctrl-R\\update directory listing / redraw screen",
  "D or Delete\\delete file",
  "Esc or x\\exit zgv",
  ""
  };
char zgvhelp2[][80]={
  "1-6 change selection video mode 1- 640x480,  2- 800x600",
  "            3- 1024x768,  4- 1152x864,  5- 1280x1024,  6- 1600x1200",
  "p\\set target directory to current",
  "P\\selecet target directory",
  "m (M)\\move file (tagged file) to target directory",
  "c (C)\\copy file (tagged file) to target directory",
  "d (D)\\remove file (tagged file)",
  "e\\rename current file",
  "o (O)\\sort files by alphabetical order (reversed)",
  "a (A)\\sort files by date, older first (newer first)",
  "s (S)\\sort files by size, smaller first (larger first)",
  "g\\goto directory",
  "j\\toggle one time prompt for multiple file",
  "\\delete/move/copy",
  "k\\toggle prompting for delete/move/copy",
  "l\\toggle looping in slideshow",
  "y\\toggle high color brightness/contrast control",
  "; (semi-colon)\\show file name and details",
  ": (colon)\\show full file name",
  ""
  };

#define fwinxpos(f) (40+(((f)-1)/YSIZ)*BARWIDTH)
#define fwinypos(f) (yofs+70+barheight*(((f)-1)%YSIZ))

/* from 18-bit RGB (as used by VGA palette) to 3:3:2 palette index */
#define MAKE332COL(r,g,b) (((r)>>3)*32+((g)>>3)*4+((b)>>4))


#define LIGHT		2
#define DARK		1
#define BLACK		15
#define MIDGREY		0
#define MARKEDCOL	14

#define GDFYBIT			18
/*#define BARWIDTH		138 */
#define BAR_RESTRICT_WIDTH	(BARWIDTH-13)

#define MDIRSIZ		4096
#define MPPOSSIZ	256

#define COLS            (VIS_WID)
#define YSIZ            ((VIS_HI-195+100*cfg.fullsel)/barheight)
#define XSIZ            ((VIS_WID-40)/BARWIDTH)

int VIS_WID;
int VIS_HI;
int VIS_MODE;
int sort_key;
int BARWIDTH, left_just;
char targetdir[256], homedir[256];

/* number of greyscales *less one* used in selector when in 640x480x16 mode */
#define GREY_MAXVAL	10

struct GDIR {
  char name[256];
  char isdir;            /* 0=file, 1=dir. */
  char xvw,xvh;		/* xvpic width and height, zero if none */
  char marked;
  int width,height;
  off_t size;
  time_t ctime;
  } *gifdir;

struct DIRTREE {
   char name[256];
   int gifdirsize;
   struct GDIR *gifdir;   
   struct DIRTREE *sib, *child;
} rootdir;

int gifdirsiz;

int zgv_ttyfd,howfar_upto;
jmp_buf setjmpbuf;   /* in case someone aborts decompressing a file */
static int xv332_how_far_xpos,xv332_how_far_ypos;
static int one_file_only=0;
int original_vt,separate_vt=0;
int zgv_vt;
int tagview_mode=0;

int idx_light,idx_medium,idx_dark,idx_black,idx_marked;
int updating_index=0;
int v256=1;	/* 1=256-col selector, 0=16-col selector */

int gdfsiz=3;		/* size of filename text in selector */
int gdfofs=0;

int barheight,yofs;	/* height of cursor, and offset of selection bit */

/* XXX this sgres stuff is bogus in the extreme */
extern int sgres;
int has_mouse=0;

int current_colour=0;	/* see zgv.h */

struct pastpos_t
  {
  int dev,inode,curent,startfrom;
  } pastpos[MPPOSSIZ];

/* prototypes */
int main(int argc, char **argv);
void openstdin_nonblocking(void);
void load_one_file(char *filename);
void copyfromconfig(void);
void mainloop(void);
void new_pastpos(int curent, int startfrom);
void get_pastpos(int *curentp,int *startfromp);
int load_single_file(int curent, int do_howfar);
void load_tagged_files(void);
int permissiondenied(char *fname);
void redrawall(int curent, int startfrom);
void inithowfar(void);
void showhowfar(int sofar, int total);
void smallhowfar(int sofar, int total);
void showtargetdir();
void showbar(int entnum, int selected, int startfrom);
int centreseltxt(int x, int fsiz, char *str);
void showgifdir(int startfrom, int unshow, int drawdirmsg);
int ispicture(char *filename);
int is_cfg_picture(char *filename);
void readgifdir(void);
int gcompare(void *gn1, void *gn2);
void prettyfile(char *buf, struct GDIR *gifdptr);
void screenon(void);
void screenoff(void);
void drawzgvlogo(int a, int b);
void cleartext(void);
void showerrmessage(int errnumber);
int delete_file(char *filename, int *num);
int move_file(char *filename, int *num);
int copy_file(char *filename);
void xv332_how_far(int sofar, int total);
void clear_xvpic(int xpos, int ypos);
int makexv332(char *filename, char *xvpicfn, unsigned int howfar);
int makedirxv332(char *filename, char *xvpicfn, unsigned int howfar);
int update_xvpics(int do_dirs_instead);
int fixvt(void);
void switchback(void);
void greyfix332(unsigned char **image, int w, int h, int xw7, int w8);
void wait_for_foreground(void);
void ctrlc(int foo);
void parentname(char *path, char*parent);
void *finddir(char *path);
void mctime(char *buf, time_t time);
struct DIRTREE *root;
void force_readgifdir(char *path);
int rmkdir(char *path);

int main(int argc,char *argv[])
{
int argsleft;

if(fixvt()==0)
  {
  fprintf(stderr,"Not running on console and no free VTs found.\n");
  exit(1);
  }

atexit(switchback);		/* this gets called after svgalib stuff */

/* this gets called after svgalib's ^C handler, on ^C */
signal(SIGINT,ctrlc);

vga_disabledriverreport();    /* quieten svgalib */
vga_init();
/* root permissions should now have been ditched */

root=malloc(sizeof(struct GDIR));
root->gifdir=NULL;
root->sib=NULL;
root->child=NULL;
strcpy(root->name,"/");
if((getenv("HOME"))&&(strlen(getenv("HOME"))<256))
   strcpy(homedir,getenv("HOME"));
 else homedir[0]='\0';


pixelsize=1;
getconfig();				/* see rcfile.c...   */
argsleft=parsecommandline(argc,argv);	/* ...for these two. */
left_just=0;
if(cfg.wide){BARWIDTH=240 ; left_just=1; } else BARWIDTH=138; 

copyfromconfig();

if(cfg.mtype != NO_MOUSE)
  has_mouse=ms_init(MACCEL, MBAUD, MDELTA, MDEV, MTOGGLE, MSAMPLE, cfg.mtype);

if(cfg.pcdres<1)cfg.pcdres=1;
if(cfg.pcdres>5)cfg.pcdres=5;

cleartext();

/* do one-file-only if have one arg. */
/* (does a chdir and carries on if it's a dir.) */
if(argsleft==1)
  load_one_file(argv[argc-1]);

/* do slideshow if more than one arg */
/* XXX this should be shoved into a separate routine sometime */
if(argsleft>1)
  {
  int i, entnum;
  struct stat buf;
  gifdir=malloc(sizeof(struct GDIR)*(argsleft+1));
  for(i=argc-argsleft,entnum=1;i<=argc-1;i++)
    if(stat(argv[i],&buf) != -1 && !S_ISDIR(buf.st_mode))
      {
      strncpy(gifdir[entnum].name,argv[i],254);
      gifdir[entnum].marked=1;
      gifdir[entnum].isdir=0;
      entnum++;
      /* avoid overflowing array */
      if(entnum>=MDIRSIZ-1) break;
      }
  gifdirsiz = entnum-1;
  one_file_only=1;
  VIS_MODE=G640x480x256;
  VIS_HI=480;
  VIS_WID=640;
  screenon(); 
  openstdin_nonblocking();
  load_tagged_files();
  screenoff();
  free(gifdir);
  exit(0);
  }

  VIS_MODE = cfg.vismode;
  switch(VIS_MODE){
   case G640x480x256:
      VIS_HI=480;
      VIS_WID=640;
      break;
   case G800x600x256:
      VIS_HI=600;
      VIS_WID=800;
      break;
   case G1024x768x256:
      VIS_HI=768;
      VIS_WID=1024;
      break;
   case G1280x1024x256:
      VIS_HI=1024;
      VIS_WID=1280;
      break;
   default:
      VIS_MODE=G640x480x256;
      VIS_HI=480;
      VIS_WID=640;
  };    

/* normal zgv startup */
cfg.errignore=0;	/* don't ignore errs if using this way */
screenon();
openstdin_nonblocking();
mainloop();
screenoff();

if(cfg.echotagged)
  {
  int f,first=1;
  
  for(f=1;f<=gifdirsiz;f++)
    if(gifdir[f].marked)
      {
      if(first) first=0; else putchar(32);
      printf(gifdir[f].name);
      }
  
  putchar('\n');
  }

exit(0);
}


void openstdin_nonblocking()
{
zgv_ttyfd=fileno(stdin);
fcntl(zgv_ttyfd,F_SETFL,O_NONBLOCK);
}


void load_one_file(char *filename)
{
hffunc hf;
struct stat sbuf;

/* before we load it, see if it's a directory. If so, we run zgv with
 * much the same result as typing '( cd whatever;zgv )'.
 */

if(stat(filename,&sbuf)!=-1 && S_ISDIR(sbuf.st_mode))
  {
  chdir(filename);
  return;
  }
  
openstdin_nonblocking();	/* we hadn't done this yet */
one_file_only=1;
VIS_MODE=G640x480x256;
VIS_HI=480;			/* necessary in order to have */
VIS_WID=640;          		/* a progress report          */
if(cfg.onefile_progress)
  {
  screenon();
  inithowfar();
  hf=showhowfar;
  }
else 
  {
  hf=NULL;
  fprintf(stderr,"Loading...");
  }

if(cfg.onefile_progress && hf!=NULL) vga_runinbackground(1);  

if(readpicture(filename,hf,1,0)!=_PIC_OK)
  {
  if(cfg.onefile_progress) screenoff();
  fprintf(stderr,"\rError loading file.\n");
  exit(1);
  }
else
  {
  if(hf==NULL)
    fprintf(stderr,"\r          \r");
  }
      
wait_for_foreground();
screenoff();
exit(0);
}


void copyfromconfig()
{
curvgamode=cfg.videomode;
zoom=cfg.zoom;
vkludge=cfg.vkludge;
brightness=cfg.brightness;
contrast=cfg.contrast;
if((curvgamode==G320x400x256)||(curvgamode==G360x480x256))
  virtual=1;
else
  virtual=0;
}

void mainloop()
{
int quit,key,curent; /* quit, key pressed, current entry */
int oldent,startfrom,oldstart,f,markchange, resort;
extern int x,y,mx,my;	/* ... don't blame me, I didn't write that ;-) -rjm */
int left_button=0, right_button=0, butstatep=0,px,py;
struct ms_event ev;

/* black out past-positions array */
for(f=0;f<MPPOSSIZ;f++)
  pastpos[f].dev=-1,pastpos[f].inode=-1;

quit=0; curent=1; oldent=1;
startfrom=1;

readgifdir();
showgifdir(startfrom,0,1);
showbar(curent,1,startfrom);

mx=my=1000;
px=x=mx/2;
py=y=my/2;
while(!quit)
  {
  oldent=curent;
  markchange=0;
  usleep(10000);
  key=readnbkey(zgv_ttyfd);
  
  /* grok mouse movements and buttons */
  if(has_mouse)
    {
    if(mx != 1000 || !x || !y || x==mx || y==my)
      {
      mx=my=1000;
      px=x=mx/2;
      py=y=my/2;
      }
      
    if(get_ms_event(&ev)>0)
      {
      if (y < py-20)
        {
        if(curent>1) curent--;
        py = y;
        }
      else if (y > py+20)
        {
        if(curent<gifdirsiz) curent++;
        py = y;
        }
          
      if (x < px-20)
        {
        curent-=YSIZ;
        if(curent<1) curent=1;
        px = x;
        }
      else if (x > px+20)
        {
        curent+=YSIZ;
        if(curent>gifdirsiz) curent=gifdirsiz;
        px = x;
        }
          
      if(ev.ev_butstate != butstatep)
        {
        butstatep = ev.ev_butstate;
        if(left_button && !(ev.ev_butstate & 4))
          key = RK_ENTER;
        if(right_button && !(ev.ev_butstate & 1))
          key = ' ';
        left_button = ev.ev_butstate & 4;
        right_button = ev.ev_butstate & 1;
        }
      }
    }	/* end of 'if(has_mouse)' */
   
  resort=0;   
  switch(key)
    {
    case -79: case -78: case -77: case -76: case -75: /* Alt1 - Alt5 */
      cfg.pcdres=key+80;
      break;
    case 'S': resort=cfg.sort_key!=5; cfg.sort_key=5; break;
    case 's': resort=cfg.sort_key!=4; cfg.sort_key=4; break;
    case 'a': resort=cfg.sort_key!=8; cfg.sort_key=8; break;
    case 'A': resort=cfg.sort_key!=9; cfg.sort_key=9; break;
    case 'O': resort=cfg.sort_key!=3; cfg.sort_key=3; break;
    case 'o': resort=cfg.sort_key!=2; cfg.sort_key=2; break;
    case '1': if(cfg.mode_allowed[G640x480x256]){
      VIS_MODE=G640x480x256;
      VIS_WID=640;
      VIS_HI=480;
      redrawall(curent,startfrom);};
      break;
    case '2':if(cfg.mode_allowed[G800x600x256]){
      VIS_MODE=G800x600x256;
      VIS_WID=800;
      VIS_HI=600;
      redrawall(curent,startfrom);};
      break;
    case '3':if(cfg.mode_allowed[G1024x768x256]){
      VIS_MODE=G1024x768x256;
      VIS_WID=1024;
      VIS_HI=768;
      redrawall(curent,startfrom);};
      break;
    case '4':if(cfg.mode_allowed[G1152x864x256]){
      VIS_MODE=G1152x864x256;
      VIS_WID=1152;
      VIS_HI=864;
      redrawall(curent,startfrom);};
      break;
    case '5':if(cfg.mode_allowed[G1280x1024x256]){
      VIS_MODE=G1280x1024x256;
      VIS_WID=1280;
      VIS_HI=1024;
      redrawall(curent,startfrom);};
      break;
    case '6':if(cfg.mode_allowed[G1600x1200x256]){
      VIS_MODE=G1600x1200x256;
      VIS_WID=1600;
      VIS_HI=1200;
      redrawall(curent,startfrom);};
      break;
    case 'z':
      cfg.fullsel=!cfg.fullsel;
      redrawall(curent,startfrom);
      break;
    case 'u':
      /* ignore it in an xvpics dir */
      {
      char cdir[MAXPATHLEN+1];
      
      getcwd(cdir,MAXPATHLEN);
      if(strstr(cdir,"/.xvpics")==NULL && cfg.xvpic_index)
        {
        updating_index=1;
        update_xvpics(0);
        updating_index=0;
        redrawall(curent,startfrom);
        }
      }
      break;
    case 'r':
      /* ignore it in an xvpics dir */
      {
      char cdir[MAXPATHLEN+1];
      
      getcwd(cdir,MAXPATHLEN);
      if(strstr(cdir,"/.xvpics")==NULL && cfg.xvpic_index)
        {
        updating_index=1;
        update_xvpics(2);
        updating_index=0;
        redrawall(curent,startfrom);
        }
      }
      break;
    case 'U':
      if(cfg.xvpic_index)
        {
        updating_index=1;
        update_xvpics(1);
        updating_index=0;
        redrawall(curent,startfrom);
        }
      break;
    case 'v':
      cfg.xvpic_index=!cfg.xvpic_index;
      redrawall(curent,startfrom);
      break;
      break;
    case 'V':
      if(BARWIDTH==138){BARWIDTH=240;} else BARWIDTH=138;
      left_just=!left_just;
      redrawall(curent,startfrom);
      break;
    case 'b':
      if(v256) break;
      cfg.fs16col=!cfg.fs16col;
      redrawall(curent,startfrom);
      break;
    case '?':
      showhelp(zgv_ttyfd,"- KEYS FOR FILE SELECT SCREEN -",zgvhelp,1);
      redrawall(curent,startfrom);
      break;
    case '/':
      showhelp(zgv_ttyfd,"- KEYS FOR FILE SELECT SCREEN (2)-",zgvhelp2,1);
      redrawall(curent,startfrom);
      break;
    case 18: case 12:{   /* ^R and ^L */
      char cdir[MAXPATHLEN+1];
      getcwd(cdir,MAXPATHLEN);
      force_readgifdir(cdir);
      if(curent>gifdirsiz) curent=gifdirsiz;
      oldent=curent;
      redrawall(curent,startfrom);
      break;
      };
    case 'e':{ /* rename current image */
           char tmp[256];
           inputbox(zgv_ttyfd,"Enter new name:",0,idx_light,idx_dark,idx_black,255,tmp);
           if((strchr(tmp,'/')==NULL)&&(strlen(tmp)>1)){
              rename(gifdir[curent].name,tmp);
              strcpy(gifdir[curent].name,tmp);
              qsort(&(gifdir[1]),gifdirsiz,sizeof(struct GDIR),(void *)gcompare);
              redrawall(curent,startfrom);
              }
           else msgbox(zgv_ttyfd,"Illegal name.",MSGBOXTYPE_OK, idx_light,idx_dark,idx_black);
       };
       break;
      
    case RK_HOME: case 'A'-0x40: /* home or ^a */
      curent=1;
      break;
    case RK_END:  case 'E'-0x40: /* end  or ^e */
      curent=gifdirsiz;
      break;
    case RK_PAGE_UP: case 'U'-0x40: /* pgup or ^u */
      curent-=YSIZ*XSIZ;
      if(curent<1) curent=1;
      break;
    case RK_PAGE_DOWN: case 'V'-0x40: /* pgdn or ^v */
      curent+=YSIZ*XSIZ;
      if(curent>gifdirsiz) curent=gifdirsiz;
      break;
    case 'd': case RK_DELETE:   /* d or the 'del' key */
      if(gifdir[curent].isdir) break;
      if(delete_file(gifdir[curent].name,&curent))
        {
        if(curent<gifdirsiz)curent++;
        oldent=curent;
        }
      redrawall(curent,startfrom);
      break;
    case 'D':   /* D key - delete tagged files */
      {int tmp_delprompt,retn=0;
        
      tmp_delprompt=cfg.nodelprompt;
      if(cfg.promptmulti){
        retn=msgbox(zgv_ttyfd,"Delete all tagged files?",MSGBOXTYPE_YESNO, idx_light,idx_dark,idx_black);  
        if(retn==1)cfg.nodelprompt=1;
      }
      if((!cfg.promptmulti)|(retn==1))
      {
      	for(f=1;f<=gifdirsiz;f++)if((!gifdir[f].isdir)&gifdir[f].marked)
                        	  delete_file(gifdir[f].name,&f);
        if(curent>gifdirsiz) curent=gifdirsiz;
        oldent=curent;
        cfg.nodelprompt=tmp_delprompt;
      };
      redrawall(curent,startfrom);
      break;
      }
    case 'c':
      if(targetdir!=NULL)
      {
        copy_file(gifdir[curent].name);
      };
      force_readgifdir(targetdir);
      break;
    case 'C':if(targetdir!=NULL)   
      {int tmp_delprompt,retn=0;
        
      tmp_delprompt=cfg.nodelprompt;
      if(cfg.promptmulti){
        retn=msgbox(zgv_ttyfd,"Copy all tagged files?",MSGBOXTYPE_YESNO, idx_light,idx_dark,idx_black);  
        if(retn==1)cfg.nodelprompt=1;
      }
      if((!cfg.promptmulti)|(retn==1))
      {
      	for(f=1;f<=gifdirsiz;f++)if((!gifdir[f].isdir)&gifdir[f].marked)
            if(copy_file(gifdir[f].name)==-2){cfg.nodelprompt=tmp_delprompt;break;};
        cfg.nodelprompt=tmp_delprompt;
      };
      force_readgifdir(targetdir);
      };
      break;
    case 'm':
      if(targetdir!=NULL)
      {
        move_file(gifdir[curent].name,&curent);
      };
      force_readgifdir(targetdir);
      redrawall(curent,startfrom);
      break;
    case 't':	/* tag current file and move on */
      if(gifdir[curent].isdir==0)
        {
        gifdir[curent].marked=1;
        markchange=1;
        }
      if(curent<gifdirsiz) curent++;
      break;
    case 'M':   
      {int tmp_delprompt,retn=0;
        
      tmp_delprompt=cfg.nodelprompt;
      if(cfg.promptmulti){
        retn=msgbox(zgv_ttyfd,"Move all tagged files?",MSGBOXTYPE_YESNO, idx_light,idx_dark,idx_black);  
        if(retn==1)cfg.nodelprompt=1;
      }
      if((!cfg.promptmulti)|(retn==1))
      {
      	for(f=1;f<=gifdirsiz;f++)if((!gifdir[f].isdir)&gifdir[f].marked)
                        	move_file(gifdir[f].name,&f);
        if(curent>gifdirsiz) curent=gifdirsiz;
        oldent=curent;
        cfg.nodelprompt=tmp_delprompt;
      };
      force_readgifdir(targetdir);
      redrawall(curent,startfrom);
      break;
      }
    case 'n':	/* untag current file and move on */
      if(gifdir[curent].isdir==0)
        {
        gifdir[curent].marked=0;
        markchange=1;
        }
      if(curent<gifdirsiz) curent++;
      break;
    case 'p':
      if(gifdir[curent].isdir){
         getcwd(targetdir,256);
         if(!strcmp(gifdir[curent].name,"..")){
            int i;
            for(i=strlen(targetdir);(i>0)&&(targetdir[i]!='/');targetdir[i--]='\0');
            targetdir[i]='\0';
         } else {
            strcat(targetdir,"/");
            strcat(targetdir,gifdir[curent].name);
         };
      showtargetdir();
      };
      break;
    case 'P':{
           char tmp[512], tmp2[512], tmp3[512];
           struct stat dir_s; 

           inputbox(zgv_ttyfd,"Enter target directory:",0,idx_light,idx_dark,idx_black,255,tmp);
           if(tmp[0]!='/'){getcwd(tmp2,256);sprintf(tmp3,"%s/%s",tmp2,tmp);strcpy(tmp,tmp3); };
           if((stat(tmp,&dir_s))&&(errno==ENOENT)){
              sprintf(tmp2,"%s does not exist. Create?",tmp);
              if(msgbox(zgv_ttyfd,tmp2,MSGBOXTYPE_YESNO,idx_light,idx_dark,idx_black)){
                 if(rmkdir(tmp)){
                    sprintf(tmp2,"Cannot create %s",tmp);
                    msgbox(zgv_ttyfd,tmp2,MSGBOXTYPE_OK,idx_light,idx_dark,idx_black);
                    };                
                 };
              } else
           if(S_ISDIR(dir_s.st_mode)){strcpy(targetdir,tmp);} else
              { sprintf(tmp2,"%s is not a directory.",tmp);
                msgbox(zgv_ttyfd,tmp2,MSGBOXTYPE_OK,idx_light,idx_dark,idx_black);
              };
           if((strlen(targetdir)>1)&&(targetdir[strlen(targetdir)-1]!='/'))
              strcat(targetdir,"/");
           showtargetdir();  
       };
       break;
    case ' ':   /* toggle tag/untag flag of current file and move on */
      if(gifdir[curent].isdir==0)
        {
        gifdir[curent].marked=1-gifdir[curent].marked;
        markchange=1;
        }
      if(curent<gifdirsiz) curent++;
      break;
    case 'i':   /* toggle ahow all files */
      cfg.allfiles=!cfg.allfiles;
      if(cfg.allfiles) 
         msgbox(zgv_ttyfd,"Showing all files",
                MSGBOXTYPE_OK,idx_light,idx_dark,idx_black);
      else
         msgbox(zgv_ttyfd,"Showing only image files",
                MSGBOXTYPE_OK,idx_light,idx_dark,idx_black);
      {
      char cdir[MAXPATHLEN+1];
      getcwd(cdir,MAXPATHLEN);
      force_readgifdir(cdir);
      curent=1; startfrom=1;
      redrawall(curent,startfrom);
      };
      break;
    case 'j':	/* toggle prompt multiple files */
      cfg.promptmulti=!cfg.promptmulti;
      if(cfg.promptmulti) 
         msgbox(zgv_ttyfd,"Prompting for multiple files",
                MSGBOXTYPE_OK,idx_light,idx_dark,idx_black);
      else
         msgbox(zgv_ttyfd,"Not prompting for multiple files",
                MSGBOXTYPE_OK,idx_light,idx_dark,idx_black);
      break;
    case 'g':{
           char tmp[512], tmp2[512], tmp3[512];
           struct stat dir_s; 

           inputbox(zgv_ttyfd,"Change to directory?",0,idx_light,idx_dark,idx_black,256,tmp);
           if(tmp[0]!='/'){getcwd(tmp2,256);sprintf(tmp3,"%s/%s",tmp2,tmp);strcpy(tmp,tmp3); };
           if((stat(tmp,&dir_s))&&(errno==ENOENT)){
              sprintf(tmp2,"%s does not exist",tmp);
              msgbox(zgv_ttyfd,tmp2,MSGBOXTYPE_OK,idx_light,idx_dark,idx_black);
              break;
              } else 
           if(S_ISDIR(dir_s.st_mode))chdir(tmp); else
              { sprintf(tmp2,"%s is not a directory.",tmp);
                msgbox(zgv_ttyfd,tmp2,MSGBOXTYPE_OK,idx_light,idx_dark,idx_black);
                break;
              };
           readgifdir();
           get_pastpos(&curent,&startfrom);
           redrawall(curent,startfrom);
       };
       break;
    case 'k':	/* per file prompting */
      cfg.nodelprompt=!cfg.nodelprompt;
      if(cfg.nodelprompt) 
         msgbox(zgv_ttyfd,"Prompting for each file",
                MSGBOXTYPE_OK,idx_light,idx_dark,idx_black);
      else
         msgbox(zgv_ttyfd,"Not prompting for each file",
                MSGBOXTYPE_OK,idx_light,idx_dark,idx_black);
      break;
    case 'l':	/* toggle loop or not in slideshow */
      cfg.loop=!cfg.loop;
      if(cfg.loop) 
         msgbox(zgv_ttyfd,"Loop in slideshow",
                MSGBOXTYPE_OK,idx_light,idx_dark,idx_black);
      else
         msgbox(zgv_ttyfd,"No looping in slideshow",
                MSGBOXTYPE_OK,idx_light,idx_dark,idx_black);
      break;
    case 'y':	
      cfg.hicontrol=!cfg.hicontrol;
      break;
    case 'T':	/* tag all files */
      for(f=1;f<=gifdirsiz;f++) if(gifdir[f].isdir==0) gifdir[f].marked=1;
      showgifdir(startfrom,0,0);
      break;
    case 'N':	/* untag all files */
      for(f=1;f<=gifdirsiz;f++) if(gifdir[f].isdir==0) gifdir[f].marked=0;
      showgifdir(startfrom,0,0);
      break;
    case RK_CURSOR_UP:
      if(curent>1) curent--; break;
    case RK_CURSOR_DOWN:
      if(curent<gifdirsiz) curent++; break;
    case RK_CURSOR_LEFT:
      curent-=YSIZ;
      if(curent<1) curent=1;
      break;
    case RK_CURSOR_RIGHT:
      curent+=YSIZ;
      if(curent>gifdirsiz) curent=gifdirsiz;
      break;
    case RK_ENTER:
      /* uhhhhh my head hurts */
      sgres = 0;
      do
        {
        if(permissiondenied(gifdir[curent].name))
          {
          msgbox(zgv_ttyfd,"Permission denied or file not found",
          	MSGBOXTYPE_OK,idx_light,idx_dark,idx_black);
          pic_incr=0;	/* stop here *even if ^p or ^n used* */
          break;
          }
          
        if(gifdir[curent].isdir)
          {
          new_pastpos(curent,startfrom);
          showbar(curent,0,startfrom);
          showgifdir(startfrom,1,1);
          chdir(gifdir[curent].name);
          readgifdir();
          get_pastpos(&curent,&startfrom);
          /* if these are off the end, it's clearly out of date (dir
           * must have changed) so might as well just start at top.
           */
          if(curent>gifdirsiz || startfrom>gifdirsiz) curent=startfrom=1;
          oldent=curent;
          showgifdir(startfrom,0,1);
          showbar(curent,1,startfrom);
          break;
          }
          
        tagview_mode=0;
	if(sgres>=2)
          {
	  if(load_single_file(curent,0)==0) pic_incr=0;
	  }
        else
          if(load_single_file(curent,1)==0) pic_incr=0;

	if(sgres==3) gifdir[curent].marked=1;
        if(pic_incr!=999)curent+=pic_incr; /* Ugly, I know */
        if(curent>gifdirsiz) { curent--; pic_incr=0; }
        if(curent<1 || gifdir[curent].isdir) { curent++; pic_incr=0; }
        oldstart=startfrom;
        while(curent<startfrom)
          startfrom-=YSIZ;
        while(fwinxpos(curent-startfrom+1)+BARWIDTH>COLS)
          startfrom+=YSIZ;
        if(startfrom<1) startfrom=1;
        if(sgres<2 || pic_incr==0)
          {
          oldent=-1;
          redrawall(curent,startfrom);
          }
        }
      while(pic_incr);
      break;
      
    case '\t':		/* tab */
      load_tagged_files();
      redrawall(curent,startfrom);
      break;
    
    case ';':		/* file details */
    { char tmp[2048], ttmp[256];
      
      getcwd(tmp,MAXPATHLEN);
      strcat(tmp,"/");
      strcat(tmp,gifdir[curent].name);
      sprintf(ttmp," %9li, ",gifdir[curent].size);
      strcat(tmp,ttmp);
      if(gifdir[curent].width!=-1){
         sprintf(ttmp,"%ix%i, ",gifdir[curent].width,gifdir[curent].height);
         strcat(tmp,ttmp);
      };
      mctime(ttmp,gifdir[curent].ctime);
      strcat(tmp,ttmp);
      msgbox(zgv_ttyfd,tmp,MSGBOXTYPE_OK,
         	idx_light,idx_dark,idx_black);
      break;
      }    
    case ':':		/* full file name */
      msgbox(zgv_ttyfd,gifdir[curent].name,MSGBOXTYPE_OK,
         	idx_light,idx_dark,idx_black);
      break;
    case 'x': case RK_ESC:
      quit=1;
    }


  if(resort){
    qsort(&(gifdir[1]),gifdirsiz,sizeof(struct GDIR),(void *)gcompare);
    redrawall(curent,startfrom);
    };
    
  oldstart=startfrom;
  while(curent<startfrom)
    startfrom-=YSIZ;
  while(fwinxpos(curent-startfrom+1)+BARWIDTH>COLS)
    startfrom+=YSIZ;
  if(startfrom<1) startfrom=1;
  if(startfrom!=oldstart)
    {
    showbar(oldent,0,oldstart);
    showgifdir(oldstart,1,0);
    showgifdir(startfrom,0,0);
    showbar(curent,1,startfrom);
    }
  else  
    if(oldent!=-1 && (curent!=oldent || markchange))
      {
      showbar(oldent,0,startfrom);
      showbar(curent,1,startfrom);
      }
  }
}


/* add new pastpos[0], shifting down all the rest of the entries. */
void new_pastpos(int curent,int startfrom)
{
struct stat sbuf;
int f;

if(cfg.forgetoldpos) return;

for(f=MPPOSSIZ-1;f>0;f--)
  {
  pastpos[f].dev      =pastpos[f-1].dev;
  pastpos[f].inode    =pastpos[f-1].inode;
  pastpos[f].curent   =pastpos[f-1].curent;
  pastpos[f].startfrom=pastpos[f-1].startfrom;
  }

if(stat(".",&sbuf)==-1) return;

pastpos[0].dev      =sbuf.st_dev;
pastpos[0].inode    =sbuf.st_ino;
pastpos[0].curent   =curent;
pastpos[0].startfrom=startfrom;
}


/* return curent from pastpos[] entry matching current directory,
 * or if none match return 1.
 */
void get_pastpos(int *curentp,int *startfromp)
{
struct stat sbuf;
int f;

*curentp=*startfromp=1;

if(cfg.forgetoldpos || stat(".",&sbuf)==-1) return;

for(f=0;f<MPPOSSIZ;f++)
  if(pastpos[f].inode==sbuf.st_ino && pastpos[f].dev==sbuf.st_dev)
    {
    *curentp=pastpos[f].curent;
    *startfromp=pastpos[f].startfrom;
    return;
    }
}


int load_single_file(int curent,int do_howfar)
{
static char buf[1024];

if(cfg.cmd[0])
  {
  /* alternative command */
  /* avoid using sprintf() just in case of stray %'s etc. */
  char *ptr,sav;
  ptr=strstr(cfg.cmd,"%s");	/* we know it's there - see rcfile.c */
  sav=*ptr; *ptr=0;
  strcpy(buf,cfg.cmd);
  strcat(buf,gifdir[curent].name);
  strcat(buf,ptr+2);
  *ptr=sav;
  screenoff();
  system(buf);
  }
else
  {
  int tmp;
  
  if(do_howfar)
    inithowfar();
    
  vga_runinbackground(1);

  /* save context for possible abort */
  if(setjmp(setjmpbuf))
    {
    /* if we get here, someone aborted loading a file. */
    if(!tagview_mode || (tagview_mode && do_howfar))
      {
      wait_for_foreground();
      msgbox(zgv_ttyfd,"File view aborted",MSGBOXTYPE_OK,
         	idx_light,idx_dark,idx_black);
      }
    return(0);
    }
  else
    {
      if((tmp=readpicture(gifdir[curent].name,
    		do_howfar?showhowfar:smallhowfar,1, 0))!=_PIC_OK)
      {
      wait_for_foreground();
      if(tagview_mode) screenon();
      showerrmessage(tmp);
      return(0);
      }
    }
  }

return(1);
}


void load_tagged_files()
{
int f,t;
int dohf;

for(f=1,t=0;f<=gifdirsiz;f++)
  if(gifdir[f].marked) t++;

if(t==0)
  {
  msgbox(zgv_ttyfd,"No files tagged",MSGBOXTYPE_OK,
  	idx_light,idx_dark,idx_black);
  return;
  }

/* vgadisp.c sets tagview_mode==0 if esc is pressed */
tagview_mode=1;
dohf=1;
do {
 for(f=1;f<=gifdirsiz && tagview_mode;f++)
  {
  if(gifdir[f].marked==0) continue;
  if(permissiondenied(gifdir[f].name))
    {
    if(f>1) screenon();
    msgbox(zgv_ttyfd,"Permission denied or file not found",
    	MSGBOXTYPE_OK,idx_light,idx_dark,idx_black);
    return;
    }
  if(load_single_file(f,dohf)==0) break;
  dohf=0;
  }
 } while (cfg.loop && sgres != 1);
}


/* this could also be file not found of course */
int permissiondenied(char *fname)
{
FILE *junk;
if((junk=fopen(fname,"rb"))==NULL)
  return(1);
fclose(junk);
return(0);
}


void redrawall(int curent,int startfrom)
{
screenon();
showgifdir(startfrom,0,1);
showbar(curent,1,startfrom);
}


void inithowfar()
{
int f;

vga_setcolor(idx_medium);
for(f=(VIS_HI>>1)-20;f<=(VIS_HI>>1)+20;f++)
  vga_drawline((VIS_WID>>1)-220,f,(VIS_WID>>1)+220,f);
draw3dbox((VIS_WID>>1)-220,(VIS_HI>>1)-20,(VIS_WID>>1)+220,(VIS_HI>>1)+20,2,1, idx_light,idx_dark);
draw3dbox((VIS_WID>>1)-211,(VIS_HI>>1)-11,(VIS_WID>>1)+211,(VIS_HI>>1)+11,1,0, idx_light,idx_dark);

howfar_upto=0;
drawtext3d((VIS_WID>>1)-65-10*cfg.linetext,(VIS_HI>>1)-5,2,"Decompressing - please wait",0,
					idx_light,idx_dark,idx_black);
}


void showhowfar(int sofar,int total)
{
int f,d;

/* test for abort */
smallhowfar(sofar,total);

if(((sofar%10)==0)||(sofar==total))
  {
  d=(420*sofar)/total;
  if(d>howfar_upto)
    {
    vga_lockvc();
    if(!vga_oktowrite())
      {
      vga_unlockvc();
      return;
      }
    vga_setcolor(idx_light); /* we set this always in case of a VC switch */
    for(f=howfar_upto;f<=d;f++)
      vga_drawline((VIS_WID>>1)-210+f,(VIS_HI>>1)-10,(VIS_WID>>1)-210+f,(VIS_HI>>1)+10);
    vga_unlockvc();
    howfar_upto=f;
    }
  }
}


/* minimal 'how far' function that just tests for an abort */
void smallhowfar(int sofar,int total)
{
/* we jump back to an abort message if Esc was pressed */
if(!one_file_only && !cfg.repeat_timer)
  if(readnbkey(zgv_ttyfd)==RK_ESC)
    {
    /* these routines blast the malloc'ed stuff, which has *for sure*
     * been allocated by now, because we must already be reading the file
     * in for us to get here!
     */
    aborted_file_cleanup();
    longjmp(setjmpbuf,1);
    }
}


void showbar(int entnum,int selected,int startfrom)
{
char ctmp[100];
int xpos,ypos;
int xt,xst;

xpos=fwinxpos(entnum-startfrom+1);
if((xpos<1)||(xpos+BARWIDTH>COLS)) return;
ypos=fwinypos(entnum-startfrom+1);
prettyfile(ctmp,&(gifdir[entnum]));

set_max_text_width(BAR_RESTRICT_WIDTH);
xt=(cfg.xvpic_index&&!left_just)?centreseltxt(xpos,gdfsiz,ctmp):xpos+10;
xst=(!left_just)?xpos+(BARWIDTH-gifdir[entnum].xvw)/2:xpos+10;
if(cfg.blockcursor)
  {
  /* block-style cursor - hopefully easier to read/see. */
  if(selected)
    draw3dbox(xpos-2,ypos-2,xpos+BARWIDTH+1,ypos+barheight+1,4,1,
    	idx_dark,idx_dark);
  else
    undraw3dbox(xpos-2,ypos-2,xpos+BARWIDTH+1,ypos+barheight+1,4);
  
  /* in case the last file was just marked/unmarked */
  vga_setcolor(gifdir[entnum].marked?idx_marked:idx_black);
  vgadrawtext(xt,ypos+3+gdfofs,gdfsiz,ctmp);
  }
else
  {
  if(selected)
    {
    draw3dbox(xpos,ypos,xpos+BARWIDTH-1,ypos+barheight-1,1,1,
    	idx_light,idx_dark);
    drawtext3d(xt,ypos+3+gdfofs,gdfsiz,ctmp,0, idx_light,idx_dark,
    		gifdir[entnum].marked?idx_marked:idx_black);
    /* box if cfg.xvpic_index and is an xvpic being used */
    if(cfg.xvpic_index && gifdir[entnum].xvw!=0)
      draw3dbox(xst-2,
                ypos+GDFYBIT+39-gifdir[entnum].xvh/2-2,
                xst+gifdir[entnum].xvw+1,
                ypos+GDFYBIT+39-gifdir[entnum].xvh/2+gifdir[entnum].xvh+1,
                1,0, idx_light,idx_dark);
    }
  else
    {
    undraw3dbox(xpos,ypos,xpos+BARWIDTH-1,ypos+barheight-1,1);
    undrawtext3d(xt,ypos+3+gdfofs,gdfsiz,ctmp);
    vga_setcolor(gifdir[entnum].marked?idx_marked:idx_black);
    vgadrawtext(xt,ypos+3+gdfofs,gdfsiz,ctmp);
    /* undraw box if cfg.xvpic_index and is an xvpic being used */
    if(cfg.xvpic_index && gifdir[entnum].xvw!=0)
      undraw3dbox(xst-2,
                  ypos+GDFYBIT+39-gifdir[entnum].xvh/2-2,
                  xst+gifdir[entnum].xvw+1,
                  ypos+GDFYBIT+39-gifdir[entnum].xvh/2+gifdir[entnum].xvh+1,
                  1);
    }
  }
set_max_text_width(NO_CLIP_FONT);
}

void showtargetdir()
{ char ctmp[256];
  char buf[562];
  
  int i;
  if(targetdir==NULL)return;
  memset(buf,idx_medium,562);
  for(i=yofs+43;i<yofs+61;i++)vga_drawscansegment(buf,39,i,562);
  set_max_text_width(560);
  if(targetdir[0]!='\0'){
    sprintf(ctmp,"Target directory: %s",targetdir);
    drawtext3d(40,yofs+43,4,ctmp,1, idx_light,idx_dark,idx_black);
    };
}


int centreseltxt(int x,int fsiz,char *str)
{
int a;

a=vgatextsize(fsiz,str);
return(x+(BARWIDTH-a)/2);
}

void mctime(char *buf, time_t time)
{ struct tm *bt;
  bt=localtime(&time);
  sprintf(buf,"%.2d:%.2d %i-%.2d-%i",bt->tm_hour,bt->tm_min,bt->tm_mday,bt->tm_mon,bt->tm_year);
};


void showgifdir(int startfrom,int unshow,int drawdirmsg)
{
char cdir[MAXPATHLEN+1],*ptr;
static char ctmp[1024],ctmp2[1024];
int f,ypos,xpos,w,h,y,xt;
unsigned char *image;
int xvpics_dir_exists=0;

getcwd(cdir,MAXPATHLEN);

if(drawdirmsg)
  {
  if(updating_index)
    sprintf(ctmp,"Updating index of %s",cdir);
  else
    sprintf(ctmp,"Directory of %s",cdir);
  
  set_max_text_width(560);
  if(unshow)
    undrawtext3d(40,yofs+25,4,ctmp);
  else
    drawtext3d(40,yofs+25,4,ctmp,1, idx_light,idx_dark,idx_black);
  if(targetdir[0]!='\0'){
    sprintf(ctmp,"Target directory: %s",targetdir);
    if(unshow)
      undrawtext3d(40,yofs+43,4,ctmp);
    else
      drawtext3d(40,yofs+43,4,ctmp,1, idx_light,idx_dark,idx_black);
    };
  }

/* see if either xvpics dir exists. if not, we can speed things
 * up a little by not bothering to look for thumbnails at all (the
 * difference is usually small, but on a dos partition it's sometimes
 * *really* painful otherwise).
 *
 * There's a little bit of duplicated code though, so it's a bit
 * nasty and should be cleaned up sometime (XXX).
 */
if(strstr(cdir,"/.xvpics")!=NULL)
  xvpics_dir_exists=1;
else
  {
  struct stat sbuf;
  
  /* must be either a dir or symlink (symlink could be to a dir...) */
  if(stat(".xvpics",&sbuf)!=-1 && (S_ISDIR(sbuf.st_mode) ||
                                   S_ISLNK(sbuf.st_mode)))
    xvpics_dir_exists=1;
  else
    {
    sprintf(ctmp2,"%s/.xvpics/",homedir);
    ptr=ctmp2+strlen(ctmp2);
    getcwd(ptr,sizeof(ctmp2)-strlen(ctmp2)-1);
    /* convert /foo/bar/baz to _foo_bar_baz */
    while((ptr=strchr(ptr,'/'))!=NULL) *ptr++='_';
    if(stat(ctmp2,&sbuf)!=-1 && (S_ISDIR(sbuf.st_mode) ||
                                 S_ISLNK(sbuf.st_mode)))
      xvpics_dir_exists=1;
    }
  }

set_max_text_width(BAR_RESTRICT_WIDTH);
for(f=startfrom;f<=gifdirsiz;f++)
  {
  xpos=fwinxpos(f-startfrom+1);
  if(xpos+BARWIDTH>COLS) break;
  ypos=fwinypos(f-startfrom+1);
  prettyfile(ctmp,&(gifdir[f]));
  xt=(cfg.xvpic_index&&!left_just)?centreseltxt(xpos,gdfsiz,ctmp):xpos+10;
  vga_setcolor(unshow?idx_medium:(gifdir[f].marked?idx_marked:idx_black));
  vgadrawtext(xt,ypos+3+gdfofs,gdfsiz,ctmp);
  if(cfg.thicktext && cfg.blockcursor)
    vgadrawtext(xt+1,ypos+3+gdfofs,gdfsiz,ctmp);
  vga_setcolor(unshow?idx_medium:idx_black);
  if(left_just&cfg.xvpic_index){
     int i,j;
     
     sprintf(ctmp,"%li",gifdir[f].size);
     j=(strlen(ctmp)-1)%3+1;
     strncpy(ctmp2,ctmp,j);
     for(i=j;i<strlen(ctmp);i+=3){
        ctmp2[j++]=',';
        memcpy(ctmp2+j,ctmp+i,3);
        j+=3;
     };
     ctmp2[j]='\0';
     if((gifdir[f].width>0)&&!gifdir[f].isdir){
        sprintf(ctmp,", %ix%i",gifdir[f].width,gifdir[f].height);
        strcat(ctmp2,ctmp);   
     };
/*     set_max_text_width(BARWIDTH-92); */
     vgadrawtext(xpos+92,ypos+28+gdfofs,gdfsiz,ctmp2);
     mctime(ctmp,gifdir[f].ctime);
     ctmp[strlen(ctmp)]='\0';
     vgadrawtext(xpos+92,ypos+53+gdfofs,gdfsiz,ctmp);
  };
  if(cfg.xvpic_index)
    {
    /* load and draw thumbnail file (or undraw it) */
    if(unshow)
      {
      image=malloc(96);
      if(image!=NULL)
        {
          xt=xpos+(BARWIDTH-80)/2-2;
          if(left_just)xt=xpos+10-2;     
        memset(image,idx_medium,96);
        for(y=-2;y<62;y++)
          vga_drawscansegment(image,
          	xt,ypos+y+GDFYBIT+9,96);
        free(image);
        }
      }
    else
      {
      if(xvpics_dir_exists)
        {
        /* if '/.xvpics' is in the (absolute) path somewhere, we're
         * in a xvpics subdir. So look for the thumbnail here! :-)
         */
        if(strstr(cdir,"/.xvpics")!=NULL)
          strcpy(ctmp,gifdir[f].name);
        else
          sprintf(ctmp ,".xvpics/%s",gifdir[f].name);
        sprintf(ctmp2,"%s/.xvpics/",homedir);
        ptr=ctmp2+strlen(ctmp2);
        getcwd(ptr,sizeof(ctmp2)-strlen(ctmp2)-1);
        /* convert /foo/bar/baz to _foo_bar_baz */
        while((ptr=strchr(ptr,'/'))!=NULL) *ptr++='_';
        strcat(ctmp2,"/"); strcat(ctmp2,gifdir[f].name);
        }
      
      gifdir[f].xvw=gifdir[f].xvh=0;
      
      if(xvpics_dir_exists && (read_xv332(ctmp,&image,&w,&h)==_PIC_OK ||
         read_xv332(ctmp2,&image,&w,&h)==_PIC_OK))
        {
        int xwant=xpos+(BARWIDTH-w)/2,w8=0;
   
      	if(left_just)xwant=xpos+10;     
        gifdir[f].xvw=w; gifdir[f].xvh=h;
        if(v256==0)
          {
          w8=((w+7)&~7)+8;
          greyfix332(&image,w,h,xwant&7,w8);
          xwant&=~7;
          }

        if(image!=NULL)
          {
          for(y=0;y<h;y++)
            vga_drawscansegment(image+y*(v256?w:w8),xwant,
              ypos+y+GDFYBIT+39-h/2,v256?w:w8);
          free(image);
          }
        }
      else
        if(gifdir[f].isdir)
          {
          /* 'folder' icon, as usual for these types of things */
          vga_setcolor(unshow?idx_medium:idx_black);
          xt=xpos+(BARWIDTH-80)/2;
          if(left_just)xt=xpos+10;     
          ypos+=GDFYBIT+9;
          
          /* main bit */
          vga_drawline(xt+10,ypos+50,xt+70,ypos+50);
          vga_drawline(xt+70,ypos+50,xt+70,ypos+20);
          vga_drawline(xt+70,ypos+20,xt+65,ypos+15);
          vga_drawline(xt+65,ypos+15,xt+15,ypos+15);
          vga_drawline(xt+15,ypos+15,xt+10,ypos+20);
          vga_drawline(xt+10,ypos+20,xt+10,ypos+50);
          
          /* top bit */
          vga_drawline(xt+15,ypos+15,xt+20,ypos+10);
          vga_drawline(xt+20,ypos+10,xt+35,ypos+10);
          vga_drawline(xt+35,ypos+10,xt+40,ypos+15);
          ypos-=GDFYBIT+9;
          
          gifdir[f].xvw=w=80; gifdir[f].xvh=h=60;
          }
        else
          {
          /* a default icon-type-thing here */
          vga_setcolor(unshow?idx_medium:idx_black);
          xt=xpos+(BARWIDTH-80)/2;
          if(left_just)xt=xpos+10;     
          ypos+=GDFYBIT+9;
          
          /* main bit */
          vga_drawline(xt+20,ypos+50,xt+60,ypos+50);
          vga_drawline(xt+60,ypos+50,xt+60,ypos+20);
          vga_drawline(xt+60,ypos+20,xt+50,ypos+10);
          vga_drawline(xt+50,ypos+10,xt+20,ypos+10);
          vga_drawline(xt+20,ypos+10,xt+20,ypos+50);
          
          /* 'folded' bit */
          vga_drawline(xt+50,ypos+10,xt+50,ypos+20);
          vga_drawline(xt+50,ypos+20,xt+60,ypos+20);
          
          ypos-=GDFYBIT+9;
          gifdir[f].xvw=w=80; gifdir[f].xvh=h=60;
          }
      
      if(gifdir[f].xvw!=0)
        {
        xt=xpos+(BARWIDTH-w)/2;
        if(left_just)xt=xpos+10;     
        draw3dbox(xt-1,ypos+GDFYBIT+38-h/2,
                  xt+w,ypos+GDFYBIT+39-h/2+h,1,0,
                  idx_light,idx_dark);
        }
      }
    }
  }
set_max_text_width(NO_CLIP_FONT);
}


void readgifdir()
{
DIR *dirfile;
struct dirent *anentry;
struct stat buf;
char cdir[MAXPATHLEN+1];
int entnum,isdir;
struct DIRTREE *dir;
void *tmp;

getcwd(cdir,MAXPATHLEN);
dir=finddir(cdir);
if(dir->gifdir!=NULL){ gifdir=dir->gifdir; gifdirsiz=dir->gifdirsize; return; };
gifdir=malloc(sizeof(struct GDIR)*MDIRSIZ);
dirfile=opendir(".");
if(dirfile==NULL)
  {
  /* we get here if we can't read the dir.
   * Zgv tests we have permission to access a file or dir before
   * selecting it, so this can only happen if it was started on the dir
   * from the cmdline, or if the directory has changed somehow since we
   * last read it.
   * the first reaction is to try $HOME instead.
   * if *that* doesn't work, we cough and die, not unreasonably. :-)
   */
  /* be sure to mention what we're doing first... :-) */
  msgbox(zgv_ttyfd,"Directory unreadable - going to $HOME...",
  	MSGBOXTYPE_OK, idx_light,idx_dark,idx_black);
  chdir(homedir);
  dir=finddir(homedir);
  if((dirfile=opendir("."))==NULL)
    {
    screenoff();
    fprintf(stderr,"$HOME is unreadable. This is a Bad Thing. TTFN...\n");
    exit(1);
    }
  
  /* this screenon() could result in us clearing the screen twice overall
   * (if we did a ^R on a now-deleted dir, for example), but this isn't
   * the kind of thing which happens all the time, so I guess this is ok.
   */
  screenon();
  }
entnum=0;
while((anentry=readdir(dirfile))!=NULL)
  {
  if(entnum>=MDIRSIZ-1) break;
  if(!cfg.showxvpicdir || (cfg.showxvpicdir &&
  				strcmp(anentry->d_name,".xvpics")!=0))
    if(anentry->d_name[0]=='.' && anentry->d_name[1]!='.')
      continue;	/* skip (most) 'dot' files */
  if((strcmp(anentry->d_name,".")!=0) &&	/* no . and no .. if at root */
     (!((strcmp(cdir,"/")==0)&&(strcmp(anentry->d_name,"..")==0))))
    {
    if((stat(anentry->d_name,&buf))==-1)
      buf.st_mode=0;
    isdir=S_ISDIR(buf.st_mode);
    /* directories, GIF/JPG/PBM/PGM/PPM tested here */
    if(isdir || ispicture(anentry->d_name))
      {
      entnum++;
      gifdir[entnum].isdir=isdir;
      strncpy(gifdir[entnum].name,anentry->d_name,254);
      gifdir[entnum].marked=0;
      gifdir[entnum].ctime=buf.st_ctime;
      gifdir[entnum].size=buf.st_size;
      if(!cfg.fast)
         magic_ident_size(anentry->d_name,&gifdir[entnum].width,&gifdir[entnum].height);
      }
    }
  }
closedir(dirfile);
gifdirsiz=entnum;
qsort(&(gifdir[1]),gifdirsiz,sizeof(struct GDIR),(void *)gcompare);
tmp=malloc(sizeof(struct GDIR)*(gifdirsiz+1));
memcpy(tmp,gifdir,sizeof(struct GDIR)*(gifdirsiz+1));
free (gifdir);
gifdir=tmp;
dir->gifdirsize=gifdirsiz;
dir->gifdir=gifdir;
}


int ispicture(char *filename)
{
int l=strlen(filename);

if(cfg.allfiles)return(1);

if(is_cfg_picture(filename))return(1);

if(l<=4) return(0);

if((!strcasecmp(filename+l-4,".gif")) ||
   (!strcasecmp(filename+l-4,".jpg")) ||
   (!strcasecmp(filename+l-5,".jpeg")) ||
   (!strcasecmp(filename+l-4,".png")) ||
   (!strcasecmp(filename+l-4,".mrf")) ||
   (!strcasecmp(filename+l-4,".pbm")) ||
   (!strcasecmp(filename+l-4,".pgm")) ||
   (!strcasecmp(filename+l-4,".ppm")) ||
   (!strcasecmp(filename+l-4,".bmp")) ||
   (!strcasecmp(filename+l-4,".tga")) ||
   (!strcasecmp(filename+l-4,".pcx")) ||
   (!strcasecmp(filename+l-4,".pcd")) ||
   (!strcasecmp(filename+l-4,".ngm")))
  return(1);
else
  return(0);
}

int is_cfg_picture(char *filename)
{
   char *tmp=cfg.image_exts;
   int l1,l2;
   
   l1=strlen(filename);
   for(;(*tmp)&&(tmp-cfg.image_exts<1024);tmp+=l2+1) {
      l2=strlen(tmp);
      if((l1>l2)&&(!strcasecmp(filename+l1-l2,tmp)))return(1);
   };
   return(0);
}


int gcompare(void *gn1,void *gn2)
{
struct GDIR *g1,*g2;
char buf1[258],buf2[258]; /* [sic] */

g1=(struct GDIR *)gn1; g2=(struct GDIR *)gn2;
prettyfile(buf1,g1); prettyfile(buf2,g2);

if(g1->isdir==g2->isdir){
   if(g1->isdir){return(strcmp(buf1,buf2));} /* always sort dirs by name */
   else switch(cfg.sort_key)
       {
       case 8:return(g1->ctime-g2->ctime); break;
       case 9:return(g2->ctime-g1->ctime); break;   
       case 4:return(g1->size-g2->size); break;
       case 5:return(g2->size-g1->size); break;
       case 3:return(-strcmp(buf1,buf2)); break;
       case 2:
       default: return(strcmp(buf1,buf2));
       };
   } else return(g2->isdir-g1->isdir);

}


void prettyfile(char *buf,struct GDIR *gifdptr)
{
if(gifdptr->isdir)
  {
  buf[0]='(';
  strcpy(buf+1,gifdptr->name);
  strcat(buf,")");
  }
else
  strcpy(buf,gifdptr->name);
}


void screenon()
{
int r,g,b,n;
unsigned char *tmp;

msgbox_draw_ok=1;
gdfsiz=3-cfg.smallfiletext;
gdfofs=6*cfg.smallfiletext;

yofs=(cfg.fullsel?10:110);

if(cfg.xvpic_index)
  {
  if(cfg.mode_allowed[VIS_MODE] && cfg.force16fs==0)
    {
    vga_setmode(VIS_MODE); v256=1;
    /* construct 3:3:2 palette */
    for(r=n=0;r<8;r++)
      for(g=0;g<8;g++)
        for(b=0;b<4;b++,n++)
          vga_setpalette(n,r*63/7,g*63/7,b*63/3);
	  
    /* find approximations to file selector colours.
     * these are then blasted with the *real* file selector colours
     * unless 'perfectindex' is set.
     */
    idx_medium=MAKE332COL(cfg.medium.r,cfg.medium.g,cfg.medium.b);
    idx_dark  =MAKE332COL(cfg.dark.r  ,cfg.dark.g  ,cfg.dark.b  );
    idx_light =MAKE332COL(cfg.light.r ,cfg.light.g ,cfg.light.b );
    idx_black =MAKE332COL(cfg.black.r ,cfg.black.g ,cfg.black.b );
    idx_marked=MAKE332COL(cfg.marked.r,cfg.marked.g,cfg.marked.b);
    }
  else
    {
    /* if 16 colour mode... */
    vga_setmode(G640x480x16); v256=0;

    if(cfg.fs16col)
      /* colour */
      for(r=0;r<2;r++)
        for(g=0;g<2;g++)
          for(b=0;b<2;b++)
            vga_setpalette(r*4+g*2+b+1,r*63,g*63,b*63);
    else
      /* greyscale */
      for(b=0;b<=GREY_MAXVAL;b++)
        vga_setpalette(b,b*63/GREY_MAXVAL,b*63/GREY_MAXVAL,b*63/GREY_MAXVAL);

    idx_black=14; idx_marked=12; idx_dark=13; idx_medium=0; idx_light=15;
    }
  
  /* fix bar height */
  barheight=GDFYBIT+6+70;
  }
else
  {
  /* even for the old file selector we use an 8-bit mode if possible,
   * since it's much faster.
   */
  vga_setmode((vga_hasmode(VIS_MODE) && cfg.force16fs==0)?
        	VIS_MODE:G640x480x16);
  barheight=GDFYBIT+6;
  idx_medium=MIDGREY; idx_dark=DARK; idx_light=LIGHT; idx_black=BLACK;
  idx_marked=MARKEDCOL;
  if(one_file_only && vga_hasmode(VIS_MODE) && cfg.force16fs==0)
    idx_medium=255;	/* well, not 0 at any rate */
  }

if(!(cfg.xvpic_index && cfg.perfectindex))
  {
  vga_setpalette(idx_medium,cfg.medium.r,cfg.medium.g,cfg.medium.b);
  vga_setpalette(idx_dark  ,cfg.dark.r,cfg.dark.g,cfg.dark.b);
  vga_setpalette(idx_light ,cfg.light.r,cfg.light.g,cfg.light.b);
  vga_setpalette(idx_black ,cfg.black.r,cfg.black.g,cfg.black.b);
  vga_setpalette(idx_marked,cfg.marked.r,cfg.marked.g,cfg.marked.b);
  }

if(one_file_only) return;

if(cfg.xvpic_index || v256==0)
  {
  /* clear screen with 'medium' (i.e. background) colour. */
  tmp=malloc(VIS_WID);
  if(tmp!=NULL)
    {
    memset(tmp,idx_medium,VIS_WID);
    for(n=0;n<VIS_HI;n++)
      vga_drawscanline(n,tmp);
    free(tmp);
    }
  }

if(cfg.fullsel)
  {
  draw3dbox(0,0,VIS_WID-1,VIS_HI-1,2,1,	idx_light,idx_dark);
  draw3dbox(10,10,VIS_WID-11,VIS_HI-11,1,0,	idx_light,idx_dark);
  }
else
  {
  draw3dbox(0,0,VIS_WID-1,99,2,1,	idx_light,idx_dark);
  draw3dbox(10,10,VIS_WID-11,89,1,0,	idx_light,idx_dark);
  draw3dbox(0,100,VIS_WID-1,VIS_HI-1,2,1,	idx_light,idx_dark);
  draw3dbox(10,110,VIS_WID-11,VIS_HI-11,1,0,	idx_light,idx_dark);
  }

if(!cfg.fullsel)
  drawzgvlogo((VIS_WID>>1)-310,10);
}


void screenoff()
{
vga_setmode(TEXT);
cleartext();
}


void drawzgvlogo(int a,int b)
{
int x,y,bw,c=0;
byte *ptr;

ptr=zgvlogo;
bw=logow>>3;
if((logow%8)>0) bw+=1;
vga_setcolor(idx_black);
for(y=0;y<logoh;y++)
  {
  ptr=zgvlogo+y*bw;
  for(x=0;x<logow;x++)
    {
    if((x%8)==0)
      c=*ptr;
    if((c&128)==0) vga_drawpixel(a+x,b+y);
    c<<=1;
    c&=0xff;
    if((x%8)==7)
      ptr++;
    }
  }
}


void cleartext()
{
if(cfg.cleartext)
  fprintf(stderr,"\033[H\033[J");
}


/* this shows any error message from GIF reading.
 * Notice that JPEG errors have already been reported!
 */
void showerrmessage(int errnumber)
{
char buf[256];

if(updating_index) return;	/* ignore if updating thumbnail index */

strcpy(buf,"Error reading file - ");
switch(errnumber)
  {
  case _PICERR_NOFILE:
    strcat(buf,"file not found"); break;
  case _PICERR_NOMEM:
    strcat(buf,"out of memory"); break;
  case _PICERR_BADMAGIC:
    strcat(buf,"not a supported format or bad extension"); break;
  case _PICERR_NOCOLOURMAP:
    strcat(buf,"no global colour map"); break;
  case _PICERR_NOIMAGE:
    strcat(buf,"no image block found"); break;
  case _PICERR_UNSUPPORTED:
    strcat(buf,"unsupported image sub-type"); break;
  case _PICERR_CORRUPT:
    strcat(buf,"corrupt or short file"); break;
  case _PICERR_ISRLE:
    strcat(buf,"can't handle RLE BMP files"); break;
  case _PICERR_SHOWN_ALREADY:
    /* this only happens with JPEGs/PNGs */
    return;
  default:
    strcat(buf,"unknown error (ulp!)");
  }
msgbox(zgv_ttyfd,buf,MSGBOXTYPE_OK, idx_light,idx_dark,idx_black);
}


/* ok, a couple of people want move, copy, and (especially) delete,
 * and it sounds like a good idea, so here goes.
 *
 * [ move and copy aren't done yet :-( ]
 */

/* delete is the easiest.
 * we also delete any matching thumbnail file in .xvpics, and
 * attempt to remove the .xvpics directory also - relying on the
 * OS to do the Right Thing if other thumbnail files remain
 * (which it does, of course).
 */
int delete_file(char *filename,int *num)
{
char buf[270];
int retn=0;

if(cfg.nodelprompt==0)
  {
  sprintf(buf,"Really delete %s?",filename);
  retn=msgbox(zgv_ttyfd,buf,MSGBOXTYPE_YESNO, idx_light,idx_dark,idx_black);
  }

if(retn==1 || cfg.nodelprompt)
  {
  if(remove(filename)==-1)
    msgbox(zgv_ttyfd,"Unable to delete file!",MSGBOXTYPE_OK,
    				idx_light,idx_dark,idx_black);
  else
    { int i;
    sprintf(buf,".xvpics/%s",filename);
    for(i=*num;i<gifdirsiz;i++)gifdir[i]=gifdir[i+1];
    gifdirsiz--;
    (*num)--;
    remove(buf);		/* don't care if it fails */
    rmdir(".xvpics");	/* same here */
    }
  }

return(retn);
}

int cp(char *source, char *dest)
{ int source_fd,dest_fd;
  char *buf;
  struct stat source_stat;
  
  if((source_fd=open(source,O_RDONLY))==-1)return -1;
  if((dest_fd=open(dest,O_WRONLY|O_CREAT|O_TRUNC))==-1)return -2;
  stat(source,&source_stat);
  buf=malloc(source_stat.st_size);
  read(source_fd,buf,source_stat.st_size);
  write(dest_fd, buf, source_stat.st_size);
  free(buf);
  fchmod(dest_fd,source_stat.st_mode&0x0fff);
  close(dest_fd);
  close(source_fd);
  return 0;
};

int copy_file(char *filename)
{
char buf[270];
char destination[256];
int retn=0;

if(cfg.nodelprompt==0)
  {
  sprintf(buf,"Really copy %s to %s?",filename,targetdir);
  retn=msgbox(zgv_ttyfd,buf,MSGBOXTYPE_YESNO, idx_light,idx_dark,idx_black);
  }

if(retn==1 || cfg.nodelprompt)
  { 
    sprintf(destination,"%s/%s",targetdir,filename);
    if(cp(filename,destination)!=0){
       msgbox(zgv_ttyfd,"Unable to create file!",MSGBOXTYPE_OK,
    				idx_light,idx_dark,idx_black);
       return -2;
    };
    sprintf(buf,".xvpics/%s",filename);
    sprintf(destination,"%s/.xvpics",targetdir);
    mkdir(destination,0777);
    sprintf(destination,"%s/.xvpics/%s",targetdir,filename);
    cp(buf,destination);		/* don't care if it fails */
    return 0;
  }
return(retn-1);
}

int move_file(char *filename, int *num)
{    int i,j,retn;
     char buf[256];
     i=cfg.nodelprompt;
     i=cfg.nodelprompt;
     cfg.nodelprompt=1;
     if(cfg.nodelprompt==0)
     {
      	sprintf(buf,"Really move %s to %s?",filename,targetdir);
        retn=msgbox(zgv_ttyfd,buf,MSGBOXTYPE_YESNO, idx_light,idx_dark,idx_black);
     };
     j=retn;
     if(cfg.nodelprompt||(retn==1)){
          cfg.nodelprompt=1;
          if((j=copy_file(filename))==0){
             j=delete_file(filename,num);
          };
     };
     cfg.nodelprompt=i;
     return j;
};
     

void xv332_how_far(int sofar,int total)
{
char tmp[128];
int done;

done=sofar*100/total;

vga_lockvc();
if(vga_oktowrite() && ((done%10)==0 || sofar==total))
  {
  clear_xvpic(xv332_how_far_xpos,xv332_how_far_ypos);

  if(sofar!=total)
    {
    vga_setcolor(idx_black);
    sprintf(tmp,"Reading - %2d%%",done);
    vgadrawtext(xv332_how_far_xpos+(BARWIDTH-70)/2,
                xv332_how_far_ypos+GDFYBIT+39-4,2,tmp);
    }
  }
vga_unlockvc();
}


void clear_xvpic(int xpos,int ypos)
{
unsigned char tmp[96];
int y;

memset(tmp,idx_medium,96);
for(y=-3;y<63;y++)
  vga_drawscansegment(tmp,xpos+(BARWIDTH-80)/2-3,ypos+y+GDFYBIT+9,96);
}

/* if howfar equals 0, no progress report is done.
 * if it is >0, then the high 16 bits are 'xpos' and low 16 are 'ypos'.
 */
int makexv332(char *filename,char *xvpicfn,unsigned int howfar)
{
FILE *out;
int tmp;
int w,h,y;
unsigned char *smallpic;

pixelsize=1;		/* ouch */

if(howfar)
  {
  /* given the way the progress reporting from readpicture() works,
   * I have to use some global variables here. :-( 
   */
  xv332_how_far_xpos=howfar>>16;
  xv332_how_far_ypos=howfar&0xFFFF;
  }

tmp=((cfg.jpegindexstyle==3)?0:1);
if((tmp=readpicture(filename,howfar?xv332_how_far:NULL,0,tmp))!=_PIC_OK)
  return(tmp);

/* image is pointed to by theimage, image_palette */

vga_lockvc();
if(vga_oktowrite())
  {
  clear_xvpic(xv332_how_far_xpos,xv332_how_far_ypos);
  vgadrawtext(xv332_how_far_xpos+(left_just?10:(BARWIDTH-62)/2),
              xv332_how_far_ypos+GDFYBIT+39-4,2,"Resampling...");
  }
vga_unlockvc();

/* resize */
w=80; h=60;
smallpic=resizepic(theimage,image_palette+512,image_palette+256,image_palette,
		width,height,&w,&h);
free(theimage); free(image_palette);	/* finished with these */

vga_lockvc();
if(vga_oktowrite())
  {
  clear_xvpic(xv332_how_far_xpos,xv332_how_far_ypos);
  vga_setcolor(idx_black);
  vgadrawtext(xv332_how_far_xpos+(left_just?10:(BARWIDTH-55)/2),
              xv332_how_far_ypos+GDFYBIT+39-4,2,"Dithering...");
  }
vga_unlockvc();

/* dither */
ditherinit(w);
for(y=0;y<h;y++)
  ditherline(smallpic+y*80*3,y,w);
ditherfinish();

/* write */
if((out=fopen(xvpicfn,"wb"))==NULL)
  return(_PICERR_NOFILE);		/* well, kind of */

fprintf(out,"P7 332\n");
fprintf(out,"# xv-compatible thumbnail file produced by zgv v%s\n",ZGV_VER);
fprintf(out,"#END_OF_COMMENTS\n");
fprintf(out,"%d %d 255\n",w,h);

for(y=0;y<h;y++)
  fwrite(smallpic+y*80*3,1,w,out);

fclose(out);
free(smallpic);

return(_PIC_OK);
}


/* hack of above routine to work on dirs.
 * `filename' is the name of the dir in this case, of course.
 */
int makedirxv332(char *filename,char *xvpicfn,unsigned int howfar)
{
static char files[4][256];
static struct { int x,y; } xypos[4]={{0,0}, {1,0}, {0,1}, {1,1}};
FILE *out;
DIR *dirfile;
int tmp;
int w,h,y;
unsigned char *smallpic,*totalpic;
struct dirent *anentry;
struct stat buf;
int f,pos;
char tmpstr[40];

/* the dir. xvpics are a montage of the top 4 files of subdir. working out
 * which these are is even less fun than you'd think, as they aren't
 * necessarily in alphabetical order. rather than read them all in, qsort
 * that, and take the top 4, we instead keep a `top 4' table and do a kind
 * of incremental sort on that as we go along.
 */

for(f=0;f<4;f++) files[f][0]=0;

dirfile=opendir(filename);
if(dirfile==NULL) return(_PICERR_NOFILE);

while((anentry=readdir(dirfile))!=NULL)
  {
  if(anentry->d_name[0]=='.') continue;		/* skip `dot' files */
  if((stat(anentry->d_name,&buf))==-1)
    buf.st_mode=0;
  if(S_ISDIR(buf.st_mode)) continue;		/* ...and subdirs */
  if(!ispicture(anentry->d_name)) continue;	/* must be picture file */
  
  /* now check against existing files in table. start at the bottom
   * and see how high we get.
   */
  pos=-1;
  for(f=3;f>=0;f--)
    if(files[f][0]==0 || strcmp(anentry->d_name,files[f])<0)
      pos=f;
  
  if(pos>=0)
    {
    if(pos<3)
      for(f=3;f>pos;f--)
        strcpy(files[f],files[f-1]);
    
    strcpy(files[pos],anentry->d_name);
    }
  }
closedir(dirfile);

if(files[0][0]+files[1][0]+files[2][0]+files[3][0]==0)
  /* no files there, just leave it alone. */
  return(_PICERR_NOFILE);

/* so we finally have the four filenames. */

pixelsize=1;		/* ouch */

if(howfar)
  {
  /* given the way the progress reporting from readpicture() works,
   * I have to use some global variables here. :-( 
   */
  xv332_how_far_xpos=howfar>>16;
  xv332_how_far_ypos=howfar&0xFFFF;
  }

/* get total image */
if((totalpic=malloc(80*60))==NULL) return(_PICERR_NOMEM);

memset(totalpic,0,80*60);

for(f=0;f<4;f++)
  {
  vga_lockvc();
  if(vga_oktowrite())
    {
    vga_setcolor(idx_black);
    sprintf(tmpstr,"Image %d/4...",f+1);
    clear_xvpic(xv332_how_far_xpos,xv332_how_far_ypos);
    vgadrawtext(xv332_how_far_xpos+(BARWIDTH-62)/2,
                xv332_how_far_ypos+GDFYBIT+39-4,2,tmpstr);
    }
  vga_unlockvc();
  
  tmp=((cfg.jpegindexstyle==3)?0:1);
  sprintf(tmpstr,"%s/%s",filename,files[f]);
  if((tmp=readpicture(tmpstr,NULL,0,tmp))!=_PIC_OK)
    if(f==0)
      {
      free(totalpic);
      remove(xvpicfn);
      return(tmp);	/* if it's the first, abort... */
      }
    else
      continue;		/* else just skip it. */
  
  /* image is pointed to by theimage, image_palette */
  
  /* resize */
  w=40; h=30;
  smallpic=resizepic(theimage,image_palette+512,image_palette+256,
  		image_palette,width,height,&w,&h);
  free(theimage); free(image_palette);	/* finished with these */
  
  /* dither */
  ditherinit(w);
  for(y=0;y<h;y++)
    ditherline(smallpic+y*40*3,y,w);
  ditherfinish();
  
  for(y=0;y<h;y++)
    memcpy(totalpic+80*30*xypos[f].y+40*xypos[f].x+y*80,smallpic+y*40*3,w);
  
  free(smallpic);
  }

/* write */
if((out=fopen(xvpicfn,"wb"))==NULL)
  return(_PICERR_NOFILE);		/* well, kind of */

fprintf(out,"P7 332\n");
fprintf(out,"# dir thumbnail file produced by zgv v%s\n",ZGV_VER);
fprintf(out,"#END_OF_COMMENTS\n");
fprintf(out,"80 60 255\n");

fwrite(totalpic,1,80*60,out);

fclose(out);

free(totalpic);

return(_PIC_OK);
}

/* update indexes (xvpics) of current directory.
 * we draw the stuff as we go along, and allow Esc to abort
 * the process. Checking for Esc happens between creating xvpics.
 * the directory '.xvpics' is created if needed.
 */
int update_xvpics(int do_dirs_instead)
{
FILE *test;
int f;
int curent,oldstart,oldent,startfrom;
struct stat realpic,xvpic,xvpic2,tmpsbuf;
static char buf[1024],buf2[1024],altdirbuf[1024];
char *ptr;
int redraw_needed=0;
int r1=0,r2=0;
int altdirexists=0;

/* for each picture in the current directory, we check to see if
 * a file exists in the .xvpics directory with the same filename.
 * If not, it is created. If a file does exist, we check the
 * modification times. If the picture file is newer than the index
 * file, a new one gets created. Hope that's clear now. :-)
 */
redrawall(1,1);
curent=startfrom=1;
vga_runinbackground(1);
/* see if the ~/.xvpics/foo_bar_baz dir exists. If not, we can
 * avoid having to test for it below, making for far fewer stat()s.
 * this also sets altdirbuf, so we don't have to work out the dir's
 * name all the time. :-)
 */
sprintf(altdirbuf,"%s/.xvpics/",homedir);
ptr=altdirbuf+strlen(altdirbuf);
getcwd(ptr,sizeof(altdirbuf)-strlen(altdirbuf)-1);
while((ptr=strchr(ptr,'/'))!=NULL) *ptr++='_';
altdirexists=(stat(altdirbuf,&tmpsbuf)!=-1);


for(f=1;f<=gifdirsiz;f++)
  {
  /* with the do_dirs_instead mode, the first non-dir entry is a
   * sensible place to stop.
   */
  if((do_dirs_instead&1) && gifdir[f].isdir==0) break;
  
  if((do_dirs_instead&2) && (gifdir[f].isdir==1) && (strcmp(gifdir[f].name,"."))&& (strcmp(gifdir[f].name,".."))){
     int tmp;
     
     chdir(gifdir[f].name);
     readgifdir();
     redrawall(1,1);
     tmp=update_xvpics(do_dirs_instead);
     chdir("..");
     readgifdir();
     redrawall(1,1);
     if(tmp==2)return 2;
  };
  
  if(gifdir[f].isdir==(do_dirs_instead&1) && stat(gifdir[f].name,&realpic)!=-1 &&
  	gifdir[f].name[0]!='.')
    {
    /* test for normal .xvpics/wibble */
    sprintf(buf,".xvpics/%s",gifdir[f].name);
    r1=stat(buf,&xvpic);
    
    /* and for ~/.xvpics/foo_bar_baz/wibble if needed */
    if(!altdirexists)
      r2=-1;
    else
      {
      strcpy(buf2,altdirbuf);
      strcat(buf2,"/"); strcat(buf2,gifdir[f].name);
      r2=stat(buf2,&xvpic2);
      }
    
    if((r1==-1 && r2==-1) ||
       (r1!=-1 && realpic.st_mtime>xvpic.st_mtime) ||
       (r2!=-1 && realpic.st_mtime>xvpic2.st_mtime) ||
       (do_dirs_instead&1))
      {
      /* this is pretty BFI and messy */
      
      if(mkdir(".xvpics",0777)!=-1 || errno==EEXIST || errno==EACCES)
        {
        /* check if we can write to the file */
        if((test=fopen(buf,"wb"))!=NULL)
          fclose(test);
        else
          goto usehomedir;
        }
      else	/* if couldn't create/use .xvpics ... */
        {
        usehomedir:
        /* couldn't create ./.xvpics, try ~/.xvpics */
        sprintf(buf,"%s/.xvpics",homedir);
        if(mkdir(buf,0777)==-1 && errno!=EEXIST && errno!=EACCES)
          {
          wait_for_foreground();
          msgbox(zgv_ttyfd,"Unable to create either .xvpics directory",
          		MSGBOXTYPE_OK,idx_light,idx_dark,idx_black);
          return 0;
          }
        
        /* also need to create ~/.xvpics/_foo_bar_baz */
        if(mkdir(altdirbuf,0777)==-1 && errno!=EEXIST && errno!=EACCES)
          {
          wait_for_foreground();
          msgbox(zgv_ttyfd,"Unable to create ~/.xvpics/... directory",
          		MSGBOXTYPE_OK,idx_light,idx_dark,idx_black);
          return 0;
          }
        
        /* make sure filename for xvpic is in buf */
        strcpy(buf,altdirbuf); strcat(buf,"/"); strcat(buf,gifdir[f].name);
        }
      
      if((do_dirs_instead&1))
        makedirxv332(gifdir[f].name,buf,(fwinxpos(f-startfrom+1)<<16) |
        					fwinypos(f-startfrom+1));
      else
        makexv332(gifdir[f].name,buf,(fwinxpos(f-startfrom+1)<<16) |
        					fwinypos(f-startfrom+1));
      
      /* redraw without mode change */
      vga_lockvc();
      if(vga_oktowrite())
        {
        if(redraw_needed)	/* if a full redraw is needed... */
          screenon();
        else
          showbar(curent,0,startfrom);
        showgifdir(startfrom,1,redraw_needed);
        showgifdir(startfrom,0,redraw_needed);
        if(!redraw_needed)
          showbar(curent,1,startfrom);
        redraw_needed=0;
        }
      else	/* if we're not on current vc */
        redraw_needed=1;
      vga_unlockvc();
      }
    }
  
  /* move down one if possible */
  oldent=curent; oldstart=startfrom;
  if(curent<gifdirsiz) curent++;
  
  /* move right if needed */
  while(fwinxpos(curent-startfrom+1)+BARWIDTH>COLS)
    startfrom+=YSIZ;

  vga_lockvc();

  /* redraw */
  if(vga_oktowrite() && !redraw_needed)
    if(startfrom!=oldstart)
      {
      showbar(oldent,0,oldstart);
      showgifdir(oldstart,1,1);
      showgifdir(startfrom,0,1);
      showbar(curent,1,startfrom);
      }
    else  
      if(curent!=oldent)
        {
        showbar(oldent,0,startfrom);
        showbar(curent,1,startfrom);
        }

  vga_unlockvc();      

  /* check for Esc */
  if(readnbkey(zgv_ttyfd)==RK_ESC)
    {
    wait_for_foreground();
    return 2;
    }
  }

wait_for_foreground();
usleep(200000);	/* for effect :-) */
return 0;
}

/* if we're not running directly on a console, try to find a free console
 * and move us to it. Notes old VT so we can switch back to it when finished,
 * if we ran on a different one to start off with.
 *
 * svgalib 1.2.11 and up do something similar (which was based on this,
 * I think), but it doesn't *quite* do what we need (e.g. it changes stdout,
 * which is sensible generally but zgv wants to keep it the same). So, er,
 * that's why this code's still here. :-)
 */
int fixvt()
{
struct stat sbuf;
struct vt_stat vts;
int major,minor;
int fd;
int num;
char vt_filename[20];

/* see if terminal is a console */
fd=dup(2);
fstat(fd,&sbuf);
major=sbuf.st_rdev>>8;
zgv_vt=minor=sbuf.st_rdev&0xff;
close(fd);
if(major==4 && minor<64)
  return(1);	/* if on a console, already ok */

/* otherwise we need to look for a free VT, redirect std{in,out,err},
 * and switch to it. If there's no free VTs, give up now.
 */

separate_vt=1;

/* still root perms, so this shouldn't be a problem... */
if((fd=open("/dev/console",O_WRONLY))<0) return(0);
ioctl(fd,VT_GETSTATE,&vts);
original_vt=vts.v_active;
ioctl(fd,VT_OPENQRY,&num);
if(num==-1) return(0);	/* no VTs free */

/* now, before we go for it, we test the *current* VT to see if they
 * own it. If so, the user's probably `genuine'.
 * (NB: the kernel now does this, but there's no harm repeating it.)
 */
sprintf(vt_filename,"/dev/tty%d",original_vt);
stat(vt_filename,&sbuf);
if(getuid()!=sbuf.st_uid)
  {
  fprintf(stderr,"You must be the owner of the current console to run zgv.\n");
  exit(1);
  }

/* switch to the new VT */
ioctl(fd,VT_ACTIVATE,num);
close(fd);

/* This is incredibly annoying, but the 2.0.x kernel just *will not*
 * work without it. :-(((
 * So, this gives really weird results for `zgv -h' etc., as the parent
 * returns immediately. Redirect stdout if this is a problem.
 */
if(fork()) exit(0);

zgv_vt=num;
sprintf(vt_filename,"/dev/tty%d",num);

setsid();

if(freopen(vt_filename,"r",stdin)==NULL)	return(0);
if(freopen(vt_filename,"w",stderr)==NULL)	return(0);
/* we keep stdout - doesn't need to be changed */

/* not needed, but... just in case... */
chown(vt_filename,getuid(),getgid());

/* ok, done it. */
return(1);
}


void switchback()
{
struct vt_mode vtm;

/* also change back to stdin blocking;
 * some versions of bash seem to be a little sensitive to it being
 * left non-blocking.
 */
fcntl(zgv_ttyfd,F_SETFL,0);

if(separate_vt)
  {
  fprintf(stderr,"%c[H%c[J",27,27);	/* seems to get junk-filled... */
  ioctl(zgv_ttyfd,VT_GETMODE,&vtm);
  vtm.mode=VT_AUTO;
  ioctl(zgv_ttyfd,VT_SETMODE,&vtm);
  ioctl(zgv_ttyfd,VT_ACTIVATE,original_vt);
  }
}


void greyfix332(unsigned char **image,int w,int h,int xw7,int w8)
{
unsigned char *new,*ptr;
int x,y,lx,c;
int c0,c1,c2,times2;
int terr0,terr1,terr2,actual0,actual1,actual2;
int start,addon,r,g,b;
int *evenerr,*odderr;
int *thiserr;
int *nexterr;

if((new=malloc(w8*h))==NULL)
  {
  free(*image);
  *image=NULL;
  return;
  }

memset(new,idx_medium,w8*h);

ptr=*image;

if(cfg.fs16col)
  {
  /* colour */
  /* uh-huh, case and paste mode on */
  
  if((evenerr=calloc(3*(w+10),sizeof(int)))==NULL ||
     (odderr =calloc(3*(w+10),sizeof(int)))==NULL)
    exit(1);
  
  for(y=0;y<h;y++)
    {
    ptr=*image+w*y;
    
    if((y&1)==0)
      {start=0; addon=1;
      thiserr=evenerr+3; nexterr=odderr+w*3;}
    else
      {start=w-1; addon=-1;
      thiserr=odderr+3; nexterr=evenerr+w*3;}

    nexterr[0]=nexterr[1]=nexterr[2]=0;
    x=start;
    for(lx=0;lx<w;lx++)
      {
      c=ptr[x];
      b=(c&3)*64; g=((c>>2)&7)*32; r=(c>>5)*32;
      r=(r-128)*3/2+128;
      g=(g-128)*3/2+128;
      b=(b-128)*3/2+128;

      terr0=r+((thiserr[0]+8)>>4);
      terr1=g+((thiserr[1]+8)>>4);
      terr2=b+((thiserr[2]+8)>>4);
    
      actual0=(terr0>=128)?255:0;
      actual1=(terr1>=128)?255:0;
      actual2=(terr2>=128)?255:0;
      
      if(actual0<0) actual0=0; if(actual0>255) actual0=255;
      if(actual1<0) actual1=0; if(actual1>255) actual1=255;
      if(actual2<0) actual2=0; if(actual2>255) actual2=255;
      
      c0=terr0-actual0;
      c1=terr1-actual1;
      c2=terr2-actual2;
    
      times2=(c0<<1);
      nexterr[-3] =c0; c0+=times2;
      nexterr[ 3]+=c0; c0+=times2;
      nexterr[ 0]+=c0; c0+=times2;
      thiserr[ 3]+=c0;
    
      times2=(c1<<1);
      nexterr[-2] =c1; c1+=times2;
      nexterr[ 4]+=c1; c1+=times2;
      nexterr[ 1]+=c1; c1+=times2;
      thiserr[ 4]+=c1;
    
      times2=(c2<<1);
      nexterr[-1] =c2; c2+=times2;
      nexterr[ 5]+=c2; c2+=times2;
      nexterr[ 2]+=c2; c2+=times2;
      thiserr[ 5]+=c2;
    
      new[y*w8+x+xw7]=(actual0&1)*4+(actual1&1)*2+(actual2&1)+1;
    
      thiserr+=3;
      nexterr-=3;
      x+=addon;
      }
    }
  
  free(evenerr);
  free(odderr);
  }
else
  /* greyscale */
  for(y=0;y<h;y++)
    for(x=0;x<w;x++,ptr++)
      {
      c=*ptr;
      c=((c&3)*2+((c>>2)&7)+(c>>5))*GREY_MAXVAL/21; /* greyscale 332 colour */
      if(c<0) c=0; if(c>GREY_MAXVAL) c=GREY_MAXVAL;
      new[y*w8+x+xw7]=c+1;
      }
    
free(*image);
*image=new;
}


void wait_for_foreground()
{
vga_runinbackground(0);
ioctl(zgv_ttyfd,VT_WAITACTIVE,zgv_vt);
}


void ctrlc(int foo)
{
switchback();		/* and go back to blocking input */
_exit(1);
}

void parentname(char *path, char *parent)
{ 
  char *tmp;

  if(strcmp(path,"/")==0){strcpy(parent,"") ; return; };
  if((tmp=strrchr(path,'/'))==NULL){strcpy(parent,""); return; };
  if(*(tmp+1)=='\0'){*tmp='\0'; if((tmp=strrchr(path,'/'))==NULL){strcpy(parent,""); return; }; };
  if(tmp==path){ strcpy(parent,"/"); return; };
  if(tmp<path){strcpy(parent,""); return; };
  strncpy(parent,path,tmp-path);
  *(parent+(tmp-path))='\0';
};

void *finddir(char *path)
{ char parent[256];
  struct DIRTREE *tmp;

  if(strcmp(path,"/")==0){return root; };
  parentname(path,parent);
  tmp=(struct DIRTREE *)finddir(parent);
  if(tmp->child==NULL)
  {
     tmp->child=malloc(sizeof(struct DIRTREE));
     strcpy(tmp->child->name,path);
     tmp->child->sib=tmp->child->child=NULL;
     tmp->child->gifdir=NULL;
     tmp->child->gifdirsize=0;
     return tmp->child;
  } else {
     tmp=tmp->child;
     while(strcmp(path,tmp->name)&&(tmp->sib!=NULL))tmp=tmp->sib;
     if(strcmp(path,tmp->name))
     {
         tmp->sib=malloc(sizeof(struct DIRTREE));
         strcpy(tmp->sib->name,path);
         tmp->sib->sib=tmp->sib->child=NULL;
         tmp->sib->gifdir=NULL;
         tmp->sib->gifdirsize=0;
         return tmp->sib;
     } else {
     return tmp;
     };
  };
};
 
int rmkdir(char *path)
{ char tmp[256];
  struct stat st;
  
  if((stat(path,&st))&&(errno==ENOENT)){
      parentname(path,tmp);
     if(rmkdir(tmp))return -1;
     return mkdir(path,st.st_mode);
  } else return S_ISDIR(st.st_mode);
};

void force_readgifdir(char *path)
{
   struct DIRTREE *dt;
   char cdir[MAXPATHLEN+1];

   dt=finddir(path);
   if(dt->gifdir!=NULL){
        free(dt->gifdir);
        dt->gifdir=NULL;
        getcwd(cdir,MAXPATHLEN);
        if(!strcmp(path,cdir)){
          readgifdir();
        };
   };
};
 
