/* Zgv v4.2 - GIF, JPEG and PBM/PGM/PPM viewer, for VGA PCs running Linux.
 * Copyright (C) 1993-1995 Russell Marks. See README for license details.
 *
 * gifeng.c - GIF decoding engine.
 *
 * Changes by Matan Ziv-Av to read multiple images (animated).
 */

#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include "zgv.h"
#include "gifeng.h"

#define MAXVAL  4100
#define MAXVALP 4200
typedef struct 
{
  unsigned char valid;
  unsigned char data;
  unsigned char first;
  unsigned char res;
  long last;
} GIF_Table;
static unsigned long root_code_size,code_size,CLEAR,EOI,INCSIZE;
static unsigned long nextab;
static unsigned long gif_mask[16] = {1,1,3,7,15,31,63,127,255,511,1023,2047,4095,8191,0,0};
static unsigned char gif_buff[MAXVALP];
static unsigned long gif_block_size;
static long gif_bits,num_bits;
static unsigned long gif_lace_flag;
static unsigned long pic_i, pic_size;
static GIF_Table table[MAXVALP];
unsigned long GIF_Decompress();
unsigned long GIF_Get_Next_Entry();
void GIF_Add_To_Table();
unsigned long GIF_Send_Data();
void GIF_Init_Table();
void GIF_Clear_Table();
unsigned long GIF_Get_Code();


static byte *image;
hffunc howfar;
FILE *global_gif_infile;   /* only used for error cleanup */
byte *global_palette_ptr;  /* same here */

static int passnum,passyloc,passstep;     /* for interlaced GIFs */
static int interlaced,width,height,bpp,numcols, gnumcols, lnumcols;

static int pwr2[16]={1,2,4,8,16,32,64,128,256,512,1024,2048,4096,8192,
                             16384,32768};

static int local_colour_map, global_colour_map;
static int transparent, delay, control;

struct {
  char sig[6];           /* should be GIF87a or GIF89a */
  byte wide_lo,wide_hi;  /* NUXI Problem Avoidance System (tm) */
  byte high_lo,high_hi;  /* these are 'screen size', BTW */
  byte misc;             /* misc, and bpp */
  byte back;             /* background index */
  byte zero;             /* if this ain't zero, problem */
  } gifhed;
  
/* BTW, the NUXI thing above is because most of this code is reused from
 * a GIF viewer I did for Tektronix 4200 series terminals. If you want a copy,
 * mail me, but I figure not many people use them.
 * (at least, not from Linux PCs :-))
 */
  
struct {
  byte left_lo,left_hi;  /* usually zero - ignore */
  byte top_lo,top_hi;
  byte wide_lo,wide_hi;  /* this is 'image size', often the same as screen */
  byte high_lo,high_hi;
  byte misc;
  } imagehed;
  
typedef struct {
  int left,top, width, height, numcols, misc, 
      delay, transparent, control;
  unsigned char *cmap, *image;
} gif_image ;

/* to this vars we read the gif */
unsigned char *global_cmap;
int swidth,sheight;
int imagecount;
gif_image *images[256];
/********************************/


/* prototypes */
int readgif(char *giffn,byte **theimageptr,byte **palptr,
            hffunc howfarfunc,int errignore, PICINFO *ginfo);
int readgiffile(char *giffn);
void dummyfunc(int a, int b);
int readgifhed(FILE *in);
void readcolmap(FILE *in, byte *palette, int numcols);
int readimagehed(FILE *in);
int decompress(FILE *in);
void inittable(void);
void addstring(int oldcode, int chr);
int readcode(int *newcode, int numbits, FILE *in);
void outputstring(int code);
void outputchr(int code);
int findfirstchr(int code);
void aborted_file_gif_cleanup(void);
void cleanup();

/* use like this:
 *   result=readgif("thingy.gif",&ptr_to_image_data,&ptr_to_palette_data,
 *                               show_how_far_func,errignore);
 * show_how_far_func, for percent complete displays etc., can be NULL.
 * any value of 'result' other than _PIC_OK means something went wrong;
 * see zgv.h for more details and error codes.
 */

int readgif(char *giffn,byte **theimageptr,byte **palptr,
            hffunc howfarfunc,int errignore, PICINFO *ginfo)
{
  int i;
  
  if(howfarfunc==NULL)
    howfar=dummyfunc;
  else
    howfar=howfarfunc;           

  if ((i=readgiffile(giffn))!=_PIC_OK){
    cleanup();
    return(i);
  };

if(imagecount==1)
{
  ginfo->width=images[0]->width;
  ginfo->height=images[0]->height;
  gifhed.wide_lo=0; 
  strcpy(ginfo->type,gifhed.sig+3);
  ginfo->bpp=(images[0]->misc&0x07)+1;
  ginfo->numcols=lnumcols ? lnumcols : gnumcols ;

  if((*theimageptr=(byte *)malloc((long)ginfo->width*(long)ginfo->height))==NULL)
    {
    cleanup();
    return(_PICERR_NOMEM);
    }

  if((*palptr=(byte *)malloc(768))==NULL)
    {
    cleanup();
    free(*theimageptr);
    return(_PICERR_NOMEM);
    }

  memcpy(*theimageptr,images[0]->image,ginfo->width*ginfo->height);
  if(images[0]->cmap)memcpy(*palptr,images[0]->cmap,768); else
  if(global_cmap)memcpy(*palptr,global_cmap,768);
} else 
{
  char *ptr;
  
  ginfo->width=swidth;
  ginfo->height=sheight*imagecount;
  gifhed.wide_lo=0; 
  strcpy(ginfo->type,gifhed.sig+3);
  ginfo->bpp=(gifhed.misc&0x07)+1;
  ginfo->numcols=256/*1<<ginfo->bpp*/;
  if((*theimageptr=(byte *)malloc(ginfo->width*ginfo->height))==NULL)
    {
    cleanup();
    return(_PICERR_NOMEM);
    }
  
  if((*palptr=(byte *)malloc(768))==NULL)
    {
    cleanup();
    free(*theimageptr);
    return(_PICERR_NOMEM);
    }

    for(i=0;i<1;i++){
      int j;
      if(images[i]->cmap) {
        for(j=0;j<images[i]->numcols;j++) {
          (*palptr)[j]=images[i]->cmap[j];
          (*palptr)[j+256]=images[i]->cmap[j+256];
          (*palptr)[j+512]=images[i]->cmap[j+512];
        };
      };
    };

  if(global_cmap){
    int j;
    for(j=0;j<gnumcols;j++) {
      (*palptr)[j]=global_cmap[j];
      (*palptr)[j+256]=global_cmap[j+256];
      (*palptr)[j+512]=global_cmap[j+512];
    };
  };
  ptr=*theimageptr;
  memset(ptr,0,swidth*sheight);
  for(i=0;i<imagecount;i++){
     int x,y;
     char *ptr1, *ptr2;
     
     if(i>0) {
       switch ((images[i-1]->control&0x1c)>>2) {
          case 0: case 1: memcpy(ptr,ptr-swidth*sheight,swidth*sheight); break;
          case 2:
            memcpy(ptr,ptr-swidth*sheight,swidth*sheight);
            for(y=images[i-1]->top;y<images[i-1]->top+images[i-1]->height;y++)
              memset(ptr+y*swidth+images[i-1]->left,0,images[i-1]->width);
            break;
          case 3:if(i==1)memset(ptr,gifhed.back,swidth*sheight); else 
            memcpy(ptr,ptr-swidth*sheight*2,swidth*sheight);
            break;
       };
     };
     ptr1=ptr+images[i]->left+images[i]->top*swidth;
     ptr2=images[i]->image;
     for(y=0;y<images[i]->height;y++) {
       for(x=0;x<images[i]->width;x++)
       if(!(images[i]->control&1)||(images[i]->transparent!=*ptr2))
           *(ptr1++)=*(ptr2++); else { ptr1++; ptr2++; };
       ptr1+=swidth-images[i]->width;
     };
  ptr+=swidth*sheight;
  };
}

  ginfo->extlength=imagecount+1;
  ginfo->extra[0]=imagecount;
  for(i=0;i<imagecount;i++)*(int *)(&ginfo->extra[2*i+1])=images[i]->delay;

  cleanup();
  return(_PIC_OK);
  
}


int readgiffile(char *giffn)
{
FILE *in;
int poserr;
int i;
gif_image *im;

for(i=0;i<256;i++)images[i]=NULL;
global_cmap=NULL;
imagecount=0;

if((in=global_gif_infile=fopen(giffn,"rb"))==NULL)
  return(_PICERR_NOFILE);
else
  {  
  if((poserr=  readgifhed(in)  )!=_PIC_OK)
    {
    fclose(in);
    return(poserr);
    }
    
  swidth=(gifhed.wide_lo+256*gifhed.wide_hi);
  sheight=(gifhed.high_lo+256*gifhed.high_hi);
  if(global_colour_map) {
    if((global_cmap=(byte *)malloc(768))==NULL) {
      fclose(in);
      return(_PICERR_NOMEM);
    }
    readcolmap(in,global_cmap,numcols);
  };

  while((poserr=readimagehed(in))==_PIC_OK) {

    im=images[imagecount++]=(gif_image *)malloc(sizeof(gif_image));  
    im->cmap=NULL;
    im->width=width;
    im->height=height;
    im->left=(imagehed.left_lo)+(imagehed.left_hi<<8);
    im->top=(imagehed.top_lo)+(imagehed.top_hi<<8);
    im->misc=imagehed.misc;
    im->numcols=lnumcols;
    im->control=control;
    im->delay=delay;
    im->transparent=transparent;
    
    if(local_colour_map) {
      im->cmap=malloc(768);
      readcolmap(in,im->cmap,lnumcols);
    };

    if((im->image=(byte *)malloc(width*height))==NULL) {
      fclose(in);
      return(_PICERR_NOMEM);
    }

    image=im->image;

    memset(im->image,0,width*height);
  
    pic_size=width*height;
    pic_i=0;
    gif_lace_flag=interlaced;
    
    if( GIF_Decompress(in,image) !=_PIC_OK) {
      fclose(in);
      return(_PICERR_CORRUPT);
    }
    fgetc(in); 
  };

  fclose(in);
  if(poserr==_PICERR_NOMORE||((poserr==_PICERR_NOIMAGE)&&(imagecount>0)))poserr=_PIC_OK;
  return(poserr);
};
}

/* use if no real howfar func. is to be used */
void dummyfunc(int a,int b)
{
}


int readgifhed(FILE *in)
{
fread(&gifhed,sizeof(gifhed),1,in);
if(strncmp(gifhed.sig,"GIF",3))
  return(_PICERR_BADMAGIC);
global_colour_map=(gifhed.misc&128)>>7;
bpp=(gifhed.misc&7)+1;
gnumcols=numcols=pwr2[bpp];
return(_PIC_OK);
}


void readcolmap(FILE *in,byte *palette,int numcols)
{
int f;

for(f=0;f<numcols;f++)
  {
  palette[    f]=(byte)fgetc(in);
  palette[256+f]=(byte)fgetc(in);
  palette[512+f]=(byte)fgetc(in);  
  }
}


int readimagehed(FILE *in)
{
int c,f,i;

control=0;
delay=0;
transparent=0;
c=fgetc(in);
while((c=='!'))      
  {
  i=fgetc(in);
  switch(i) {
    case 0xf9:
      fgetc(in);
      control=fgetc(in);
      delay=fgetc(in);
      delay+=fgetc(in)<<8;
      transparent=fgetc(in);
      c=fgetc(in);
    break;
    default:
      c=fgetc(in);
      while(c!=0)
        {    /* well then, c = number of bytes, so ignore that many */
        for(f=1;f<=c;f++) fgetc(in);
        c=fgetc(in);
        }
  };
  c=fgetc(in);     /* test for image again */
  }
if(c==';')return(_PICERR_NOMORE);
if(c!=',')
  return(_PICERR_NOIMAGE);
fread(&imagehed,sizeof(imagehed),1,in);

local_colour_map=(imagehed.misc&0x80)>>7;

if((imagehed.misc&64)!=0)
  {
  interlaced=1;
  passnum=1;
  passyloc=0;
  passstep=8;
  }
else
  interlaced=0;

width=(imagehed.wide_lo+256*imagehed.wide_hi);
height=(imagehed.high_lo+256*imagehed.high_hi);

lnumcols=0;
if(local_colour_map) lnumcols=pwr2[(imagehed.misc&7)+1];

return(_PIC_OK);
}

void aborted_file_gif_cleanup()
{
cleanup();
}


void cleanup()
{
int i;

for(i=0;i<256;i++)if(images[i]){
   if(images[i]->cmap)free(images[i]->cmap);
   if(images[i]->image)free(images[i]->image);
   free(images[i]);
   images[i]=NULL;
};
};


unsigned long GIF_Decompress(FILE *in,char *pic)
{
  register long code,old;

  gif_bits = 0;
  num_bits=0;
  gif_block_size=0;

  root_code_size=fgetc(in); 
  GIF_Clear_Table();  

  code=GIF_Get_Code(in);

  if (code==CLEAR) 
  {
    GIF_Clear_Table(); 
    code=GIF_Get_Code(in);
  }
  
  if (GIF_Send_Data(code,pic) == 0) return(_PICERR_CORRUPT);
  old=code;
  code=GIF_Get_Code(in);
  do
  {
    if (table[code].valid == 1)    
    {
      if (GIF_Send_Data(code,pic)==0) return(_PICERR_CORRUPT);
      if (GIF_Get_Next_Entry(in) == 0) return(_PICERR_CORRUPT);
      GIF_Add_To_Table(old,code,nextab); 
      old=code;
    }
    else
    {
      GIF_Add_To_Table(old,old,code); 
      if (GIF_Send_Data(code,pic) == 0) return(_PICERR_CORRUPT);
      old=code;
    }
    code=GIF_Get_Code(in);
    if (code==CLEAR)
    { 
      GIF_Clear_Table();
      code=GIF_Get_Code(in);
      if (GIF_Send_Data(code,pic) == 0) return(_PICERR_CORRUPT);
      old=code;
      code=GIF_Get_Code(in);
    }
  } while(code!=EOI);
  return(_PIC_OK);
}

unsigned long GIF_Get_Next_Entry(FILE *in)
{
  while( (table[nextab].valid==1) && (nextab<MAXVAL) ) nextab++;

  if (nextab>=MAXVAL)    
  { if ( feof(in) ) 
	{ return(0); }
    else 
    { 
      nextab = MAXVAL - 1;
    }
  }
  if (nextab == INCSIZE)   
  {
    code_size++; INCSIZE=(INCSIZE*2)+1;
    if (code_size>=12) code_size=12;
  }
  return(1);
}


void GIF_Add_To_Table(body,next,index)
register unsigned long body,next,index;
{
  if (index>MAXVAL)
  { 
  }
  else
  {
    table[index].valid=1;
    table[index].data=table[next].first;
    table[index].first=table[body].first;
    table[index].last=body;
  }
}

unsigned long GIF_Send_Data(index,pic)
register long index;
char *pic;
{
  register long i,j;
  i=0;
  do      
  { 
    gif_buff[i] = table[index].data; 
    i++;
    index = table[index].last;
    if (i>MAXVAL)
    { 
      return(0);
    }
  } while(index >= 0);

  i--;
  for(j=i;j>=0;j--) 
  {
    if (pic_i < pic_size) pic[pic_i++] = gif_buff[j];
    if (gif_lace_flag)
    {
      if ((pic_i % width) == 0 )
      { 
        switch(gif_lace_flag)
        {
          case 1: pic_i += width * 7; break; 
          case 2: pic_i += width * 7; break; 
          case 3: pic_i += width * 3; break; 
          case 4: pic_i += width    ; break; 
        }
      }
      if (pic_i >= pic_size)
      {
        gif_lace_flag++;
        switch(gif_lace_flag)
        {
          case 2: pic_i = width << 2; break;  
          case 3: pic_i = width << 1; break;  
          case 4: pic_i = width;      break;  
          default: gif_lace_flag = 0; pic_i = 0; break;
        }
      }
    } 
  } 
  return(1);
}


void GIF_Init_Table()       
{
  register long maxi,i;

  maxi=pwr2[root_code_size];
  for(i=0; i<maxi; i++)
  {
    table[i].data=i;   
    table[i].first=i;
    table[i].valid=1;  
    table[i].last = -1;
  }
  CLEAR=maxi; 
  EOI=maxi+1; 
  nextab=maxi+2;
  INCSIZE = (2*maxi)-1;
  code_size=root_code_size+1;
}


void GIF_Clear_Table()   
{
 register int i;
 for(i=0;i<MAXVAL;i++) table[i].valid=0;
 GIF_Init_Table();
}


unsigned long GIF_Get_Code(FILE *in) 
{
  unsigned long code;
  register unsigned long tmp;

  while(num_bits < code_size)
  {
    if (gif_block_size==0)
    {
      tmp = fgetc(in);
      if (tmp > 0 ) gif_block_size=tmp;
      else { return(EOI); }
    }

    tmp = fgetc(in);   gif_block_size--;
    if (tmp >= 0)
    {
      gif_bits |= ( ((unsigned long)(tmp) & (unsigned long)(0xff)) << num_bits );
      num_bits+=8;
    }
    else { return(EOI); }
  }

  code = gif_bits & gif_mask[code_size];
  gif_bits >>= code_size;  
  num_bits -= code_size;

  if (code>MAXVAL)
  {
    code=EOI;
  }

  if (code==INCSIZE)
  {
    if (code_size<12)
    {
      code_size++; INCSIZE=(INCSIZE*2)+1;
    }
  }
  return(code);
}

