/*
 * xcid.C - XCallerID daemon source file
 * Copyright (c) 1998 Joe Yandle <yandle@cs.unc.edu>
 * 
 * The xcid daemon runs in the background, updating the database
 * and announcing calls as they come in.
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 * Version 1.1: Jun 1998
 * 
 * Todo for v1.2:
 *	add interprocess communication
 *
 * Changes for v1.1:
 * 	show call name in display ***DONE***
 * 	add support for CallerID Name service ***DONE***
 * 	completed 26Jun98
 */

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/file.h>
#include <fcntl.h>
#include <unistd.h>
#include <termios.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include <msql.h>
#include <signal.h>

// global constants
const bool 	ECHO_INIT = false;
const int 	WIDTH  = 400;
const int 	HEIGHT = 100;
const int 	XPOS = 200;
const int 	YPOS = 200;
const double 	WAIT = 0.5;

// global strings
char* 		OK = "OK";
char* 		RING = "RING";

char 		ENDLINE = 10;
bool 		unmapped = true;

// msql strings
char* 		dbase = "CallerID";
char* 		create = "CREATE TABLE calls ( number char(15) not null, name text(20), call_date date, call_time time, incoming char(50) )";
char* 		drop = "DROP TABLE calls";
char* 		insert = "INSERT INTO calls ( number, name, call_date, call_time, incoming ) VALUES ";

// command buffer
char  		command[256];

// msql socket variables
int 		sock;
m_result*	query_ret;

// XWindows global variables
Display*	dpy = NULL;
Window		win;
int 		screen;
GC		gc;
XGCValues	values;
XFontStruct*	font;
XEvent 		evt;

// file descriptor
int 		fd;

// function prototypes

// signal handler
void sighandle( int sig );

// speak .wav file
void speak( char* file );

// display msg, unmapping the display if killdpy, 
// and pausing for pause seconds
void display( char* msg1, char* msg2, bool killdpy, double pause );

// read a line from fd into buffer
void readline( int fd, char* buffer );

// write buffer into fd, waiting for it to echo
// and for the subsequent OK
void writeline( int fd, char* buffer );

// convert nmbr into either xxx-xxx-xxxx, 
// Out of Area, or Private
void convert( char* nmbr, char* number );


int main() {
  char buf[100];
  char fnumber[20];
  char initstring[50];
  char call_name[20];
  char incoming[100];
  char speakbuf[100];
  time_t time_now;
  buf[0] = 0;

  sprintf( speakbuf, "%s/wav/welcome.wav", XCID_HOME );
  speak(speakbuf);

  display( "Connecting to mSQL server", NULL, false, WAIT );
  if( (sock=msqlConnect(NULL)) == -1 ) {
    display( "Error connecting to server", NULL, true, WAIT );
    exit(1);
  }

  sprintf( initstring, "Opening database %s", dbase );
  display( initstring, NULL, false, WAIT );
  if( msqlSelectDB(sock,dbase) == -1 ) {
    display( "Error opening database", NULL, true, WAIT );
    exit(1);
  }

  // create table if it doesn't exist
  msqlQuery(sock,create);

  sprintf( initstring, "Opening device %s", MODEM );
  display( initstring, NULL, false, WAIT );
  if( (fd = open( MODEM, O_RDWR | O_SYNC )) == -1 ) {
    display( "Error opening device", NULL, true, WAIT );
    exit(1);
  }
  flock( fd, LOCK_EX );
  signal(SIGTERM, sighandle);

  sprintf( initstring, "Initializing CallerID on %s", MODEM );
  display( initstring, NULL, false, 0 );

  writeline(fd,MODEM_RESET);
  writeline(fd,MODEM_INIT);

  sprintf( initstring, "Initialization complete.\n");
  display( initstring, NULL, true, WAIT );

  while(true) {
    readline(fd,buf);
    if( strstr(buf,"NO CARRIER")) {
      display("NO CARRIER", NULL, true, WAIT);
      exit(1);
    }
    if( strstr(buf,"NMBR") ) {
      sprintf( speakbuf, "%s/wav/incoming.wav", XCID_HOME );
      speak(speakbuf);

      time_now = time(NULL);
      convert(buf,fnumber);

      sprintf( command, "SELECT name, incoming FROM calls WHERE number = '%s'", fnumber );

      if( msqlQuery(sock,command) > 0 ) {
	query_ret = msqlStoreResult();
	m_row row = msqlFetchRow(query_ret);
	strcpy(call_name, row[0]);
	strcpy(incoming, row[1]);
      }
      else if( !strcmp(SERVICE,"NMNB") ) {
	readline(fd,buf);
	for( unsigned int i = 0; i < strlen(buf)-7; i++ )
	  call_name[i] = buf[i+7];
	call_name[strlen(buf)-7] = 0;
	sprintf(incoming, "%s/wav/unknown.wav", XCID_HOME);
      }
      else {
	strcpy(call_name, "Unknown");
	sprintf(incoming, "%s/wav/unknown.wav", XCID_HOME);
      }

      sprintf( command, "INSERT INTO calls ( number, name, call_date, call_time, incoming ) VALUES ( '%s', '%s', '%s', '%s', '%s' )", fnumber, call_name, msqlUnixTimeToDate(time_now), msqlUnixTimeToTime(time_now), incoming );

      msqlQuery(sock,command);
      speak(incoming);
      display( fnumber, call_name, true, 10 );
    }
  }

  close( fd );
  return 0;

}


void sighandle( int sig ) {
  if( sig == SIGTERM ) {
    flock( fd, LOCK_UN );
    close(fd);
    char speakbuf[100];
    sprintf(speakbuf, "%s%s", XCID_HOME, "/wav/leaving.wav");
    speak(speakbuf);
    exit(0);
  }
}

void speak( char* file ) {
  if( !fork() ) {
    execlp( "wavplay", "wavplay", "-q", file, NULL );
    exit(1);
  }
}

void display( char* msg1, char* msg2, bool killdpy, double pause ) {
  
  if(!dpy) {
    dpy=XOpenDisplay(NULL);
    screen=DefaultScreen(dpy);
    win=XCreateSimpleWindow(dpy,RootWindow(dpy,screen),XPOS,YPOS,WIDTH,HEIGHT,0,
			    WhitePixel(dpy,screen),WhitePixel(dpy,screen));
    XSelectInput(dpy, win, ExposureMask | ButtonPressMask);
    XSetStandardProperties(dpy, win, "CallerID", "CallerID", None, NULL,0, NULL);
    values.foreground = BlackPixel(dpy,screen);
    values.background = WhitePixel(dpy,screen);
    values.function   = GXcopy;
    gc=XCreateGC(dpy,win,GCForeground | GCBackground | GCFunction,&values);    
    font=XLoadQueryFont(dpy,
	 "-adobe-times-medium-r-normal--24-240-75-75-p-124-iso8859-1");
    XSetFont(dpy,gc,font->fid);
  }

  if( unmapped ) {
    XMapWindow(dpy, win);
    bool bunny = true;
    while(bunny) {
      XNextEvent(dpy,&evt);
      if(evt.type == Expose ) bunny = false;
    }
    unmapped =false;
  }

  if(msg2) {
    int width1 = XTextWidth( font, msg1, strlen(msg1) );
    int height1 = HEIGHT/2 - font->descent;
    int width2 = XTextWidth( font, msg2, strlen(msg2) );
    int height2 = HEIGHT/2 + font->ascent;

    // clear window, draw text, flush display
    XClearWindow(dpy,win);
    XDrawString(dpy,win,gc, (WIDTH-width1)/2, height1, msg1, strlen(msg1) );
    XDrawString(dpy,win,gc, (WIDTH-width2)/2, height2, msg2, strlen(msg2) );
    XFlush(dpy);
  }
  else {
    int width1 = XTextWidth( font, msg1, strlen(msg1) );
    int top = HEIGHT/2 - font->ascent;
    int bot = HEIGHT/2 + font->descent;

    // clear window, draw text, flush display    
    XClearWindow(dpy,win);
    XDrawString(dpy,win,gc, (WIDTH-width1)/2, HEIGHT/2+font->ascent-(bot-top)/2,
		msg1, strlen(msg1) );
    XFlush(dpy);
  }
  // pause for specified time, allowing user 
  // to end pause with mouse click
  long double now = clock();
  bool bunny = true;
  while( (clock() < (now + pause*CLOCKS_PER_SEC)) && bunny ) {
    while(XPending(dpy)) {
      XNextEvent(dpy,&evt);
      if(evt.type == ButtonPress ) bunny = false;
    }
  }

  // kill display if necessary
  if( killdpy ) {
    XUnmapWindow(dpy,win);
    unmapped = true;
    XFlush(dpy);
  }

}

void readline( int fd, char* buffer ) {
  char bytebuf[1];
  bytebuf[0] = '\0';
  int i = 0;
  while( bytebuf[0] != ENDLINE ) {
    read(fd,bytebuf,1);
    if( bytebuf[0] > 31 && bytebuf[0] < 127 ) {
      buffer[i]=bytebuf[0];
      i++;
    }
  }
  buffer[i] = '\0';
}

void writeline( int fd, char* buffer ) {
  char buf1[20];
  char buf2[20];
  sprintf(buf1,"%s\r",buffer);
  write( fd, buf1, strlen(buf1) );
  while( !strstr(buf2,buffer) )
    readline(fd,buf2);
  while( !strstr(buf2,OK) )
    readline(fd,buf2);
}

void convert( char* nmbr, char* number ) {

  if( nmbr[7] == 'O' ) {
    strcpy( number, "Out of Area" );
  }
  else if( nmbr[7] == 'P' ) {
    strcpy( number, "Private" );
  }
  else {
    number[0]  = nmbr[7];
    number[1]  = nmbr[8];
    number[2]  = nmbr[9];
    number[3]  = '-';
    number[4]  = nmbr[10];
    number[5]  = nmbr[11];
    number[6]  = nmbr[12];
    number[7]  = '-';
    number[8]  = nmbr[13];
    number[9]  = nmbr[14];
    number[10] = nmbr[15];
    number[11] = nmbr[16];
    number[12] = '\0';
  }
}
