//////////////////////////////////////////////////////////////////////
//
// Class:	  BN5399SnmpCollector
//
// Description:	  Hosts an Array of SnmpGet objects which represent
//		  5399 Blades. Performs getting of led MIB, and 
//		  decodes OID returned.
//
//		  Color Keys:
//		    1000 = 100%	    (Orange)
//		     100 = 75%-100% (Green - 100%)
//		      75 = 50-75%   (Green - 75%)
//		      ....
//
// History:
// 11/14/1998 sean
// Initial Version
//
//////////////////////////////////////////////////////////////////////

#include "BN5399SnmpCollector.h"

#include <stdlib.h>
#include <stdio.h>
#include <assert.h>

BN5399SnmpCollector::BN5399SnmpCollector()
{
  // Initialize Variables
  _noBlades	      = 0;
  _snmpCollectors     = NULL;

  // Load list of blades, communitys from file
  loadBlades();

  // Allocate an array to hold SnmpGet objects
  _snmpCollectors = new SnmpGet * [_noBlades];
  assert (_snmpCollectors);

  // Connect to each blade. If un-reachable, print error message
  // and mark blade as active
  for (int i = 0; i < getNoBlades(); i++) {
  
    _snmpCollectors[i] = new SnmpGet (_blades[i].hostname,
				      _blades[i].community);
    // Connect
    if (_snmpCollectors[i]->connect())  _blades[i].active = 1;
    else {
      printf ("Failed to connect to: %s@%s. De-activating\n",
	    _blades[i].hostname, _blades[i].community);

      _blades[i].active = 0;
    }
  }
}

BN5399SnmpCollector::~BN5399SnmpCollector()
{
  delete _snmpCollectors;
}

void BN5399SnmpCollector::loadBlades()
{
  FILE *fp;
  char buffer[STRSIZE];

  fp = fopen ("5399-hosts", "r");
  assert (fp);

  while (!feof(fp)) {
    if (fgets (buffer, STRSIZE, fp) != NULL) {
      sscanf (buffer, "%s %s", _blades[_noBlades].hostname,
			 _blades[_noBlades].community);
      _noBlades++;
    }
  }
  fclose(fp);
}


void BN5399SnmpCollector::decodeBladeLeds(int bladeNo)
{
  unsigned char wanA, wanB;

  assert (_snmpCollectors[bladeNo]->_result);  

  // Wan 1
  wanA = _snmpCollectors[bladeNo]->_result[22];
  wanB = _snmpCollectors[bladeNo]->_result[23];

  // FULL
  if (wanA == 0x22 && wanB == 0x22)       _blades[bladeNo].wan1Usage = 1000; // FULL
  else if (wanA == 0x11 && wanB == 0x11)  _blades[bladeNo].wan1Usage = 100;  // 100%
  else if (wanA == 0x11 && wanB == 0x10)  _blades[bladeNo].wan1Usage = 75;   // 75%
  else if (wanA == 0x11 && wanB == 0x00)  _blades[bladeNo].wan1Usage = 50;   // 50%
  else if (wanA == 0x10 && wanB == 0x00)  _blades[bladeNo].wan1Usage = 25;   // 25%
  else if (wanA == 0x00 && wanB == 0x00)  _blades[bladeNo].wan1Usage = 0;    // 0%
  
  // Wan 2
  wanA = _snmpCollectors[bladeNo]->_result[30];
  wanB = _snmpCollectors[bladeNo]->_result[31];

  // FULL
  if (wanA == 0x22 && wanB == 0x22)       _blades[bladeNo].wan2Usage = 1000; // FULL
  else if (wanA == 0x11 && wanB == 0x11)  _blades[bladeNo].wan2Usage = 100;  // 100%
  else if (wanA == 0x11 && wanB == 0x10)  _blades[bladeNo].wan2Usage = 75;   // 75%
  else if (wanA == 0x11 && wanB == 0x00)  _blades[bladeNo].wan2Usage = 50;   // 50%
  else if (wanA == 0x10 && wanB == 0x00)  _blades[bladeNo].wan2Usage = 25;   // 25%
  else if (wanA == 0x00 && wanB == 0x00)  _blades[bladeNo].wan2Usage = 0;    // 0%  

}

void BN5399SnmpCollector::poll()
{
  oid request[] = {1,3,6,1,4,1,15,2,102,1,0};	    // OID for blade status
  int len;

  len = sizeof(request)/sizeof(oid);  

  // For each host we have
  for (int i = 0; i < _noBlades; i++) {

    if (_blades[i].active == 1)
    // If host was contactable, poll
      if (_snmpCollectors[i]->poll (request, len))

      // Then Decode Annex Wan Status Leds
	decodeBladeLeds(i);

  }
}

void BN5399SnmpCollector::getWanLeds (const int host, int &wan1, int &wan2)
{
  // Assert Host no is valid
  assert (host < getNoBlades());

  // Default Values
  wan1 = -1;
  wan2 = -1;

  // If host is active, return usage
  if (_blades[host].active == 1) {
    wan1 = _blades[host].wan1Usage;
    wan2 = _blades[host].wan2Usage;
  } 
}

char* BN5399SnmpCollector::getBladeName (const int bladeNo)
{
  assert (bladeNo < getNoBlades());
  return &(_blades[bladeNo].hostname[0]);
}
