#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <unistd.h>
#include <getopt.h>
#include <string.h>

#include "scarab.h"

#define MAXVER 0
#define MINVER 5

int main (int argc, char **argv)
{
  /* Struttura che custodisce la configurazione del processo */
  struct setupsolve setup;
  
  /* Variabili di consumo */
  char comandi[L_tmpnam]; /* Path del file dove verranno riposti i comandi */
  FILE *cmds;             /* Puntatore alla struttura del file dei comandi */
  
  /* Stringa di comando da passare alla shell */
  char execgnuplot[L_tmpnam+10] = "gnuplot ";
  
  /* Setup standard */
  setup.algoritmo = 'e';	/* Algoritmo : Eulero in avanti */
  setup.delta = 10;	/* Parametri delle equazioni di Lorenz */
  setup.b = 1.6;
  setup.r = 28;
  setup.cond_iniz.x = 1;	/* Condizioni iniziali */
  setup.cond_iniz.y = 1;
  setup.cond_iniz.z = 1;
  setup.intervallo = 10; /* Lunghezza dell'intervallo di tempo */
                         /* lungo il quale integrare */
  setup.debug = 0;
  
  /* Processa la linea di comando */
  process_args (argc, argv, &setup);
  
  
  printf("Scarab ver. %d.%d : a Lorenz attractor numerical solver\n", MAXVER, MINVER);
  printf("by Alberto Menegazzi\n\nE-mail : flash.egon@iol.it\n\n");
  
  if (setup.debug == 1)
    {
      printf("Setup of the program\n====================\n");
      printf("Method of integration : %c\n", setup.algoritmo);
      printf("System coefficients : delta=%f r=%f b=%f\n", setup.delta, setup.r, setup.b);
      printf("Starting conditions : x=%f y=%f z=%f\n", setup.cond_iniz.x, setup.cond_iniz.y, setup.cond_iniz.z);
      printf("Range of integration : t=%f\n", setup.intervallo);
    }
  
  tmpnam(setup.filedati);
  if ((setup.dati=fopen(setup.filedati,"w"))==NULL)
    {
      printf("SCARAB : unable to open temporary file %s\n\n", setup.filedati);
      exit (-1);
    }
  
  tmpnam(comandi);
  if ((cmds=fopen(comandi,"w"))==NULL)
    {
      printf("SCARAB : unable to open temporary file %s\n\n", comandi);
      exit (-1);
    }
  fprintf (cmds, "set param\nsplot \"%s\" with line\npause -1\n", setup.filedati);  
  
  /* Scelta del metodo di integrazione numerica */
  switch (setup.algoritmo)
    {
  case 'e' : eulero_avanti (setup);
	     break;
  case 'k' : runge_kutta (setup);
	     break;
    }
  
  fclose (cmds);
  fclose (setup.dati);
  
  printf ("\nPress enter to finish ...\n");
  strcat (execgnuplot, comandi);
  system (execgnuplot);
  
  unlink (setup.filedati);
  unlink (comandi);
  
  exit(0);
}

/* Soluzione tramite lo schema di Eulero in avanti */
void eulero_avanti (struct setupsolve setup)
{   
  float delta=setup.delta;
  float b=setup.b;
  float r=setup.r;
  
  float passo=0.01;   /* Step tra due passi registrati */
  float dt=0.001;     /* Step di integrazione */
  float t1=0, t2=0;   /* Contatori degli intervalli di integrazione */
  
  float x=setup.cond_iniz.x, xdt;
  float y=setup.cond_iniz.y, ydt;
  float z=setup.cond_iniz.z, zdt;
  
  while (t1<setup.intervallo)
    {
      fprintf (setup.dati, "%.4f %.4f %.4f\n", x, y, z);
      while (t2 < passo)
	  { 
	    xdt=x+delta*(-x+y)*dt;
	    ydt=y+(r*x-y-x*z)*dt;
	    zdt=z+(-b*z+x*y)*dt; 		
	    x=xdt, y=ydt, z=zdt;
	    t2 += dt;
	  }
      t1 += t2;
      t2=0;
    } 
}

/* Soluzione tramite lo schema di Runge Kutta */
void runge_kutta (struct setupsolve solve)
{
}


/* Funzione per analizzare la linea di comando */
void process_args ( int argc, char **argv, struct setupsolve *setup)
{
  char opt;
  
  while ((opt = getopt(argc, argv, "ekd:b:r:x:y:z:t:hD")) != -1)
    {
      switch (opt)
	{
	  /* Schema di Eulero in avanti */
      case 'e': setup->algoritmo = 'e'; 
		break;
		/* Schema di Runge Kutta */
      case 'k': setup->algoritmo = 'k'; 
		break;
		/* Variabili delta, b e r */
      case 'd': setup->delta = atof(optarg); 
		break;
      case 'b': setup->b = atof(optarg); 
		break;
      case 'r': setup->r = atof(optarg);
		break;
		/* Condizioni iniziali */
      case 'x': setup->cond_iniz.x = atof(optarg);
		break;
      case 'y': setup->cond_iniz.y = atof(optarg);
		break;
      case 'z': setup->cond_iniz.z = atof(optarg);
		break;
		/* Intervallo di integrazione */
      case 't': setup->intervallo = atof(optarg);
		break;
      case 'h': print_help ();
		break;
      case 'D': setup->debug = 1;
		printf("OK\n");
		break;
		
      default : print_help ();
	}
    }
  
}


/* Funzione per stampare l'help */
void print_help (void)
{	
  printf("Scarab ver. %d.%d : a Lorenz attractor numerical solver\n", MAXVER, MINVER);
  printf("by Alberto Menegazzi\n\nE-mail : flash.egon@iol.it\n\n");
  
  printf("avaible options :\n");
  printf("-e   : use forward Eulero method\n");
  printf("-k   : use Runge-Kutta 4-th order method (not avaible!)\n");
  printf("-d n : change delta coefficient\n");
  printf("-r n : change r coefficient\n");
  printf("-b n : change b coefficient\n");
  printf("-x n : change the starting conditions, x coord\n");
  printf("-y n : change the starting conditions, y coord\n");
  printf("-z n : change the starting conditions, z coord\n");
  printf("-t n : change the range of integration\n");
  printf("-h   : print this help\n");
  
  exit(1);
}
  