/*
 * mtk - Maths Toolkit for X11
 *
 * Copyright 1994-1997   andrewr@chiark.greenend.org.uk (Andrew Ross)
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 ********/

//Class for Graph window

#include <qpainter.h>
#include <qmsgbox.h>
#include <qpdevmet.h>
#include <qkeycode.h>
#include <qstring.h>
#include <qpushbt.h>
#include "graphwin.h"
#include "mainwin.h"

const QColor plotColour[5]={red,blue,green,yellow,cyan};

GraphWin::GraphWin(Math_Main *parent, double *data, int n, graphsize g, bool a,QPrinter *p, const char *name) : QWidget(NULL,name)
{
  Parent = parent;
  printer = p;

  AutoRange = a;
  numplots = 0;
  addPlot(data,n,g);
  
  setFocusPolicy( QWidget::StrongFocus );

  // Set up menu
  graphMenu = new QPopupMenu();
  DeleteID = graphMenu->insertItem("Delete graph", this, SLOT(delete_pressed()), Key_Delete );
  PrintID = graphMenu->insertItem("Print graph", this, SLOT(printPlot()), CTRL + Key_P );

  menu = new QMenuBar(this);
  menu->insertItem("Graph", graphMenu);

}

GraphWin::~GraphWin()
{
  int i;
  
  for (i=0;i<numplots;i++) {
    delete[] Array[i];
  }
  delete graphMenu;
  delete menu;
}

// Add another plot to an existing graph
void 
GraphWin::addPlot(double *d, int n, graphsize g)
{
  Equation *eqn,*eqn2;
  
  if (numplots==NUMPLOTS) {
    QMessageBox::message("Error!","Too many plots already");
    return;
  }
  GraphType[numplots] = g;
  Array[numplots] = d;
  ArraySize[numplots] = n;
  Points[numplots].resize(n);
  eqn = Parent->Equations.at(Parent->CurrentEqu-1);
  eqn2 = Parent->Equations.at(Parent->Equ2-1);
  if (GraphType[numplots].Cartesian) {
    if (GraphType[numplots].Parametric) {
      Title[numplots].sprintf("x = %s, y = %s",eqn->EquationText,eqn2->EquationText);
    }
    else {
      Title[numplots].sprintf("y = %s",eqn->EquationText);
    }
  }
  else {
    if (GraphType[numplots].Parametric) {
      Title[numplots].sprintf("r = %s, theta = %s",eqn->EquationText,eqn2->EquationText);
    }
    else {
      Title[numplots].sprintf("r = %s",eqn->EquationText);
    }
  }
  numplots++;
  if (numplots == 1) {
    setRange();
    resize(800,500);
  }
  else {
    resizeEvent(NULL);
  }
  repaint(TRUE);
}

// Draw first plot without deleting and recreating class
void
GraphWin::addFirstPlot(double *d, int n, graphsize g, bool a )
{
  int i;
  
  for (i=0;i<numplots;i++) {
    delete[] Array[i];
  }
  AutoRange = a;  
  numplots = 0;
  addPlot(d,n,g);
}

// Draw graph with given painter
void
GraphWin::drawGraph( QPainter *paint, bool isScreen )
{
  int i;

  // Plot graphs
  if (isScreen) paint->setClipRect(GraphRect);
  for (i=0;i<numplots;i++) {
    if (isScreen) paint->setPen(plotColour[i%5]);
    paint->drawPolyline(Points[i]);
  }
  // Draw axes
  if (isScreen) paint->setPen(black);
  paint->drawLine(Xaxis_min, Xaxis_max);
  paint->drawLine(Yaxis_min, Yaxis_max);
  // Label axes
  if (isScreen) paint->setClipRect(0,0,width(),height());
  paint->drawText(Xlabel_min,Xmin_text);
  paint->drawText(Xlabel_max,Xmax_text);
  paint->drawText(Ylabel_min,Ymin_text);
  paint->drawText(Ylabel_max,Ymax_text);
  for (i=0;i<numplots;i++) {
    if (isScreen) paint->setPen(plotColour[i%5]);
    paint->drawText(TextRect[i], AlignCenter, Title[i]);
  }
}

// Print the plot out
void
GraphWin::printPlot()
{
  QPainter paint;
  int h,w,sh,sw;
  double factor;
  
  if ( printer->setup() ) {
    paint.begin(printer);
    //    paint.setFont( QFont("times",24) );

    QPaintDeviceMetrics pm(printer);
    h = pm.height();
    w = pm.width();
    sh = height();
    sw = width();

    factor = ( ((double)sh)/((double)h) <? ((double)sw)/((double)w) );
    paint.scale(factor,factor);

    drawGraph( &paint, FALSE );

    paint.end();
  }
}

// Set the ranges for the graph
void
GraphWin::setRange()
{
  int i;

  QPainter paint;
  paint.begin(this);
  QFontMetrics font = paint.fontMetrics();

  if (AutoRange) {
    xmin = Array[0][0];
    xmax = Array[0][0];
    ymin = Array[0][1];
    ymax = Array[0][1];
    for (i = 0; i < 2*ArraySize[0]; i += 2) {
      if (Array[0][i] > xmax) xmax = Array[0][i];
      if (Array[0][i] < xmin) xmin = Array[0][i];
      if (Array[0][i+1] > ymax) ymax = Array[0][i+1];
      if (Array[0][i+1] < ymin) ymin = Array[0][i+1];
    }
    if (xmax == xmin) {
      xmin = xmin - 0.5;
      xmax = xmax + 0.5;
    }
    if (ymax == ymin) {
      ymin = ymin - 0.5;
      ymax = ymax + 0.5;
    }
  }
  else {
    xmin = GraphType[0].xmin;
    xmax = GraphType[0].xmax;
    ymin = GraphType[0].ymin;
    ymax = GraphType[0].ymax;
  }

  Xmin_text.sprintf("%.3G",xmin);
  Xmax_text.sprintf("%.3G",xmax);
  Ymin_text.sprintf("%.3G",ymin);
  Ymax_text.sprintf("%.3G",ymax);
  fontheight = font.height();
  Xmin_shift.setX(-font.width(Xmin_text)/2);
  Xmin_shift.setY(fontheight);
  Xmax_shift.setX(-font.width(Xmax_text)/2);
  Xmax_shift.setY(fontheight);
  Ymin_shift.setX(-font.width(Ymin_text)-2);
  Ymin_shift.setY(fontheight/2);
  Ymax_shift.setX(-font.width(Ymax_text)-2);
  Ymax_shift.setY(fontheight/2);

  paint.end();
}

// Respond to a paint event by drawing graph
void
GraphWin::paintEvent(QPaintEvent *)
{
  QPainter paint;

  paint.begin(this);
  drawGraph( &paint, TRUE );
  paint.end();
}

// Respond to delete button being pressed
void
GraphWin::delete_pressed()
{
  emit deleteGraph();
}

// Respond to close event
void
GraphWin::closeEvent( QCloseEvent * )
{
  emit deleteGraph(false);
}


// Respond to a resize event by rescaling points
void
GraphWin::resizeEvent( QResizeEvent *)
{
  int i,j,x,y,h,w, menuheight;
  double xwin,ywin,wleft,wright,htop,hbot;

  h = height();
  w = width();
  menuheight = menu->height();
  htop = h*0.05 + fontheight*1.1*numplots + menuheight;
  hbot = h*0.95;
  wleft = w*0.05;
  wright = w*0.95;
  xwin = (wright-wleft)/(xmax-xmin);
  ywin = (hbot-htop)/(ymax-ymin);
  
  for (j=0;j<numplots;j++) {
    for(i=0; i<ArraySize[j]; i++) {
      x = (int)(wleft + (Array[j][2*i]-xmin)*xwin);
      y = (int)(hbot - (Array[j][2*i+1]-ymin)*ywin);
      Points[j].setPoint(i,x,y);
    }
  }

  x = (int)(wleft);
  if (ymin>=0) {
    y = (int)(hbot);
  }
  else {
    if (ymax<=0) {
      y = (int)(htop);
    }
    else {
      y = (int)((hbot*ymax - htop*ymin)/(ymax-ymin));
    }
  }
  Xaxis_min.setX(x);
  Xaxis_min.setY(y);
  x = (int)(wright);
  Xaxis_max.setX(x);
  Xaxis_max.setY(y);

  y = (int)(hbot);
  if (xmin>=0) {
    x = (int)(wleft);
  }
  else {
    if (xmax<=0) {
      x = (int)(wright);
    }
    else {
      x = (int)((wleft*xmax - wright*xmin)/(xmax-xmin));
    }
  }
  Yaxis_min.setX(x);
  Yaxis_min.setY(y);
  y = (int)(htop);
  Yaxis_max.setX(x);
  Yaxis_max.setY(y);

  Xlabel_min = Xaxis_min + Xmin_shift;
  Xlabel_max = Xaxis_max + Xmax_shift;
  Ylabel_min = Yaxis_min + Ymin_shift;
  Ylabel_max = Yaxis_max + Ymax_shift; 

  for (i=0;i<numplots;i++) {
    TextRect[i].setLeft( (int)(wleft) );
    TextRect[i].setRight( (int)(wright) );
    TextRect[i].setTop( (int)(h*0.0125 + i*1.1*fontheight +  menuheight) );
    TextRect[i].setBottom( (int)(h*0.0125 + (i+1)*1.1*fontheight +  menuheight) );
  }

  GraphRect.setLeft( (int)(wleft) );
  GraphRect.setRight( (int)(wright) );
  GraphRect.setTop( (int)(htop) );
  GraphRect.setBottom( (int)(hbot) );
}

#include "graphwin.moc"


