//-------------------------------------------------------------------//
// Synopsis: Short-Time MAGnitude.

// Syntax:   </ TSCALE ; Y /> = stmag ( X, FRAME, OVERLAP, FS, WIN )

//      stmag computes the short-time magnitude of X using a size
//      FRAME (seconds) and a percentage OVERLAP (%) between
//      successive frames using a rectangular data window.  The
//      sampling frequency is given by FS. The short-time magnitude
//      curve is returned in Y and a time scale corresponding to the
//      end of the data frame is returned in TSCALE.  The curve may be
//      displayed with the command 'plot([tscale,y])'.

//      The last, and optional argument, WIN, specifies the data
//      window used before computing the short-time energy.  WIN can
//      be one of the following:

//          hamming    "hamm"
//          hanning    "hann"
//          blackman   "blac"
//          triangular "tria"

//      See also: steng, stzcr, avsmooth, mdsmooth

//      Matlab original by:
//       LT Dennis W. Brown 7-11-93, DWB 11-11-94
//       Naval Postgraduate School, Monterey, CA
//       May be freely distributed.
//       Not for use in commercial products.

//      Ref: Rabiner & Schafer, Digital Processing of Speech
//       Signals, 1978, ss 4.2, pp 120-126.

require window

//-------------------------------------------------------------------//

stmag = function (x, frame, overlap, fs, win)
{
  local (x, frame, overlap, fs, win)

  // default values
  y = [];
  tscale = [];

  // must have at least 3 args
  if (nargs < 3)
  {
    error("sp_stmag: Requires first three arguments.");
  }

  // percentage must be in range 0-100
  if (overlap < 0 || overlap > 100)
  {
    error("sp_stmag: Overlap percentage must be in range 0-100%");
  }

  // figure out if we have a vector
  if (min(size(x)) != 1)
  {
    error("sp_stmag: Input arg \"x\" must be a 1xN or Nx1 vector.");
  }

  // work with Nx1 vectors
  x = x[:];

  // variables
  Ns = length(x);                         // number of samples
  N = floor(fs * frame);                  // samples-per-frame
  Ndiff = floor(N * (1 - overlap/100));   // samples between windows
  L = floor((Ns-N)/Ndiff);                // number of windows
  y = zeros(L,1);                         // space for answer
  tscale = zeros(L,1);	                  // space for time

  // data window
  datawindow = ones(N,1);                 // rectangular default
  if (nargs == 5)
  {
    datawindow = window (N, win);
  }

  // use the absolute value of x
  x = abs(x);

  // windows with overlap
  for (k in 1:L)
  {
    s1 = (k-1) * Ndiff + 1;                 // start of window
    tscale[k;1] = k * Ndiff/fs;
    y[k;1] = sum(x[s1:s1+N-1;1] .* datawindow);
  }
  
  return << tscale=tscale ; y=y >>;
};
