//-------------------------------------------------------------------//

//  Syntax: polyval(p, x), polyvalm(p, X)

//  Description:

//  Polynomial evaluation
//
//  p is a vector (colum or row) whose elements are the coefficients 
//  of a polynomial in decending powers.
//
//  polyval evaluates and returns
//      t = p[1]*x.^n + p[2]*x.^(n-1) + ... + p[n+1]
//  x can be a vector (row or column) or a matrix.
//
//  polyvalm evaluates and returns
//      t = p[1]*x^n + p[2]*x^(n-1) + ... + p[n+1]
//  X must be a square matrix.


//  Example 1:  Evaluate 2*x^2 + x + 5 at points x=1,2,3
//  > p=[2,1,5]
//   p =
//          2          1          5
//  > x=[1,2,3]
//   x =
//          1          2          3
//  > polyval(p,x)
//          8         15         26
//
//  Example 2:  Evaluate 2*x^2 + x + 5 at its roots
//  > p=[2,1,5]
//   p =
//          2          1          5
//  > r=roots(p)
//   r =
//             -0.25 - 1.56i
//             -0.25 + 1.56i
//  > polyval(p,r)
//          0
//          0
//  
//  Example 3: Test Cayley-Hamilton theorem
//  > X=[1,-2;-2,1];
//  > p=poly(X)
//   p =
//          1         -2         -3
//  > polyvalm(p,X)
//          0          0
//          0          0
//

//  See Also: poly, polyfit, roots
//
//-------------------------------------------------------------------//

polyval = function (p, x)
{
   if (class(p) == "num") {   
      if (p.n <= 0) {
         error("The first argument of polyval() is null.");
      else if (min(size(p)) != 1) {
         error("The first argument of polyval() must be a column or a row vector.");
      }}

      // scalar polynomial 
      if (p.n == 1) {
         return p;
      }     
      t = p[1]*ones(size(x));
      for(i in 2:p.n)
      {
        t = t.*x + p[i];
      }
   else {
     error("Wrong argument class in polyval().");
   }}
   return t;
};

polyvalm = function (p, x)
{
   if (x.nr != x.nc) {
      error("The 2nd argument of polyvalm() must be a square matrix.");
   }
   if (class(p) == "num") {   
      if (p.n <= 0) {
         error("The first argument of polyvalm() is null.");
      else if (min(size(p)) != 1) {
         error("The first argument of polyvalm() must be a column or a row vector.");
      }}

      // scalar polynomial 
      if (p.n == 1) {
         return p;
      }     
      s = p[1]*ones(size(x));
      for(i in 2:p.n)
      {
        s = s*x + p[i]; 
      }
   else {
     error("Wrong argument class in polyvalm().");
   }}
   return s;
};
