#include <math.h>
#include <stdio.h> /* for NULL */
#include "mathlib.h"

double LogNfactorial (double n)
{ /* Stirling's approximation */
  return n < 0.1 ? 0 : log(sqrt(2*M_PI*n)*(1+1/12.0/n))+n*(log(n) - 1);
}

double CumulativeBinomial (int n, int x, double pi)
{
  double p, sum = 0;
  int i, swapped = 0;
  
  if (x+x > n) {
    x = n - x - 1;
    pi = 1 - pi;
    sum = 1;
    p = -pow(1-pi,n);
    swapped = 1;
  }
  else p = pow (1 - pi, n);
  
  if (pi == 1) return x >= n ? 1 - sum : sum;
  if (fabs(p) < 1e-100) {
    //printf(".");
    p = sqrt (n * pi * (1 - pi));
    sum = CumulativeNormalDistribution ((x+0.5-pi*n)/p)-
          CumulativeNormalDistribution ((-0.5-pi*n)/p);
    return swapped ? 1-sum : sum;
  }
  
  for (i=0;i<=x;i++) {
    sum += p;
    p *= pi*(n-i) / ((1-pi)*(i+1));
  }
  return sum;
}

double NormalDensity (double x)
{ /* sigma = 1, mu = 0 */
  return exp(-0.5*x*x) / sqrt(2*M_PI);
}

double CumulativeNormalDistribution (double x)
{
  int i;
  double y = fabs(x), z = -0.5*y*y, r = 0;
  
  if (y < 4.2) {
    for (i = 80; i >= 0; i--) {
      r *= z / (i+1);
      r += 1.0 / (2*i+1);
      /* Here i indicates the number of time we will still multiply with */
      /* z, so the coeficient of z^i is 1.0 / (2*i+1)/i!. */
    }
    r *= y / sqrt (2 * M_PI);
  }
  else r=0.5-NormalDensity(y)/y*(1-0.865172/y/y);

  if (x < 0) return 0.5 - r;
  return 0.5 + r;
}

double InverseNormal (double p)
{ /* Use the Newton-Raphson method. */
  double err, x = (p-0.5)*2.5;
  
  while (1) {
    err = CumulativeNormalDistribution (x) - p;
    if (fabs(err) < 1e-10) return x;
    x -= err / NormalDensity(x);
  }
}

double ChiSquareDensity (double x, int freedom)
{
  double m = freedom / 2.0;

  return exp(
    (x <= 0 ? x : (m-1)*log(x/2)) - x/2 - LogNfactorial (m - 1)
  )/2;
}

double CumulativeChiSquare (double x, int freedom)
{
  double m = freedom / 2.0, sum = 0, mul;

  if (x > 1000 || freedom > 230) {
    return CumulativeNormalDistribution ((x - freedom) / sqrt(2*freedom));
  }

  if (freedom%2!=0) {
    return (CumulativeChiSquare (
      (x-freedom)*sqrt((freedom+1)/(double)freedom)+freedom+1, freedom + 1) +
      CumulativeChiSquare (
      (x-freedom)*sqrt((freedom-1)/(double)freedom)+freedom-1, freedom - 1))
      / 2;
    // *(char*)NULL = 0;
  }
  
  if (x<1e-100) return 0;
  
  mul = exp(-x/2+(m-1)*log(x/2));
  for (sum = 0; m > 1.2 || freedom%2 != 0 && m>0.2; m-=1) {
    sum -= mul;
    sum /= m-1;
    mul *= 2/x;
  }
  sum += mul*(exp(x/2)-1);
/*
  mul = pow(0.5, m);
  for (sum = 0; m > 1.1; m-=1) {
    sum -= mul*2*pow(x,m-1)*exp(-x/2);
    sum /= m-1;
    mul *= 2;
  }
  sum += 2*mul*(1-exp(-x/2));
*/
/*
  mul = exp((m-1)*log(0.5) - LogNfactorial (m-1))/2;
  for (sum = 0; m > 1.1; m-=1) {
    sum -= mul*2*pow(x,m-1)*exp(-x/2);
    mul *= 2 * (m-1);
  }
  sum += 2*mul*(1-exp(-x/2));
*/
  return sum;
}

double InverseChiSquare (double p, int freedom)
{ /* Use the Newton-Raphson method. */
  double err, x = freedom - 2; /* Start at maximum of ChiSquareDensity */
  
  while (1) {
    //printf("%lf\n",x);
    err = CumulativeChiSquare (x, freedom) - p;
    if (fabs(err) < 1e-10) return x;
    x -= err / ChiSquareDensity (x, freedom);
  }
}

double TDensity (double x, int v)
{
  int n = (v - 1) / 2;
  double r = pow (1+x*x/v, -n-1);
  
  if (v % 2 == 0) *(char*)NULL=0;
  /* now we want to calculate
  r <- int_infty^infty (1+t^2/v)^(-(v+1)/2)
     = int_infty^infty (1+t^2/v)^-n-1
  */
  for (;n > 0; n--) r /= 1 - 0.5/n;
  r /= M_PI;
  r /= sqrt(v);

  return r;
}

double CumulativeTDistribution (double x, int v)
{
  int n = (v - 1) / 2;
  double sum = 0, mul = 1;
  
  if (v % 2 == 0) {
    return (CumulativeTDistribution (x, v-1)+CumulativeTDistribution(x,v+1))
      /2;
  }
  x /= sqrt (v);
  for (;n > 0; n--) {
    sum += mul*x/2.0/n/pow(x*x+1, n);
    mul *= 1-0.5/n;
  }
  sum += mul*atan (x);
  return 0.5 + sum / mul / M_PI;
}

double InverseTDistribution (double p, int freedom)
{
  double err, x = (p-0.5)*2.5;

  while (1) {
    err = CumulativeTDistribution (x, freedom) - p;
    //printf("%lf %12.10lf %lf\n",x,err,p);
    if (fabs(err) < 1e-10) return x;
    x -= err / TDensity(x, freedom%2!=0 ? freedom : freedom-1);
  }
}

void AddToRegressor (regressor *c, double x, double y)
{
  c->sumx+=x;    c->sumy+=y;
  c->sumxx+=x*x; c->sumyy+=y*y;
  c->sumxy+=x*y; c->n++;
}

double RegressorB (regressor *c)
/* slope of best fit. */
{
  return (c->sumxy - c->sumx*c->sumy/c->n)/(c->sumxx-c->sumx*c->sumx/c->n);
}

double RegressorA (regressor *c)
/* y-intersect of best fit. */
{
  return (c->sumy - RegressorB(c)*c->sumx)/c->n;
}

double RegressorY (regressor *c, double x)
/* Returns the expected value of y */
{
  return RegressorA (c) + x * RegressorB(c);
}

double RegressorSyz2 (regressor *c)
{
  return (c->sumyy - RegressorA (c)*c->sumy - RegressorB(c)*c->sumxy) /
         (c->n-2);
}

double RegressorD0 (regressor *c, double x)
{
  return sqrt (RegressorSyz2 (c) * (1+1.0/c->n+
    (x-c->sumx/c->n)*(x-c->sumx/c->n)/(c->sumxx-c->sumx*c->sumx/c->n)
  ));
}

double RegressorP (regressor *c, double x, double deltay)
/* Returns the probability that if you where to take a sample at x, */
/* that sample will be less than deltay+RegressorY(c, x) */
{
  return CumulativeTDistribution (deltay / RegressorD0 (c, x), c->n-2);
}

double RegressorYEstimate (regressor *c, double x, double p)
/* RegressorYEstimate (c, x, RegressorP (c, x, deltay)) == deltay i.e. */
/* P(-infy < y < RegressorY + RegressorYEstimate) = p */
/* P(RegressorY-RegressorYEstimate < y < RegressorY + RegressorYEstimate) */
/*   = 2p-1 */
{
  return RegressorD0 (c, x) * InverseTDistribution (p, c->n-2);
}

