/* ----------------------------------------------------------------------------
 * Copyright (C) 1995-1997 by Karim Kaschani
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * to rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHOR BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 * IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE. 
 * ------------------------------------------------------------------------- */

/* ----------------------------------------------------------------------------
 * spline - evenly spaced cubic spline interpolation of two-column
 *          ASCII tables
 *
 * command options:
 *
 *        -x   first value, last value
 *        -dx  step size
 *        -i   infile
 *        -o   outfile
 * ------------------------------------------------------------------------- */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <time.h>
#include <math.h>

#define  MAXCHARS 256
#define  INITSIZE 128
#define  TRUE     1
#define  FALSE    0

#define  MAX(a,b) ((a>b) ? a : b)
#define  MIN(a,b) ((a<b) ? a : b)





/* ----------------------------------------------------------------------------
 * usage
 * ------------------------------------------------------------------------- */

void usage()
{
    printf("usage: spline -x <from>,<to> [-dx <step>] [-i <infile>] [-o <outfile>]\n");
}





/* ----------------------------------------------------------------------------
 * isnumber - check character for number
 * ------------------------------------------------------------------------- */

int isnumber(char sign)
{
    int status;

    switch (sign) {
           case '0':
           case '1':
           case '2':
           case '3':
           case '4':
           case '5':
           case '6':
           case '7':
           case '8':
           case '9':
           case '+':
           case '-':
           case '.': status = TRUE;
                     break;
           default : status = FALSE;
                     break;
    }
    return (status);
}





/* ----------------------------------------------------------------------------
 * main
 * ------------------------------------------------------------------------- */

int main(int argc, char *argv[])
{
    typedef struct data_set {
        int	num;		/* How many points  */
        int	size;		/* Allocated size   */
        double	*x;		/* X values         */
        double	*y;		/* Y values         */
        double	*y2;		/* Y 2nd derivative */
        double	*tmp;		/* temporary values */
    } DataSet;

    DataSet	array;
    int		i, j, INFILE, OUTFILE, AUTODX, LAST, CONTINUE;
    double	xfrom, xto, dx, x, y, xold, yold, sig, p, a, b, d;
    time_t	seconds, *tloc=NULL;
    size_t	maxdate=30;
    char	*command, *date, *source, *target;
    char	*word1, *word2;
    char	buffer[MAXCHARS];
    FILE        *infile, *outfile;

    /* ----------------------------------------------------- set up defaults */

    command = malloc(sizeof(char) * MAXCHARS);
    date    = malloc(sizeof(char) * MAXCHARS);
    source  = malloc(sizeof(char) * MAXCHARS);
    target  = malloc(sizeof(char) * MAXCHARS);
    word1   = malloc(sizeof(char) * MAXCHARS);
    word2   = malloc(sizeof(char) * MAXCHARS);

    INFILE   = FALSE;
    OUTFILE  = FALSE;
    AUTODX   = TRUE;
    CONTINUE = TRUE;

    outfile = stdout;
    infile  = stdin;

    (void) strcpy(source,"stdin\0");
    (void) strcpy(target,"stdout\0");

    seconds = time(tloc);
    (void) strftime(date,maxdate,"%a %d %b %Y (%X)",localtime(&seconds));

    (void) strcpy(command, "spline");

    /* ------------------------------------------ parse command line options */

    if (argc < 3) {
       usage();
       exit(1);
    }

    for (i=1; i<argc; i++) {
        if ((strcmp(argv[i],"-x") == 0) && (i < argc - 1)) {
           if (sscanf(argv[i+1],"%lf,%lf",&xfrom,&xto)) {
              (void) strcat(command," ");
              (void) strcat(command,argv[i]);
              (void) strcat(command," ");
              (void) strcat(command,argv[i+1]);
              i++;
           } else {
              usage();
              exit(1);
           }
        } else if ((strcmp(argv[i],"-dx") == 0) && (i < argc - 1)) {
           if (isnumber(argv[i+1][0])) {
              sscanf(argv[i+1],"%lf",&dx);
              (void) strcat(command," ");
              (void) strcat(command,argv[i]);
              (void) strcat(command," ");
              (void) strcat(command,argv[i+1]);
              AUTODX = FALSE;
              i++;
           } else {
              usage();
              exit(1);
           }
        } else if ((strcmp(argv[i],"-i") == 0) && (i < argc - 1)) {
           if (sscanf(argv[i+1],"%s",source)) {
              (void) strcat(command," ");
              (void) strcat(command,argv[i]);
              (void) strcat(command," ");
              (void) strcat(command,argv[i+1]);
              INFILE = TRUE;
              i++;
           } else {
              usage();
              exit(1);
           }
        } else if ((strcmp(argv[i],"-o") == 0) && (i < argc - 1)) {
           if (sscanf(argv[i+1],"%s",target)) {
              (void) strcat(command," ");
              (void) strcat(command,argv[i]);
              (void) strcat(command," ");
              (void) strcat(command,argv[i+1]);
              OUTFILE = TRUE;
              i++;
           } else {
              usage();
              exit(1);
           }
        } else {
           break;
        }
    }

    /* ---------------------------------------------------------- open files */

    if (INFILE) {
       if (! (infile = fopen(source,"r"))) {
          fprintf(stderr,"spline: cannot open %s\n", source);
          exit(2);
       }
    }

    if (OUTFILE) {
       if (! (outfile = fopen(target,"w"))) {
          fprintf(stderr,"spline: cannot open %s\n", target);
          if (INFILE)
             (void) fclose(infile);
          exit(2);
       }
    }

    fprintf(outfile,"# %s --> %s\n",date,command);

    /* --------------------------------------------------------- scan infile */

    i = 0;		/* number of values within (xfrom,xto) */
    j = 0;		/* number of values outside            */
    LAST = TRUE;
    while (CONTINUE) {

          if (fgets(buffer, MAXCHARS, infile)) {
             (void) sscanf(buffer,"%s %s",word1, word2);
          } else {
             strcpy(buffer,"\n");
             CONTINUE = FALSE;
          }

          if ((buffer[0] != '\n') && isnumber(word1[0]) && 
               isnumber(word2[0])) {

             sscanf(buffer,"%lf %lf",&x,&y);

             /* check for requested interval */

             if ((x > xfrom) && (x < xto)) {
             
                /* initialise arrays if necessary */

                if (i == 0) {
                   LAST       = FALSE;
                   array.num  = 0;
                   array.size = INITSIZE;
                   array.x    = (double *) malloc((unsigned) 
                                (INITSIZE * sizeof(double)));
                   array.y    = (double *) malloc((unsigned) 
                                (INITSIZE * sizeof(double)));
                   array.y2   = (double *) malloc((unsigned) 
                                (INITSIZE * sizeof(double)));
                   array.tmp  = (double *) malloc((unsigned) 
                                (INITSIZE * sizeof(double)));

                   /* is there an old value preceeding `xfrom' ? */

                   if (j > 0) {
                      array.x[i] = xold;
                      array.y[i] = yold;
                      i++;
                   }
                }
                array.x[i] = x;
                array.y[i] = y;

                /* evaluate step size */

                if (AUTODX) {
                   if (i == 1)
                      dx = array.x[i]-array.x[i-1];
                   if (i > 1)
                      dx = MIN(array.x[i]-array.x[i-1], dx);
                }

                i++;

                /* Time to make the arrays bigger ? */

                if (i >= array.size-1) {
                   array.size *= 2;
                   array.x   = (double *) realloc((char *) array.x, 
                               (unsigned) (array.size * sizeof(double)));
                   array.y   = (double *) realloc((char *) array.y,
                               (unsigned) (array.size * sizeof(double)));
                   array.y2  = (double *) realloc((char *) array.y2,
                               (unsigned) (array.size * sizeof(double)));
                   array.tmp = (double *) realloc((char *) array.tmp,
                               (unsigned) (array.size * sizeof(double)));
                }
             } else {
                xold = x;
                yold = y;
                j++;

                if (! LAST) {
                   LAST       = TRUE;
                   array.x[i] = x;
                   array.y[i] = y;

                   if (AUTODX) {
                      if (i == 1)
                         dx = array.x[i]-array.x[i-1];
                      if (i > 1)
                         dx = MIN(array.x[i]-array.x[i-1], dx);
                   }

                   i++;
                }
             }
          } else {
             if (i > 0) {
                array.num = i;

                /* --------------------------------- evaluate 2nd derivative */

                array.y2[0]  = -0.5;	/* keep the 1st derivative */
                array.tmp[0] = 0.0;	/* at the beginning        */

                for (i=1; i<array.num-2; i++) {
                    sig          = (array.x[i] - array.x[i-1]) 
                                   / (array.x[i+1] - array.x[i-1]);
                    p            = sig * array.y2[i-1] + 2.0;
                    array.y2[i]  = (sig - 1.0) / p;
                    array.tmp[i] = (6.0 * ((array.y[i+1] - array.y[i]) 
                                   / (array.x[i+1] - array.x[i]) 
                                   - (array.y[i] - array.y[i-1]) 
                                   / (array.x[i] - array.x[i-1])) 
                                   / (array.x[i+1] - array.x[i-1]) 
                                   - sig * array.tmp[i-1]) / p;
                }

                array.y2[array.num-1]  = 0.5;	/* keep the 1st derivative */
                array.tmp[array.num-1] = 0.0;	/* at the end too */

                array.y2[array.num-1]  = (array.tmp[array.num-1] 
                                         - array.y2[array.num-1]
                                         * array.tmp[array.num-2])
                                         / (array.y2[array.num-1] 
                                         * array.y2[array.num-2] 
                                         + 1.0);

                for (i=array.num-2; i>0; i--)
                    array.y2[i] = array.y2[i] * array.y2[i+1] + array.tmp[i];

                /* ----------------------------------- evaluate cubic spline */

                i = 1;
                j = 0;

                for (x=xfrom; x<xto+dx; x+=dx) {
                    while (array.x[i] <= x)
                          i++;
       
                    if ((array.x[i-1] < x) && (x < array.x[i])) {
                       d = array.x[i] - array.x[i-1];
                       a = (array.x[i] - x) / d;
                       b = (x - array.x[i-1]) / d;
                       y = a * array.y[i-1] + b * array.y[i]
                           + ((pow(a,3.0)-a) * array.y2[i-1]
                           + (pow(b,3.0)-b) * array.y2[i]) * pow(d,2.0) / 6.0;
                    } else {
                       y = array.y[i-1];
                    }

                    fprintf(outfile,"  %1.8E  %1.8E\n", x, y);

                    j++;
                }

                /* ---------------------------- set-up for the next data set */

                i = 0;
                array.size = 0;
                free((char *) array.x);
                free((char *) array.y);
                free((char *) array.y2);
                free((char *) array.tmp);
             }
             fprintf(outfile,"%s",buffer);
          }
    }

/* --------------------------------------------------------------------- end */

    if (INFILE)
       (void) fclose(infile);

    if (OUTFILE)
       (void) fclose(outfile);

    exit(0);
}
