/* ----------------------------------------------------------------------------
 * Copyright (C) 1995-1997 by Karim Kaschani
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * to rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHOR BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 * IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE. 
 * ------------------------------------------------------------------------- */

/* ----------------------------------------------------------------------------
 * merge - merge two-column ASCII tables of two files
 *
 * command options:
 *
 *         -a  add
 *         -s  subtract
 *         -m  multiply
 *         -d  divide
 *         -c  combine
 *         -o  output file (default stdout)
 * ------------------------------------------------------------------------- */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <time.h>

#define  MAXCHARS 256
#define  TRUE     1
#define  FALSE    0

#define UNDEF    -1
#define ADD       0
#define SUB       1
#define MUL       2
#define DIV       3
#define COM       4





/* ----------------------------------------------------------------------------
 * usage
 * ------------------------------------------------------------------------- */

void usage()
{
    printf("usage: merge -[a|s|m|d|c] [-o <outfile>] <infile1> <infile2>\n");
}





/* ----------------------------------------------------------------------------
 * isnumber - check character for number
 * ------------------------------------------------------------------------- */

int isnumber(char sign)
{
    int status;

    switch (sign) {
           case '0':
           case '1':
           case '2':
           case '3':
           case '4':
           case '5':
           case '6':
           case '7':
           case '8':
           case '9':
           case '+':
           case '-':
           case '.': status = TRUE;
                     break;
           default : status = FALSE;
                     break;
    }
    return (status);
}





/* ----------------------------------------------------------------------------
 * read_data - read next data
 *
 * status: TRUE  = next data of same dataset found
 *         FALSE = 1st data of next dataset found
 *         EOF   = end-of-file reached (no data found)
 * ------------------------------------------------------------------------- */

int read_data(FILE *in, FILE *out, double *x, double *y)
{
    int  status;
    char *word1, *word2, *pointer;
    char buffer[MAXCHARS];

    word1   = malloc(sizeof(char) * MAXCHARS);
    word2   = malloc(sizeof(char) * MAXCHARS);

    status = TRUE;

    while ((pointer = fgets(buffer, MAXCHARS, in)) != NULL) {
          (void) sscanf(buffer,"%s %s",word1, word2);

          if (isnumber(word1[0]) && isnumber(word2[0])) {
             sscanf(buffer,"%lf %lf",x,y);
             break;
          } else {
             fprintf(out,"%s",buffer);
             status = FALSE;
          }

          (void) strcpy(word1,"\0");
          (void) strcpy(word2,"\0");
    }

    if (pointer == NULL) {
       status = EOF;
       x = y = 0;
    }

    return (status);
}





/* ----------------------------------------------------------------------------
 * main
 * ------------------------------------------------------------------------- */

int main(int argc, char *argv[])
{
    int		i, OUTFILE, mode, status1, status2;
    double	xn1, yn1, xo1, yo1, dx1;
    double	xn2, yn2, xo2, yo2, dx2;
    double	x, y, y1, y2;
    char	*target, *source1, *source2, *date, *command;
    FILE	*infile1, *infile2, *outfile;
    time_t	seconds, *tloc=NULL;
    size_t	maxdate=30;

    target  = malloc(sizeof(char) * MAXCHARS);
    source1 = malloc(sizeof(char) * MAXCHARS);
    source2 = malloc(sizeof(char) * MAXCHARS);
    date    = malloc(sizeof(char) * MAXCHARS);
    command = malloc(sizeof(char) * MAXCHARS);

    /* ----------------------------------------------------- set up defaults */

    seconds = time(tloc);
    (void) strftime(date,maxdate,"%a %d %b %Y (%X)",localtime(&seconds));
 
    OUTFILE = FALSE;

    (void) strcpy(target,"stdout");
    outfile = stdout;
    
    /* ------------------------------------------ parse command line options */

    if (argc < 4) {
       usage();
       exit(1);
    }

    (void) strcpy(command, "merge");

    for (i=1; i<argc-2; i++) {
        if ((strcmp(argv[i],"-o") == 0) && (i < argc - 1)) {
           if (sscanf(argv[i+1],"%s",target)) {
              (void) strcat(command," ");
              (void) strcat(command,argv[i]);
              (void) strcat(command," ");
              (void) strcat(command,argv[i+1]);
              OUTFILE = TRUE;
              i++;
           } else {
              usage();
              exit(1);
           }
        } else if (strcmp(argv[i],"-a") == 0) {
              (void) strcat(command," ");
              (void) strcat(command,argv[i]);
              mode = ADD;
        } else if (strcmp(argv[i],"-s") == 0) {
              (void) strcat(command," ");
              (void) strcat(command,argv[i]);
              mode = SUB;
        } else if (strcmp(argv[i],"-m") == 0) {
              (void) strcat(command," ");
              (void) strcat(command,argv[i]);
              mode = MUL;
        } else if (strcmp(argv[i],"-d") == 0) {
              (void) strcat(command," ");
              (void) strcat(command,argv[i]);
              mode = DIV;
        } else if (strcmp(argv[i],"-c") == 0) {
              (void) strcat(command," ");
              (void) strcat(command,argv[i]);
              mode = COM;
        } else {
           usage();
           exit(1);
        }
    }

    if (sscanf(argv[i],"%s",source1) && sscanf(argv[i+1],"%s",source2)) {
       (void) strcat(command," ");
       (void) strcat(command,argv[i]);
       (void) strcat(command," ");
       (void) strcat(command,argv[i+1]);
    } else {
       usage();
       exit(1);
    }

    /* -------------------------------------------------------- open infiles */

    if (! (infile1 = fopen(source1,"r"))) {
       fprintf(stderr,"merge: cannot open %s\n", source1);
       exit(2);
    }

    if (! (infile2 = fopen(source2,"r"))) {
       fprintf(stderr,"merge: cannot open %s\n", source2);
       exit(2);
    }

    if (OUTFILE) {
       if (! (outfile = fopen(target,"w"))) {
          fprintf(stderr,"merge: cannot open %s\n", target);
          exit(2);
       }
    }

/* ----------------------------------------------------------- evaluate data */

    fprintf(outfile,"# %s --> %s\n",date,command);

    if ((status1 = read_data(infile1,outfile,&xn1,&yn1)) == EOF) {
       fprintf(stderr,"merge: cannot read %s - EOF reached\n", source1);
       exit(2);
    } else {
       xo1 = xn1;
       yo1 = yn1;
    }

    if ((status2 = read_data(infile2,outfile,&xn2,&yn2)) == EOF) {
       fprintf(stderr,"merge: cannot read %s - EOF reached\n", source2);
       exit(2);
    } else {
       xo2 = xn2;
       yo2 = yn2;
    }

    if (xn1 < xn2)
       while (xn1 < xn2) {
             xo1 = xn1;
             yo1 = yn1;
             status1 = read_data(infile1,outfile,&xn1,&yn1);
       }
    else if (xn2 < xn1)
       while (xn2 < xn1) {
             xo2 = xn2;
             yo2 = yn2;
             status2 = read_data(infile2,outfile,&xn2,&yn2);
       };

    do {
       if (xn1 == xn2) {
          x  = xn1;
          y1 = yn1;
          y2 = yn2;
       } else if (xn1 < xn2) {
          x  = xn1;
          y1 = yn1;
          y2 = (yn2-yo2)/(xn2-xo2) * (x-xo2) + yo2;
       } else {
          x  = xn2;
          y1 = (yn1-yo1)/(xn1-xo1) * (x-xo1) + yo1;
          y2 = yn2;
       }

       switch (mode) {
              case ADD: y = y1+y2;
                        break;
              case SUB: y = y1-y2;
                        break;
              case MUL: y = y1*y2;
                        break;
              case DIV: y = y1/y2;
                        break;
              case COM: x = y1;
                        y = y2;
                        break;
              default : break;
       }

       fprintf(outfile,"  %1.8E  %1.8E\n",x,y);

       if (xn1 == xn2) {
          xo1 = xn1;
          yo1 = yn1;
          status1 = read_data(infile1,outfile,&xn1,&yn1);
          xo2 = xn2;
          yo2 = yn2;
          status2 = read_data(infile2,outfile,&xn2,&yn2);
       } else if (xn1 < xn2) {
          xo1 = xn1;
          yo1 = yn1;
          status1 = read_data(infile1,outfile,&xn1,&yn1);
       } else {
          xo2 = xn2;
          yo2 = yn2;
          status2 = read_data(infile2,outfile,&xn2,&yn2);
       }

    } while ((status1 == TRUE) && (status2 == TRUE));

    exit(0);
}
