/* ----------------------------------------------------------------------------
 * Copyright (C) 1995-1997 by Karim Kaschani
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * to rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHOR BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 * IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE. 
 * ------------------------------------------------------------------------- */

/* ----------------------------------------------------------------------------
 * fft - fast fourier transformation of evenly spaced two-column
 *       ASCII tables
 *
 * command options:
 *
 *        -I   inverse fourier transformation
 *        -re  real part
 *        -im  imaginary part
 *        -a   absolute value
 *        -p   phase value
 *        -i   infile
 *        -o   outfile
 * ------------------------------------------------------------------------- */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <time.h>
#include <math.h>

#define  PI       3.14159265358979323846
#define  MAXCHARS 256
#define  INITSIZE 128
#define  TRUE     1
#define  FALSE    0

#define  UNDEF    0
#define  REAL     1
#define  IMAG     2
#define  ABSOL    3
#define  PHASE    4  

#define  MAX(a,b) ((a>b) ? a : b)
#define  MIN(a,b) ((a<b) ? a : b)





/* ----------------------------------------------------------------------------
 * usage
 * ------------------------------------------------------------------------- */

void usage()
{
    printf("usage: fft -[re|im|a|p] [-I] [-i <infile>] [-o <outfile>]\n");
}





/* ----------------------------------------------------------------------------
 * isnumber - check character for number
 * ------------------------------------------------------------------------- */

int isnumber(char sign)
{
    int status;

    switch (sign) {
           case '0':
           case '1':
           case '2':
           case '3':
           case '4':
           case '5':
           case '6':
           case '7':
           case '8':
           case '9':
           case '+':
           case '-':
           case '.': status = TRUE;
                     break;
           default : status = FALSE;
                     break;
    }
    return (status);
}





/* ----------------------------------------------------------------------------
 * main
 * ------------------------------------------------------------------------- */

int main(int argc, char *argv[])
{
    typedef struct data_set {
        int	num;		/* How many points  */
        int	size;		/* Allocated size   */
        double	*v;		/* data values      */
    } DataSet;

    DataSet	array;
    int		i, j, m, INFILE, OUTFILE, MODE, CONTINUE, inverse, mmax, istep;
    double	x, y, wr, wi, wpr, wpi, wtemp, theta, dx, xold, tempr, tempi;
    time_t	seconds, *tloc=NULL;
    size_t	maxdate=30;
    char	*command, *date, *source, *target;
    char	*word1, *word2;
    char	buffer[MAXCHARS];
    FILE        *infile, *outfile;

    /* ----------------------------------------------------- set up defaults */

    command = malloc(sizeof(char) * MAXCHARS);
    date    = malloc(sizeof(char) * MAXCHARS);
    source  = malloc(sizeof(char) * MAXCHARS);
    target  = malloc(sizeof(char) * MAXCHARS);
    word1   = malloc(sizeof(char) * MAXCHARS);
    word2   = malloc(sizeof(char) * MAXCHARS);

    INFILE   = FALSE;
    OUTFILE  = FALSE;
    MODE     = UNDEF;
    CONTINUE = TRUE;
    inverse  = 1;

    outfile = stdout;
    infile  = stdin;

    (void) strcpy(source,"stdin\0");
    (void) strcpy(target,"stdout\0");

    seconds = time(tloc);
    (void) strftime(date,maxdate,"%a %d %b %Y (%X)",localtime(&seconds));

    (void) strcpy(command, "fft");

    /* ------------------------------------------ parse command line options */

    for (i=1; i<argc; i++) {
        if (strcmp(argv[i],"-I") == 0) {
           (void) strcat(command," ");
           (void) strcat(command,argv[i]);
           inverse = -1;
        } else if (strcmp(argv[i],"-re") == 0) {
           (void) strcat(command," ");
           (void) strcat(command,argv[i]);
           MODE = REAL;
        } else if (strcmp(argv[i],"-im") == 0) {
           (void) strcat(command," ");
           (void) strcat(command,argv[i]);
           MODE = IMAG;
        } else if (strcmp(argv[i],"-a") == 0) {
           (void) strcat(command," ");
           (void) strcat(command,argv[i]);
           MODE = ABSOL;
        } else if (strcmp(argv[i],"-p") == 0) {
           (void) strcat(command," ");
           (void) strcat(command,argv[i]);
           MODE = PHASE;
        } else if ((strcmp(argv[i],"-i") == 0) && (i < argc - 1)) {
           if (sscanf(argv[i+1],"%s",source)) {
              (void) strcat(command," ");
              (void) strcat(command,argv[i]);
              (void) strcat(command," ");
              (void) strcat(command,argv[i+1]);
              INFILE = TRUE;
              i++;
           } else {
              usage();
              exit(1);
           }
        } else if ((strcmp(argv[i],"-o") == 0) && (i < argc - 1)) {
           if (sscanf(argv[i+1],"%s",target)) {
              (void) strcat(command," ");
              (void) strcat(command,argv[i]);
              (void) strcat(command," ");
              (void) strcat(command,argv[i+1]);
              OUTFILE = TRUE;
              i++;
           } else {
              usage();
              exit(1);
           }
        } else {
           break;
        }
    }

    if (MODE == UNDEF) {
       usage();
       exit(1);
    }

    /* ---------------------------------------------------------- open files */

    if (INFILE) {
       if (! (infile = fopen(source,"r"))) {
          fprintf(stderr,"fft: cannot open %s\n", source);
          exit(2);
       }
    }

    if (OUTFILE) {
       if (! (outfile = fopen(target,"w"))) {
          fprintf(stderr,"fft: cannot open %s\n", target);
          if (INFILE)
             (void) fclose(infile);
          exit(2);
       }
    }

    fprintf(outfile,"# %s --> %s\n",date,command);

    /* --------------------------------------------------------- scan infile */

    i = 0;

    while (CONTINUE) {

          if (fgets(buffer, MAXCHARS, infile)) {
             (void) sscanf(buffer,"%s %s",word1, word2);
          } else {
             strcpy(buffer,"\n");
             CONTINUE = FALSE;
          }

          if ((buffer[0] != '\n') && isnumber(word1[0]) && 
               isnumber(word2[0])) {

             sscanf(buffer,"%lf %lf",&x,&y);

             /* initialise array if necessary */

             if (i == 0) {
                array.num  = 0;
                array.size = INITSIZE;
                array.v    = (double *) malloc((unsigned) 
                             (INITSIZE * sizeof(double)));
             }

             /* get sampling interval */

             if (i == 2)
                dx = x - xold;

             /* check on constant sampling interval */

             if ((i > 2) && (fabs(x - xold - dx)/dx > 1e-3)) {
                fprintf(stderr,"fft: error - samples not evenly spaced.\n");
                exit(2);
             }

             xold = x;

             array.v[i] = y;
             array.v[i+1] = 0.0;

             i += 2;

             /* Time to make the array bigger ? */

             if (i >= array.size) {
                array.size *= 2;
                array.v   = (double *) realloc((char *) array.v, 
                            (unsigned) (array.size * sizeof(double)));
             }
          } else {
             if (i > 0) {
                array.num = (int) pow(2.0,ceil(log((double) i) / log(2.0)));

                if (array.num > array.size) {
                   fprintf(stderr,"fft: array evaluation error.\n");
                   exit(2);
                }

                for (j=i; j<array.num; j++)
                    array.v[j] = 0.0;

                /* ----------------------------------------------------- fft */

                j = 1;

                /* bit reversal */

                for (i=1; i<array.num; i+=2) {
                    if (j > i) {
                       tempr        = array.v[j-1];
                       tempi        = array.v[j];
                       array.v[j-1] = array.v[i-1];
                       array.v[j]   = array.v[i];
                       array.v[i-1] = tempr;
                       array.v[i]   = tempi;
                    }

                    m = array.num / 2;

                    while ((m >= 2) && (j > m)) {
                          j -= m;
                          m /= 2;
                    }

                    j += m;
                }

                /* Danielson-Lanczos algorithm */

                mmax = 2;

                while (array.num > mmax) {
                      istep = 2 * mmax;
                      theta = 2.0 * PI / (inverse * mmax);
                      wpr = -2.0 * pow(sin(0.5 * theta), 2.0);
                      wpi = sin(theta);
                      wr = 1.0;
                      wi = 0.0;

                      for (m=1; m<mmax; m+=2) {
                          for (i=m; i<array.num; i+=istep) {
                              j = i + mmax;
                              tempr = wr * array.v[j-1] - wi * array.v[j];
                              tempi = wr * array.v[j] + wi * array.v[j-1];
                              array.v[j-1]  = array.v[i-1] - tempr;
                              array.v[j]    = array.v[i] - tempi;
                              array.v[i-1] += tempr;
                              array.v[i]   += tempi;
                          }
                          wtemp = wr;
                          wr += wr * wpr - wi * wpi;
                          wi += wi * wpr + wtemp * wpi;
                      }
                      mmax = istep;
                }

                /* ------------------------------------------ set-up results */
  
                for (i=0; i<array.num/2; i+=2) {
                    if (inverse < 0) {
                       array.v[i]   *= (double) 2.0 / array.num;
                       array.v[i+1] *= (double) 2.0 / array.num;
                    }

                    x = (double) i / (array.num*dx);

                    switch (MODE) {
                           case REAL : y = array.v[i];
                                       break;
                           case IMAG : y = array.v[i+1];
                                       break;
                           case ABSOL: y = sqrt(pow(array.v[i],2.0) 
                                           + pow(array.v[i+1],2.0));
                                       break;
                           case PHASE: y = atan2(array.v[i+1],array.v[i]);
                                       break; 
                           default   : break;
                    }

                    fprintf(outfile,"  %1.8E  %1.8E\n", x, y);
                }

                /* ---------------------------- set-up for the next data set */

                i = 0;
                array.size = 0;
                free((char *) array.v);
             }
             fprintf(outfile,"%s",buffer);
          }
    }

/* --------------------------------------------------------------------- end */

    if (INFILE)
       (void) fclose(infile);

    if (OUTFILE)
       (void) fclose(outfile);

    exit(0);
}
