/* ----------------------------------------------------------------------------
 * Copyright (C) 1995-2000 by Karim Kaschani
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * to rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHOR BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 * IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE. 
 * --------------------------------------------------------------------------*/

#include "globals.h"
#include "str_utils.h"
#include <stdarg.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <fcntl.h>
#include <unistd.h>
#include <dirent.h>
#include <signal.h>





/* ----------------------------------------------------------------------------
 * bsort - bubble sort algorithm
 *         *f = first line to be sorted
 * ------------------------------------------------------------------------- */

void bsort(line *f)
{
  line		*f1, *f2;
  char		tmp[MAXCHARS];
  boolean	swapped;

  do {
        swapped = FALSE;
        f1 = f2 = f;
        f2++;

        while (strcmp((char *) f2, "\0") != 0) {
              if (strcmp((char *) f2, (char *) f1) < 0) {
                 strcpy(tmp, (char *) f1);
                 strcpy((char *) f1, (char *) f2);
                 strcpy((char *) f2, tmp);
                 swapped = TRUE;
              }

              f1++;
              f2++;
        }
  } while (swapped);
}





/* ----------------------------------------------------------------------------
 * delaySigWinch - delay SIGWINCH by remembering its occurrence
 * ------------------------------------------------------------------------- */

void delaySigWinch()
{
     extern boolean	SigWinch;

     SigWinch = TRUE;
}





/* ----------------------------------------------------------------------------
 * scanRcDir - scan resource directory
 *             *resourcepath = name of resource path
 *             *n            = number of resource files found
 *             RESULT        : pointer to first resource file
 * ------------------------------------------------------------------------- */

line *scanRcDir(char *resourcepath, int *n)
{
  line			*files0, *files;
  boolean		Status;
  DIR			*dirp;
  struct dirent		*content;

  dirp = opendir(resourcepath);
  files0 = (line *) malloc(sizeof(line) * MAXFILES);
  files  = files0;
 
  *n = 0;
  Status = FALSE;

  while ((content = readdir(dirp)) != NULL) {
        if ((strcmp(content->d_name, ".") != 0) && 
            (strcmp(content->d_name, "..") != 0)) {
            strcpy((char *) files, content->d_name);
            files++;
            (*n)++;
        }
        if (strcmp(content->d_name, "default") == 0)
           Status = TRUE;
  }
    
  if (Status == FALSE) {
     strcpy((char *) files, "default");
     files++;
     (*n)++;
  }
    
  strcpy((char *) files, "\0");

  files = files0;
  bsort(files);

  return (files);
}





/* ----------------------------------------------------------------------------
 * listDatasets - list all available datasets
 *                *set = pointer to datasets
 * ------------------------------------------------------------------------- */

void listDatasets(line *set)
{
  line		*s;

  s = set;

  while (strcmp((char *) s, "\0") != 0) {
        fprintf(stdout, "%s\n", (char *) s);
        s++;
  }
}





/* ----------------------------------------------------------------------------
 * readDataset - read resourcefile data
 *               *resourcefile = name of resulting resource file
 *               *resourcepath = name of resource path
 *               *dataset      = name of dataset
 *               n             = number of dataset parameters to be read
 *               ...           = the names of "n" dataset parameters to be
 *                               read
 * ------------------------------------------------------------------------- */

void readDataset(char *resourcefile, char *resourcepath, char *dataset,
                 int n, ...)
{
  FILE		*id;
  va_list	parm;
  int		i;
  char		*rc, *TEMP;

  strcpy(resourcefile, resourcepath);
  strcat(resourcefile, "/");
  strcat(resourcefile, dataset);
  id = fopen(resourcefile, "r");

  va_start(parm, n);

  for (i=0; i<n; i++) {
      rc = va_arg(parm, char *);

      if (id != NULL) {
         fgets(rc, MAXCHARS, id);
         TEMP = strchr(rc, '\n');
         if (TEMP != NULL) *TEMP = 0;
      } else {
         strcpy(rc, "\0");
      }
  }

  va_end(parm);
  if (id != NULL) fclose(id);
}





/* ----------------------------------------------------------------------------
 * plotData - launch plotting tool
 *            *xy0 = first data line to be plotted
 *            n    = total number of lines to be plotted
 * ------------------------------------------------------------------------- */

void plotData(line *xy0, int n)
{
  int		i;
  line		*xy;
  FILE		*id;
  pid_t		pid;

  /* fork independent child process */

  if ((pid = fork()) == 0) {
     signal(SIGCLD, SIG_DFL);
 
     xy = xy0;
     if ((id = popen(PLOTTOOL, "w")) != NULL) {
        for (i=1; i<=n; i++)
            fprintf(id, "%s\n", (char *) xy++);
        pclose(id);
     }
 
     exit(0);
  }
}





/* ----------------------------------------------------------------------------
 * plot2Data - launch plotting tool and plot 2 different sets of data
 *             *xIn = pointer to x data of 1st set
 *             *yIn = pointer to y data of 1st set
 *             lIn  = number of data points of 1st set
 *             *xy0 = first data line of 2nd set
 *             n    = total number of lines of 2nd set
 * ------------------------------------------------------------------------- */

void plot2Data(double *xIn, double *yIn, int lIn, line *xy0, int n)
{
  int		i;
  line		*xy;
  FILE		*id;
  pid_t		pid;

  /* fork independent child process */
 
  if ((pid = fork()) == 0) {
     signal(SIGCLD, SIG_DFL);
 
     xy = xy0;
     if ((id = popen(PLOTTOOL, "w")) != NULL) {
        for (i=0; i<lIn; i++)
            fprintf(id, "   % .8E   % .8E\n", xIn[i], yIn[i]);
        for (i=1; i<=n; i++)
            fprintf(id, "%s\n", (char *) xy++);
        pclose(id);
     }
 
     exit(0);
  }
}





/* ----------------------------------------------------------------------------
 * mkdirhier - make directory hierarchy
 *             *path = directory hierarchy
 *             mode  = permissions of new directories
 *             RETURN = 0 if successful, -1 otherwise
 * ------------------------------------------------------------------------- */

int mkdirhier(char *path, mode_t mode)
{
  char	*p, *tmp;
  int	n, status = 0;
  DIR	*dirp;

  tmp = (char *) malloc(sizeof(char) * Strlen(path));

  p = path;

  /* ------------------------------------------------------ skip leading "/" */
     
  while (p[0] == '/') {
        p++;
  }

  do {

     /* -------------------------------------------------- find next subpath */
  
     p = strchr(p, '/');
     
     if (p != NULL) {
        n = Strlen(path) - Strlen(p);
        p++;
     } else {
        n = Strlen(path);
     }

     Strncpy(tmp, path, n);

     /* ---------------------------------------- create subpath if necessary */

     if ((dirp = opendir(tmp)) != NULL) {
        closedir(dirp);
     } else {
        mode = S_IRWXU | S_IRWXG | S_IRWXO;
        if (mkdir(tmp, mode) != 0) {
           status = -1;
           break;
        }
     }
     
  } while (p != NULL);

  free(tmp);

  return (status);
}
