/* ----------------------------------------------------------------------------
 * Copyright (C) 1995-2000 by Karim Kaschani
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * to rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHOR BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 * IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE. 
 * --------------------------------------------------------------------------*/

#include "globals.h"
#include <math.h>
#include <values.h>

#ifndef LN_MAXDOUBLE
#  define LN_MAXDOUBLE log(MAXDOUBLE)
#endif

extern double	*phi;
extern int	iphi, nphi;

extern boolean	runtimeError;
extern void	printError(char *);

static complex l  = {1, 0};
static complex j  = {0, 1};
static complex j2 = {0, 2};
static complex O  = {0, 0};





/* ---------------------------------------------------------------------- RE */

double re(complex z)
{
  return z.re;
}





/* ---------------------------------------------------------------------- IM */

double im(complex z)
{
  return z.im;
}





/* -------------------------------------------------------------------- INT_ */

boolean int_(double x)
{
  if ((long)x == x)
    return TRUE;
  else
    return FALSE;
}





/* --------------------------------------------------------------------- SGN */

double sgn(double x)
{
  if (x > 0.0)
    return 1.0;         /* positiv */
  else if (x < 0.0)
    return -1.0;        /* negativ */
  else
    return 0.0;         /* Null */
}





/* ----------------------------------------------------------------------SRE */

complex sre(complex z)
{
  complex w;

  if (z.im != 0) {
     w = O;
     printError(" re!() not real     ");
     runtimeError = TRUE;
  } else {
     w = z;
  }

  return w;
}





/* ----------------------------------------------------------------------SIM */

complex sim(complex z)
{
  complex w;

  if (z.re != 0) {
     w = O;
     printError(" im!() not imaginary");
     runtimeError = TRUE;
  } else {
     w = z;
  }

  return w;
}





/* ------------------------------------------------------------------- CCMPLX*/

complex ccmplx(double a, double p)
{
  complex w;

  w.re = a * cos(p);
  w.im = a * sin(p);

  return w;
}





/* -------------------------------------------------------------------- CABS */

double cabs(complex z)
{
  return sqrt(z.re * z.re + z.im * z.im);
}





/* -------------------------------------------------------------------- CPHI */

double cphi(complex z)
{
  double w, w0;
  int    n0;

  /* ......................... reallocate memory for phase pointer if needed */

  if (iphi >= nphi && nphi > 0) {
     nphi += PHIBUF;
     if ((phi = (double *) realloc(phi, sizeof(double) * nphi)) == NULL)
        nphi = 0;
  }

  /* ................................................. calculate phase angle */

  if (z.re == 0) {
     if (z.im > 0) {
        w = M_PI_2;
     } else if (z.im < 0) {
        w = 3*M_PI_2;
     } else {
        w = 0;
     }
  } else {

     /* w = (-pi/2, pi/2) */
  
     w = atan(z.im / z.re);

     /* w = [-pi, pi) */     
     
     if (z.re < 0 && z.im > 0) w += M_PI;
     if (z.re < 0 && z.im <= 0) w -= M_PI;

     /* w = [0, 2pi) */
     
     if (w < 0) w += M_PI*2;
  }

  /* .. adapt phase angle to lately calculated value stored in phase pointer */

  if (nphi > 0) {
     w0 = fmod(phi[iphi], 2*M_PI);
     n0 = (int) (phi[iphi] / (2*M_PI));
     if (w0 < 0) {
        w0 += 2*M_PI;
        n0--;
     }

     if (fabs(w - w0) > M_PI) {
        if (w > w0) {
           n0--;
        } else {
           n0++;
        }
     }

     w += (double) n0 * 2 * M_PI;
     phi[iphi] = w;
     iphi++;
  }

  return w;
}





/* -------------------------------------------------------------------- CINT */

complex cint(complex z)
{
  complex w;

  w.re = (long) z.re;
  w.im = (long) z.im;

  return w;
}





/* -------------------------------------------------------------------- CSGN */

complex csgn(complex z)
{
  complex w;
  
  if (z.re > 0.0)
     w.re = 1.0;        /* positiv */
  else if (z.re < 0.0)
     w.re = -1.0;       /* negativ */
  else
     w.re = 0.0;        /* Null */
  
  if (z.im > 0.0)
     w.im = 1.0;        /* positiv */
  else if (z.im < 0.0)
     w.im = -1.0;       /* negativ */
  else
     w.im = 0.0;        /* Null */

  return w;
}





/* ------------------------------------------------------------------- CFRAC */

complex cfrac(complex z)
{
  complex w;

  w.re = z.re - (long) z.re;
  w.im = z.im - (long) z.im;

  return w;
}





/* ------------------------------------------------------------------ CROUND */

complex cround(complex z)
{
  complex w;

  w.re = (long) floor(z.re + 0.5);
  w.im = (long) floor(z.im + 0.5);

  return w;
}





/* -------------------------------------------------------------------- CNEG */

complex cneg(complex z)
{
  complex w;

  w.re = -z.re;
  w.im = -z.im;

  return w;
}





/* -------------------------------------------------------------------- CADD */

complex cadd(complex z1, complex z2)
{
  complex w;

  w.re = z1.re + z2.re;
  w.im = z1.im + z2.im;

  return w;
}





/* -------------------------------------------------------------------- CSUB */

complex csub(complex z1, complex z2)
{
  complex w;

  w.re = z1.re - z2.re;
  w.im = z1.im - z2.im;

  return w;
}





/* ------------------------------------------------------------------- CMULT */

complex cmult(complex z1, complex z2)
{
  complex w;

  w.re = z1.re * z2.re - z1.im * z2.im;
  w.im = z1.re * z2.im + z1.im * z2.re;

  return w;
}





/* -------------------------------------------------------------------- CDIV */

complex cdiv(complex z1, complex z2)
{
  complex w;
  double  n;

  if (z2.re == 0.0 && z2.im == 0.0) {
     w.re = sgn(z1.re)*MAXDOUBLE;
     w.im = sgn(z1.im)*MAXDOUBLE;
     printError(" division by zero   ");
     runtimeError = TRUE;
  } else {
     n = z2.re * z2.re + z2.im * z2.im;
     w.re = (z1.re * z2.re + z1.im * z2.im) / n;
     w.im = (z1.im * z2.re - z1.re * z2.im) / n;
  }

  return w;
}





/* -------------------------------------------------------------------- CEXP */

complex cexp(complex z)
{
  complex w;

  if (z.re > LN_MAXDOUBLE) {
     w.re = MAXDOUBLE;
     w.im = MAXDOUBLE;
     printError(" exp() overflow     ");
     runtimeError = TRUE;
  } else {
     w.re = exp(z.re) * cos(z.im);
     w.im = exp(z.re) * sin(z.im);
  }

  return w;
}





/* -------------------------------------------------------------------- CLOG */

complex clog(complex z)
{
  complex w;

  if (cabs(z) > 0) {
     w.re = log(cabs(z));
     w.im = cphi(z);
  } else {
     w.re = -MAXDOUBLE;
     w.im = 0;
     printError(" ln() undefined     ");
     runtimeError = TRUE;
  }

  return w;
}





/* ------------------------------------------------------------------ CLOG10 */

complex clog10(complex z)
{
  complex w;

  if (cabs(z) > 0) {
     w.re = log(cabs(z))/log(10);
     w.im = cphi(z)/log(10);
  } else {
     w.re = -MAXDOUBLE;
     w.im = 0;
     printError(" log() undefined    ");
     runtimeError = TRUE;
  }

  return w;
}





/* ------------------------------------------------------------------- CSQRT */

complex csqrt(complex z)
{
  double a, p;

  a = sqrt(cabs(z));
  p = cphi(z)/2;

  return ccmplx(a, p);
}





/* ------------------------------------------------------------------ CPOWER */

complex cpower(complex z1, complex z2)
{
  complex w;

  if (z1.re == 0 && z1.im == 0) {
     if (z2.re == 0 && z2.im == 0) {
       w.re = w.im = 1;
       printError(" power() undefined   ");
       runtimeError = TRUE;
     } else {
       w.re = w.im = 0;
     }
  } else {
     w = cmult(z2, clog(z1));
     if (w.re > LN_MAXDOUBLE) {
        w.re = MAXDOUBLE;
        w.im = MAXDOUBLE;
        printError(" power() overflow   ");
        runtimeError = TRUE;
     } else {
        w = cexp(w);
     }
  }

  return w;
}





/* -------------------------------------------------------------------- CSIN */

complex csin(complex z)
{
  complex u, v, w;

  u = cmult(j,z);
  v = cneg(u);

  w = csub(cexp(u),cexp(v));
  w = cdiv(w,cadd(j,j));

  return w;
}





/* -------------------------------------------------------------------- CCOS */

complex ccos(complex z)
{
  complex u, v, w;

  u = cmult(j,z);
  v = cneg(u);

  w = cadd(cexp(u),cexp(v));
  w.re /= 2;
  w.im /= 2;

  return w;
}





/* -------------------------------------------------------------------- CTAN */

complex ctan(complex z)
{
  complex u, v, w;

  u = csin(z);
  v = ccos(z);

  if (v.re == 0 && v.im == 0) {
     w.re = w.im = 1;
     printError(" tan() undefined    ");
     runtimeError = TRUE;
  } else {
     w = cdiv(u,v);
  }

  return w;
}





/* -------------------------------------------------------------------- CCOT */

complex ccot(complex z)
{
  complex u, v, w;

  u = ccos(z);
  v = csin(z);

  if (v.re == 0 && v.im == 0) {
     w.re = w.im = 1;
     printError(" cot() undefined    ");
     runtimeError = TRUE;
  } else {
     w = cdiv(u,v);
  }

  return w;
}





/* ------------------------------------------------------------------- CSINH */

complex csinh(complex z)
{
  complex w;

  w = csub(cexp(z),cexp(cneg(z)));
  w.re /= 2;
  w.im /= 2;

  return w;
}





/* ------------------------------------------------------------------- CCOSH */

complex ccosh(complex z)
{
  complex w;

  w = cadd(cexp(z),cexp(cneg(z)));
  w.re /= 2;
  w.im /= 2;

  return w;
}





/* ------------------------------------------------------------------- CTANH */

complex ctanh(complex z)
{
  complex u, v, w;

  u = csinh(z);
  v = ccosh(z);

  if (v.re == 0 && v.im == 0) {
     w.re = w.im = 1;
     printError(" tanh() undefined   ");
     runtimeError = TRUE;
  } else {
     w = cdiv(u,v);
  }

  return w;
}





/* ------------------------------------------------------------------- CCOTH */

complex ccoth(complex z)
{
  complex u, v, w;

  u = ccosh(z);
  v = csinh(z);

  if (v.re == 0 && v.im == 0) {
     w.re = w.im = 1;
     printError(" coth() undefined   ");
     runtimeError = TRUE;
  } else {
     w = cdiv(u,v);
  }

  return w;
}





/* ----------------------------------------------------------------- CARCSIN */

complex carcsin(complex z)
{
  complex w;

  w = cadd(cmult(j,z),csqrt(csub(l,cmult(z,z))));

  if (w.re == 0 && w.im == 0) {
     w.re = w.im = 1;
     printError(" arcsin() undefined ");
     runtimeError = TRUE;
  } else {
     w = cneg(cmult(j,clog(w)));
  }

  return w;
}





/* ----------------------------------------------------------------- CARCCOS */

complex carccos(complex z)
{
  complex w;

  w = cadd(z,csqrt(csub(cmult(z,z),l)));

  if (w.re == 0 && w.im == 0) {
     w.re = w.im = 1;
     printError(" arccos() undefined ");
     runtimeError = TRUE;
  } else {
     w = cneg(cmult(j,clog(w)));
  }

  return w;
}





/* ----------------------------------------------------------------- CARCTAN */

complex carctan(complex z)
{
  complex w;

  w = cdiv(cadd(l,cmult(j,z)),csub(l,cmult(j,z)));

  if (w.re == 0 && w.im == 0) {
     w.re = w.im = 1;
     printError(" arctan() undefined ");
     runtimeError = TRUE;
  } else {
     w = cdiv(clog(w),j2);
  }

  return w;
}





/* ----------------------------------------------------------------- CARCCOT */

complex carccot(complex z)
{
  complex w;

  w = cdiv(cadd(cmult(j,z),l),csub(cmult(j,z),l));

  if (w.re == 0 && w.im == 0) {
     w.re = w.im = 1;
     printError(" arccot() undefined ");
     runtimeError = TRUE;
  } else {
     w = cneg(cdiv(clog(w),j2));
  }

  return w;
}





/* ----------------------------------------------------------------- CARSINH */

complex carsinh(complex z)
{
  complex w;

  w = cadd(z,csqrt(cadd(cmult(z,z),l)));

  if (w.re == 0 && w.im == 0) {
     w.re = w.im = 1;
     printError(" arsinh() undefined ");
     runtimeError = TRUE;
  } else {
     w = clog(w);
  }

  return w;
}





/* ----------------------------------------------------------------- CARCOSH */

complex carcosh(complex z)
{
  complex w;

  w = cadd(z,csqrt(csub(cmult(z,z),l)));

  if (w.re == 0 && w.im == 0) {
     w.re = w.im = 1;
     printError(" arcosh() undefined ");
     runtimeError = TRUE;
  } else {
     w = clog(w);
  }

  return w;
}





/* ----------------------------------------------------------------- CARTANH */

complex cartanh(complex z)
{
  complex w;

  w = cdiv(cadd(l,z),csub(l,z));

  if (w.re == 0 && w.im == 0) {
     w.re = w.im = 1;
     printError(" artanh() undefined ");
     runtimeError = TRUE;
  } else {
     w = clog(w);
     w.re /= 2;
     w.im /= 2;
  }

  return w;
}





/* ----------------------------------------------------------------- CARCOTH */

complex carcoth(complex z)
{
  complex w;

  w = cdiv(cadd(z,l),csub(z,l));

  if (w.re == 0 && w.im == 0) {
     w.re = w.im = 1;
     printError(" arcoth() undefined ");
     runtimeError = TRUE;
  } else {
     w = clog(w);
     w.re /= 2;
     w.im /= 2;
  }

  return w;
}






/* -------------------------------------------------------------------- CFACT */

complex cfact(complex z)
{
  complex w;
  long    i, p;

  if (int_(z.re) && z.re >= 0 && z.im == 0) {
    p = 1;
    for (i = 1; i <= (long) z.re; i++)
      p *= i;
    w.re = p;
    w.im = 0;
  } else {
    w.re = w.im = 1;
    printError(" fact() undefined   ");
    runtimeError = TRUE;
  }

  return w;
}





/* -------------------------------------------------------------------- CERF */

complex cerf(complex z)
{
  complex  w;
  double   c1=-1.26551223, c2= 1.00002368, c3= 0.37409196;
  double   c4= 0.09678418, c5=-0.18628806, c6= 0.27886807;
  double   c7=-1.13520398, c8= 1.48851587, c9=-0.82215223;
  double   x_abs, x, y;

  if (z.im == 0) {
     x_abs = fabs(z.re);
     x = 1/(1+0.5*x_abs);

     y = 0.17087277;
     y = c9+x*y;
     y = c8+x*y;
     y = c7+x*y;
     y = c6+x*y;
     y = c5+x*y;
     y = c4+x*y;
     y = c3+x*y;
     y = c2+x*y;
     y = x*exp(-x_abs*x_abs+c1+x*y);

     if (z.re < 0) {
        w.re = y - 1;
        w.im = 0;
     } else {
        w.re = 1 - y;
        w.im = 0;
     }
  } else {
     w.re = w.im = 1;
     printError(" erf() undefined    ");
     runtimeError = TRUE;
  }

  return w;
}





/* ------------------------------------------------------------------- CERFC */

complex cerfc(complex z)
{
  complex w;

  if (z.im == 0) {
     w = csub(l,cerf(z));
  } else {
     w.re = w.im = 1;
     printError(" erfc() undefined   ");
     runtimeError = TRUE;
  }

  return w;
}
